/*
 * Copyright (c) 2006-2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.chart.ui.screen;

import java.awt.BorderLayout;
import java.awt.event.AdjustmentEvent;
import java.awt.event.AdjustmentListener;

import javax.swing.JScrollBar;

import jp.sf.orangesignal.chart.data.ChartDataset;
import jp.sf.orangesignal.chart.event.ChartEvent;
import jp.sf.orangesignal.chart.ui.Icons;
import jp.sf.orangesignal.chart.ui.ChartScreenType;
import jp.sf.orangesignal.chart.ui.canvas.StepChartCanvas;

/**
 * 非時系列チャート画面を提供します。
 * 
 * @author 杉澤 浩二
 */
public class StepChartScreen extends ChartScreen implements AdjustmentListener {

	private static final long serialVersionUID = -4065158006724751295L;

	/**
	 * 主画面を保持します。
	 */
	private final StepChartCanvas canvas;

	/**
	 * スクロールバーを保持します。
	 */
	private final JScrollBar scrollbar = new JScrollBar(JScrollBar.HORIZONTAL, 0, 0, 0, 0);

	/**
	 * コンストラクタです。
	 * 
	 * @param icons アイコン情報
	 * @param screenType 非時系列チャートの種類
	 */
	public StepChartScreen(final Icons icons, final ChartScreenType screenType) {
		// 各コントロールを初期化します。
		this.canvas = new StepChartCanvas(icons, this, screenType);
		this.scrollbar.addAdjustmentListener(this);

		// レイアウトを処理します。
		setLayout(new BorderLayout());
		add(this.canvas, BorderLayout.CENTER);
		add(this.scrollbar, BorderLayout.SOUTH);
	}

	@Override
	public void datasetSwitch(final ChartEvent e) {
		processChartEvent(e);
	}

	/**
	 * 期間コンボボックスが変更されると呼び出されます。
	 * この実装はスクロールバーとチャートの描画位置や描画範囲を設定します。
	 */
	@Override
	public void periodSwitch(final ChartEvent e) {
		processChartEvent(e);
	}

	/**
	 * サイド画面の設定値が変更されると呼び出されます。
	 */
	@Override
	public void settingChanged(final ChartEvent e) {
		processChartEvent(e);
	}

	private void processChartEvent(final ChartEvent e) {
		if (!e.isIgnoreStart()) {
			final ChartDataset dataset = e.getDataset();
			if (dataset == null) {
				this.scrollbar.setMaximum(0);
				this.scrollbar.setVisibleAmount(0);
				this.scrollbar.setValue(0);
				this.scrollbar.setBlockIncrement(0);
				this.scrollbar.setEnabled(false);
			} else {
				// 設定順序に気おつけること
				// @see javax.swing.DefaultBoundedRangeModel
				this.scrollbar.setValue(0);	// XXX - 最初にこれ重要
				this.scrollbar.setMaximum(dataset.getCount());
				this.scrollbar.setVisibleAmount(e.getPeriod());
				this.scrollbar.setValue(e.getStart());
				this.scrollbar.setBlockIncrement(e.getPeriod() / 2);
				this.scrollbar.setEnabled(true);
			}
		}

		this.canvas.switchDataset(e);
	}

	/**
	 * スクロールバーの値が変更されると呼び出されます。
	 * この実装は、チャートの描画開始位置を設定します。
	 */
	@Override
	public void adjustmentValueChanged(final AdjustmentEvent e) {
		this.canvas.setStart(e.getValue());
	}

}
