/*
 * Copyright (c) 2008-2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.trading;

import java.io.Serializable;

/**
 * 残高不足時でも引出し可能な口座情報を提供します。
 * 
 * @author 杉澤 浩二
 */
public class VirtualAccount implements Account, Serializable {

	private static final long serialVersionUID = 5749635909805874526L;

	/**
	 * デフォルトコンストラクタです。
	 */
	public VirtualAccount() {}

	/**
	 * コンストラクタです。
	 * 
	 * @param cash 初期残高
	 */
	public VirtualAccount(final double cash) {
		this(cash, true);
	}

	/**
	 * コンストラクタです。
	 * 
	 * @param cash 初期残高
	 * @param unlimited 残高不足時に引出し可能とするかどうか
	 */
	public VirtualAccount(final double cash, final boolean unlimited) {
		this.cash = cash;
		this.unlimited = unlimited;
	}

	/**
	 * 残高を保持します。
	 */
	private double cash;

	@Override public double getCash() { return cash; }

	/**
	 * 残高を設定します。
	 * 
	 * @param cash 残高
	 */
	public void setCash(final double cash) { this.cash = cash; }

	/**
	 * 実装は必ず <code>0</code> を返します。
	 */
	@Override public double getInterestRate() { return 0; }

	/**
	 * 残高不足時に引出し可能かどうか
	 */
	private boolean unlimited = true;

	/**
	 * 残高不足時に引出し可能かどうかを返します。
	 * 
	 * @return 残高不足時に引出し可能かどうか
	 */
	public boolean isUnlimited() { return unlimited; }

	@Override
	public void deposit(double amount) {
		if (amount <= 0)
			throw new IllegalArgumentException();
		cash = cash + amount;
	}

	@Override
	public boolean withdraw(final double amount) {
		if (amount <= 0)
			throw new IllegalArgumentException();

		final double balance = cash - amount;
		if (!unlimited && balance < 0)
			return false;
		cash = balance;
		return true;
	}

}
