/*
 * Copyright (c) 2009-2010 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.csv.handlers;

import java.io.IOException;
import java.lang.reflect.Field;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.Format;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Currency;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.SortedMap;
import java.util.TimeZone;
import java.util.TreeMap;

import jp.sf.orangesignal.csv.CsvReader;
import jp.sf.orangesignal.csv.CsvWriter;
import jp.sf.orangesignal.csv.annotation.CsvColumn;
import jp.sf.orangesignal.csv.annotation.CsvColumns;
import jp.sf.orangesignal.csv.annotation.CsvEntity;
import jp.sf.orangesignal.csv.filters.CsvNamedValueFilter;
import jp.sf.orangesignal.csv.manager.CsvEntityManager;

/**
 * 区切り文字形式データ注釈要素 {@link CsvEntity} で注釈付けされた Java プログラム要素のリストで区切り文字形式データアクセスを行うハンドラを提供します。
 * 
 * @author 杉澤 浩二
 * @see CsvEntity
 * @see CsvColumn
 * @see CsvColumns
 * @see CsvEntityManager
 */
public class CsvEntityListHandler<T> extends BeanListHandlerSupport<T, CsvEntityListHandler<T>> {

	/**
	 * 区切り文字形式データフィルタを保持します。
	 */
	private CsvNamedValueFilter valueFilter;

	/**
	 * コンストラクタです。
	 * 
	 * @param entityClass 区切り文字形式データ注釈要素 {@link CsvEntity} で注釈付けされた Java プログラム要素の型
	 * @throws IllegalArgumentException <code>entityClass</code> が <code>null</code> または不正な場合
	 */
	public CsvEntityListHandler(final Class<T> entityClass) {
		super(entityClass);
		if (entityClass.getAnnotation(CsvEntity.class) == null) {
			throw new IllegalArgumentException(String.format("No CsvEntity is available %s", entityClass.getName()));
		}
	}

	/**
	 * 区切り文字形式データフィルタを設定します。
	 * 
	 * @param filter 区切り文字形式データフィルタ
	 * @return このオブジェクトへの参照
	 * @since 1.2.3
	 */
	public CsvEntityListHandler<T> filter(final CsvNamedValueFilter filter) {
		this.valueFilter = filter;
		return this;
	}

	private static Format createFormat(final CsvColumn column, final Field f) {
		final String pattern = column.format();
		if (pattern.isEmpty()) {
			return null;
		}

		final Locale locale = column.language().isEmpty() ? Locale.getDefault() : new Locale(column.language(), column.country());
		if (Date.class.isAssignableFrom(f.getType())) {
			final SimpleDateFormat format = new SimpleDateFormat(pattern, locale);
			if (!column.timezone().isEmpty()) {
				format.setTimeZone(TimeZone.getTimeZone(column.timezone()));
			}
			return format;
		}
		final DecimalFormat format = new DecimalFormat(pattern, DecimalFormatSymbols.getInstance(locale));
		if (!column.currency().isEmpty()) {
			format.setCurrency(Currency.getInstance(column.currency()));
		}
		return format;
	}

	@Override
	public List<T> load(final CsvReader reader, final boolean ignoreScalar) throws IOException {
		if (reader == null) {
			throw new IllegalArgumentException("CsvReader must not be null");
		}

		// ヘッダ行が有効な場合は項目名の一覧を取得します。
		final List<String> names;
		if (getType().getAnnotation(CsvEntity.class).header()) {
			names = reader.readValues();
		} else {
			names = getColumnNames(getType());
		}

		final Field[] fields = getType().getDeclaredFields();
		prepare(names, fields);

		// すべてのデータを読取って繰返し処理します。
		final List<T> results = new ArrayList<T>();
		final boolean order = ignoreScalar || (orders != null && !orders.isEmpty());
		int offset = 0;
		List<String> values;
		while ((values = reader.readValues()) != null && (order || limit <= 0 || results.size() < limit)) {
			if (valueFilter != null && !valueFilter.accept(names, values)) {
				continue;
			}
			if (beanFilter == null && !order && offset < this.offset) {
				offset++;
				continue;
			}

			final T entity = createBean();
			for (final Field f : fields) {
				Object o = null;
				final CsvColumns columns = f.getAnnotation(CsvColumns.class);
				if (columns != null) {
					final StringBuilder sb = new StringBuilder();
					for (final CsvColumn column : columns.value()) {
						final int pos = getPosition(column, f, names);
						if (pos != -1) {
							final String s = values.get(pos);
							if (s != null) {
								sb.append(s);
							}
						}
					}
					o = stringToObject(f, sb.toString());
				}
				final CsvColumn column = f.getAnnotation(CsvColumn.class);
				if (column != null) {
					final int pos = getPosition(column, f, names);
					if (pos != -1) {
						o = stringToObject(f, values.get(pos));
					}
				}
				if (o != null) {
					setFieldValue(entity, f, o);
				}
			}
			if (beanFilter != null) {
				if (!beanFilter.accept(entity)) {
					continue;
				}
				if (!order && offset < this.offset) {
					offset++;
					continue;
				}
			}
			results.add(entity);
		}

		if (ignoreScalar || !order) {
			return results;
		}
		return processScalar(results);
	}

	@Override
	public void save(final List<T> entities, final CsvWriter writer) throws IOException {
		if (entities == null) {
			throw new IllegalArgumentException("CsvEntities must not be null");
		} else if (writer == null) {
			throw new IllegalArgumentException("CsvWriter must not be null");
		}

		final List<String> names = getColumnNames(getType());
		// ヘッダ行が有効な場合は項目名の一覧を出力します。
		if (getType().getAnnotation(CsvEntity.class).header()) {
			writer.writeValues(names);
		}

		prepare(names, getType().getDeclaredFields());

		// データ出力
		final int columnCount = names.size();
		for (final T entity : entities) {
			if (entity == null || entity.getClass().getAnnotation(CsvEntity.class) == null) {
				writer.writeValues(null);
				continue;
			} else if (beanFilter != null && !beanFilter.accept(entity)) {
				continue;
			}

			final String[] values = new String[columnCount];
			for (final Field f : entity.getClass().getDeclaredFields()) {
				final CsvColumns columns = f.getAnnotation(CsvColumns.class);
				if (columns != null) {
					for (final CsvColumn column : columns.value()) {
						int pos = column.position();
						if (pos < 0) {
							pos = names.indexOf(defaultIfEmpty(column.name(), f.getName()));
						}
						if (pos == -1) {
							throw new IOException(String.format("Invalid CsvColumn field %s", f.getName()));
						}
						values[pos] = objectToString(pos, getFieldValue(entity, f));
					}
				}
				final CsvColumn column = f.getAnnotation(CsvColumn.class);
				if (column != null) {
					int pos = column.position();
					if (pos < 0) {
						pos = names.indexOf(defaultIfEmpty(column.name(), f.getName()));
					}
					if (pos == -1) {
						throw new IOException(String.format("Invalid CsvColumn field %s", f.getName()));
					}
					values[pos] = objectToString(pos, getFieldValue(entity, f));
				}
			}
			final List<String> _values = Arrays.asList(values);
			if (valueFilter != null && !valueFilter.accept(names, _values)) {
				continue;
			}
			writer.writeValues(_values);
		}
	}

	private void prepare(final List<String> names, final Field[] fields) {
		super.valueParserMapping(new HashMap<String, Format>(0));
		super.valueFormatterMapping(new HashMap<Object, Format>(0));

		// 書式オブジェクトの準備を行います。
		for (final Field f : fields) {
			final CsvColumns columns = f.getAnnotation(CsvColumns.class);
			if (columns != null) {
				for (final CsvColumn column : columns.value()) {
					final Format format = createFormat(column, f);
					if (format != null) {
						setValueParser(f.getName(), format);
						setValueFormatter(getPosition(column, f, names), format);
					}
				}
			}
			final CsvColumn column = f.getAnnotation(CsvColumn.class);
			if (column != null) {
				final Format format = createFormat(column, f);
				if (format != null) {
					setValueParser(f.getName(), format);
					setValueFormatter(getPosition(column, f, names), format);
				}
			}
		}
	}

	/**
	 * 指定された区切り文字形式データ注釈要素 {@link CsvEntity} で注釈付けされた Java プログラム要素の型から項目名のリストを構築して返します。
	 * 
	 * @param entityClass エンティティクラス
	 * @return
	 */
	private static List<String> getColumnNames(final Class<?> entityClass) {
		final SortedMap<Integer, String> positionMap = new TreeMap<Integer, String>();
		final List<String> adding = new ArrayList<String>();

		for (final Field f : entityClass.getDeclaredFields()) {
			final CsvColumns columns = f.getAnnotation(CsvColumns.class);
			if (columns != null) {
				for (final CsvColumn column : columns.value()) {
					final int pos = column.position();
					final String name = defaultIfEmpty(column.name(), f.getName());
					if (pos >= 0) {
						if (positionMap.containsKey(pos)) {
							continue;
						}
						positionMap.put(pos, name);
					} else {
						adding.add(name);
					}
				}
			}
			final CsvColumn column = f.getAnnotation(CsvColumn.class);
			if (column != null) {
				final int pos = column.position();
				final String name = defaultIfEmpty(column.name(), f.getName());
				if (pos >= 0) {
					if (positionMap.containsKey(pos)) {
						continue;
					}
					positionMap.put(pos, name);
				} else {
					adding.add(name);
				}
			}
		}

		final int max = positionMap.size() > 0 ? positionMap.lastKey().intValue() + 1 : 0;
		final String[] names = new String[max];
		for (final Map.Entry<Integer, String> entry : positionMap.entrySet()) {
			names[entry.getKey().intValue()] = entry.getValue();
		}

		final List<String> results = new ArrayList<String>(Arrays.asList(names));
		if (adding.size() > 0) {
			results.addAll(adding);
		}
		return results;
	}

	private static int getPosition(final CsvColumn column, final Field f, final List<String> names) {
		// 項目位置が指定されている場合は、項目位置から値を取得します。
		int pos = column.position();
		// 項目位置が指定されておらずヘッダ行が使用可能な場合は項目名から値を取得します。
		if (pos < 0 && names != null) {
			pos = names.indexOf(defaultIfEmpty(column.name(), f.getName()));
		}
		// 項目名と項目位置のどちらも使用できない場合は例外をスローします。
		if (pos == -1) {
			throw new IllegalStateException(String.format("Invalid CsvColumn field %s", f.getName()));
		}
		return pos;
	}

	private static String defaultIfEmpty(final String str, final String defaultStr) {
		return (str == null || str.isEmpty()) ? defaultStr : str;
	}

}
