<?php

require_once 'PHPUnit/Framework/TestCase.php';

class FormatTest extends PHPUnit_Framework_TestCase {
    protected function setUp() {
        $context = array(
            'sfjp.group_name' => 'unittest',
            'site_root_url' => 'http://sourceforge.jp',
            'internal_url_regex' => '^http://sourceforge\.jp/',
            'svn_base_url' => 'http://svn/view',
        );
        $this->p = new \sfjp\Wiki\Parser($context);
    }

    protected function tearDown() {

    }

    public function testNew() {
        self::assertTrue(isset($this->p));
    }

    public function testParseSimpleRun() {
        $this->try_parse('', '');
        $this->try_parse('abc', '<p>abc</p>');
        $this->try_parse("abc abc\nabc", "<p>abc abc\nabc</p>");
    }

    public function testParseHeadings() {
        $this->try_parse('= abc =', '<h1 id="h1-abc">abc</h1>');
        $this->try_parse("= 日本語 =\r\n", '<h1 id="h1-.E6.97.A5.E6.9C.AC.E8.AA.9E">日本語</h1>');
        $this->try_parse('= abc', '<h1 id="h1-abc">abc</h1>');
        $this->try_parse('= abc = ', '<h1 id="h1-abc">abc</h1>');
        $this->try_parse('= abc ', '<h1 id="h1-abc">abc</h1>');
        $this->try_parse('= abc = #moge', '<h1 id="moge">abc</h1>');
        $this->try_parse('= abc #moge', '<h1 id="moge">abc</h1>');
        $this->try_parse('=abc', '<p>=abc</p>');
        $this->try_parse(' = abc', '<div class="indent">= abc</div>');
    }

    public function testParseListMark() {
        $text = '
  * list1
    * level2
   * level2, too.
     continue list item.
   continue!
    continue also!
    * moge
 * level1 item
  * level1
  *this not list
';

        $expect = '<ul><li>list1
<ul><li>level2
</li><li>level2, too.
continue list item.
continue!
continue also!
</li><li>moge
</li></ul></li><li>level1 item
</li><li>level1
*this not list
</li></ul>';
        self::assertEquals($expect, $this->p->parse($text));
    }

    public function testParseMixedList() {
        $text = '
  * ul
  1. ol num
  a. ol alpha
  A. ol ALPHA
  * ul
  i. ol roma
  * ul
  I. ol ROMA
';

        $expect = '<ul><li>ul
</li></ul><ol><li>ol num
</li></ol><ol style="list-style-type: lower-alpha;"><li>ol alpha
</li></ol><ol style="list-style-type: upper-alpha;"><li>ol ALPHA
</li></ol><ul><li>ul
</li></ul><ol style="list-style-type: lower-roman;"><li>ol roma
</li></ol><ul><li>ul
</li></ul><ol style="list-style-type: upper-roman;"><li>ol ROMA
</li></ol>';
        self::assertEquals($expect, $this->p->parse($text));

        $text = '
  * list1
    1. level2
    100. ordered
    99. moge
  * level1 item
    * level2 unordered
      1. abc
      1. def
    * foo
  1. level1 ordered
    * unordered in ordered
  2. ordered list
';

        $expect = '<ul><li>list1
<ol><li>level2
</li><li>ordered
</li><li>moge
</li></ol></li><li>level1 item
<ul><li>level2 unordered
<ol><li>abc
</li><li>def
</li></ol></li><li>foo
</li></ul></li></ul><ol><li>level1 ordered
<ul><li>unordered in ordered
</li></ul></li><li>ordered list
</li></ol>';
        self::assertEquals($expect, $this->p->parse($text));
    }

  /*
    public function testParseParagraph() {
        self::assertEquals('yet', $this->p->parse('=abc'));
    }
  */

    public function testParseListNum() {
        $text = '
  1. list1
    3. level2
   999999. level2, too.
    1234. moge
 3. level1 item
  0000. level1
  1.3. this not list
';

        $expect = '<ol><li>list1
<ol><li>level2
</li><li>level2, too.
</li><li>moge
</li></ol></li><li>level1 item
</li><li>level1
1.3. this not list
</li></ol>';
        self::assertEquals($expect, $this->p->parse($text));
    }

  /*
    public function testParseDefinitionList() {
        self::assertEquals('yet', $this->p->parse('=abc'));
    }
  */

    public function testParseHTMLEscape() {
        $this->try_parse("'''>moge'''<", '<p><strong>&gt;moge</strong>&lt;</p>');
    }

    public function testParseInlineBold() {
        $this->try_parse("'''moge'''", '<p><strong>moge</strong></p>');
    }

    public function testParseInlineItalic() {
        $this->try_parse("''abc''", '<p><em>abc</em></p>');
        $this->try_parse("''ab", '<p><em>ab</em></p>');
        $this->try_parse("ab ''bc\n''cd", "<p>ab <em>bc\n</em>cd</p>");
    }

    public function testParseInlineBolditalic() {
        $this->try_parse("'''''moge'''''",
            '<p><strong><em>moge</em></strong></p>');
        $this->try_parse("'''''''''abc'",
            "<p><strong><em><strong>'abc'</strong></em></strong></p>");
    }

    public function testParseInlineUndeline() {
        $this->try_parse("__abc__",
            '<p><span style="text-decoration: underline;">abc</span></p>');
    }

    public function testParseInlineStrike() {
        $this->try_parse('~~abc~~', '<p><del>abc</del></p>');
        $this->try_parse('~~~abcef~~~~~', '<p><del>~abcef</del><del>~</del></p>');
    }

    public function testParseInlineSuperscript() {
        $this->try_parse('^ abc^', '<p><sup> abc</sup></p>');
        $this->try_parse('^^^ abc ^^', '<p><sup></sup><sup> abc </sup><sup></sup></p>');
        $this->try_parse("head ^ abc\ndef^tail", "<p>head <sup> abc\ndef</sup>tail</p>");
    }

    public function testParseInlineSubscript() {
        $this->try_parse(',,abc ,,', '<p><sub>abc </sub></p>');
    }

    public function testParseLinks() {
        $this->try_parse('http://url.com/path', '<p><a href="http://url.com/path" class="external" rel="nofollow">http://url.com/path</a></p>');
        $this->try_parse('https://url.com/path', '<p><a href="https://url.com/path" class="external" rel="nofollow">https://url.com/path</a></p>');
    }

    public function testParseBlacketLink() {
        self::assertEquals(
            '<p><a href="./a%2Fb%2F%E6%97%A5%E6%9C%AC%E8%AA%9E">a/b/日本語</a></p>',
            $this->p->parse("[a/b/日本語]")
        );
    }

    public function testParseLinkFragment() {
        $this->try_parse('[test1#frag]', '<p><a href="./test1#frag">test1#frag</a></p>');
        $this->try_parse('[test2#frag text]', '<p><a href="./test2#frag">text</a></p>');
        $this->try_parse('wiki:test3#frag', '<p><a href="./test3#frag">wiki:test3#frag</a></p>');

        $this->try_parse('wiki:"test4#frag"', '<p><a href="./test4#frag">wiki:&quot;test4#frag&quot;</a></p>');
        $this->try_parse('["test5#frag" text]', '<p><a href="./test5#frag">text</a></p>');
        $this->try_parse('[wiki:"test6#frag" text]', '<p><a href="./test6#frag">text</a></p>');
        $this->try_parse('[wiki:"test7 page name#frag" text]', '<p><a href="./test7%20page%20name#frag">text</a></p>');
        $this->try_parse('[#frag]', '<p><a href="#frag">#frag</a></p>');
        $this->try_parse('["#frag" text]', '<p><a href="#frag">text</a></p>');

    }

    public function testParseQuote() {
        self::assertEquals(
            "<blockquote class=\"citation\"><blockquote class=\"citation\"><p>abc
</p></blockquote><p>def
ghi
</p><blockquote class=\"citation\"><blockquote class=\"citation\"><p>jkl
</p></blockquote></blockquote></blockquote><p>normal</p>",
            $this->p->parse(">> abc\n> def\n> ghi\n>>> jkl\nnormal")
        );
$this->try_parse("> abc\n> > with space\n> >   >  3rd", '<blockquote class="citation"><p>abc
</p><blockquote class="citation"><p>with space
</p><blockquote class="citation"><p> 3rd</p></blockquote></blockquote></blockquote>');
    }

    public function testParseIndent() {
      $this->try_parse(" abc", '<div class="indent">abc</div>');
      $this->try_parse(" abc\n  def\nghi",
                         '<div class="indent">abc
def
</div><p>ghi</p>');
      self::assertEquals('<div class="indent">abc
def
</div><div class="indent"><div class="indent">2nd nest
</div></div><div class="indent"><div class="indent"><div class="indent">3rd
</div></div></div><div class="indent"><div class="indent">2nd
</div></div><ul><li>clear by list</li></ul>',
                         $this->p->parse(" abc\n  def\n    2nd nest\n      3rd\n    2nd\n * clear by list"));
    }

    public function testParseInternalURIPrweb() {
        $this->try_parse("prweb:/", '<p><a href="http://unittest.sourceforge.jp/" class="project-web">prweb:/</a></p>');
        $this->try_parse("prweb:/path/to/file", '<p><a href="http://unittest.sourceforge.jp/path/to/file" class="project-web">prweb:/path/to/file</a></p>');
        $this->try_parse("prweb:project-name:/url-to/the.page", '<p><a href="http://project-name.sourceforge.jp/url-to/the.page" class="project-web">prweb:project-name:/url-to/the.page</a></p>');
    }

    public function testParseInternalURIUser() {
        $this->try_parse("user:sugi", '<p><a href="/users/sugi" class="user">user:sugi</a></p>');
        $this->try_parse("id:sugi", '<p><a href="/users/sugi" class="user">id:sugi</a></p>');
        $this->try_parse("users:sugi", '<p><a href="/users/sugi" class="user">users:sugi</a></p>');
    }

    public function testParseURIMailTo() {
        $this->try_parse("mailto:sugi@osdn.jp", '<p><a href="mailto:sugi@osdn.jp" class="mail">mailto:sugi@osdn.jp</a></p>');
        $this->try_parse("[mailto:a.b=c+d@e.f メール]", '<p><a href="mailto:a.b=c+d@e.f" class="mail">メール</a></p>');
        $this->try_parse("mailto:bad@てすと", '<p>mailto:bad@てすと</p>');
    }

    public function testParseEscape() {
        $this->try_parse("!`", '<p>`</p>');
        $this->try_parse("!^てすと!^", '<p>^てすと^</p>');
        $this->try_parse("!~~", '<p>~~</p>');
        $this->try_parse("!__", '<p>__</p>');
        $this->try_parse("!WikiName", '<p>WikiName</p>');
        $this->try_parse("![[Plugin]]", '<p>[[Plugin]]</p>');
        $this->try_parse("!!", '<p>!!</p>');
    }

    public function testParseEscapeBlock() {
        $this->try_parse("!> Equote", '<p>&gt; Equote</p>');
        $this->try_parse("!------", '<p>------</p>');
        $this->try_parse("!||escaped||table||", '<p>||escaped||table||</p>');
        $this->try_parse("!= not header =", '<p>= not header =</p>');
        $this->try_parse(" !* abc", '<div class="indent">* abc</div>');
    }

    public function testHttpUrl() {
        self::assertTrue(!!$this->p->processor->gen_uri_link("http://てすと"));
        self::assertTrue(!!$this->p->processor->gen_uri_link("http://abc"));
        self::assertTrue(!$this->p->processor->gen_uri_link("http://"));
        self::assertTrue(!$this->p->processor->gen_uri_link("http:// abc"));
    }

    public function testIsbnLink() {
        $this->try_parse("isbn:123",
            '<p><a href="http://www.amazon.co.jp/gp/product/123" class="isbnbook" rel="nofollow">isbn:123</a></p>');
        $this->p->setContext(array('amazon_affiliate_id' => 'afid-test'));
        $this->try_parse("isbn:123",
            '<p><a href="http://www.amazon.co.jp/gp/product/123?tag=afid-test&amp;linkCode=as2&amp;camp=247&amp;creative=1211&amp;creativeASIN=123" class="isbnbook" rel="nofollow">isbn:123</a></p>');
    }

    public function testParseGroupWikiPage() {
        $this->try_parse('wiki:groupname:PageName', '<p><a href="http://sourceforge.jp/projects/groupname/wiki/PageName" class="external-wiki">wiki:groupname:PageName</a></p>');
        $this->try_parse('wiki:group-name:PageName', '<p><a href="http://sourceforge.jp/projects/group-name/wiki/PageName" class="external-wiki">wiki:group-name:PageName</a></p>');
        $this->try_parse('wiki:group_name:PageName', '<p><a href="./group_name">wiki:group_name</a>:<a href="./PageName">PageName</a></p>');

    }

    public function testParseOrer() {
        $this->try_parse('[PageName [[BR]]]',
            '<p><a href="./PageName">[[BR</a>]]</p>');
      /* do not support currently
      $this->try_parse("abc'''def[PageName la''bee''eel]tex'''t",
                       '<p>abc<b>def<a href="./PageName">la<i>bee</i>eel</a>tex</b>t</p>');
       */
        $this->try_parse("abc'''def[PageName la''bee''eel]tex'''t",
            "<p>abc<strong>def<a href=\"./PageName\">la''bee''eel</a>tex</strong>t</p>");

        $this->try_parse('[__Page,,Name^^ label]', '<p><a href="./__Page%2C%2CName%5E%5E">label</a></p>');
        $this->try_parse('["__Page,,Name^^" label]', '<p><a href="./__Page%2C%2CName%5E%5E">label</a></p>');
        $this->try_parse('[wiki:"__Page,,Name^^" label]', '<p><a href="./__Page%2C%2CName%5E%5E">label</a></p>');
        $this->try_parse('http://aaaa[bbb]ccc', '<p><a href="http://aaaa" class="external" rel="nofollow">http://aaaa</a><a href="./bbb">bbb</a>ccc</p>');
        $this->try_parse('!http://aaaa[bbb]ccc', '<p>http://aaaa<a href="./bbb">bbb</a>ccc</p>');
        $this->try_parse('http://aaaa![bbb]ccc', '<p><a href="http://aaaa" class="external" rel="nofollow">http://aaaa</a>[bbb]ccc</p>');
        $this->try_parse('!http://aaaa![bbb]ccc', '<p>http://aaaa[bbb]ccc</p>');
    }

    public function testQuotedLink() {
        $this->try_parse('["test page__ name"]', '<p><a href="./test%20page__%20name">test page__ name</a></p>');
        $this->try_parse('["a b c" label]', '<p><a href="./a%20b%20c">label</a></p>');
        $this->try_parse('[wiki:"a b c" label]', '<p><a href="./a%20b%20c">label</a></p>');
        $this->try_parse('wiki:",,a b c__"', '<p><a href="./%2C%2Ca%20b%20c__">wiki:&quot;,,a b c__&quot;</a></p>');
    }

    public function testBracket() {
        $this->try_parse('[http://www label]', '<p><a href="http://www" class="external" rel="nofollow">label</a></p>');
        $this->try_parse('[WikiName label]', '<p><a href="./WikiName">label</a></p>');
        $this->try_parse('[normal text]', '<p><a href="./normal">text</a></p>');
    }

    public function testParseFalseLink() {
        $this->try_parse("[0]", '<p><a href="./0">0</a></p>');
        $this->try_parse('["0"]', '<p><a href="./0">0</a></p>');
        $this->try_parse("[0 text]", '<p><a href="./0">text</a></p>');
        $this->try_parse("[0#0]", '<p><a href="./0#badid-0">0#0</a></p>');
        $this->try_parse('["0#0" text]', '<p><a href="./0#badid-0">text</a></p>');
        $this->try_parse("wiki:0", '<p><a href="./0">wiki:0</a></p>');
        $this->try_parse("wiki:0#0", '<p><a href="./0#badid-0">wiki:0#0</a></p>');
        $this->try_parse("[#0]", '<p><a href="#badid-0">#0</a></p>');
        $this->try_parse("[#0 text]", '<p><a href="#badid-0">text</a></p>');
    }

    public function testTableAndHr() {
        $this->try_parse('
----
||table||
----
||table||
----
',
                         '<hr /><table class="wikitable"><tbody><tr><td>table</td></tr></tbody></table><hr /><table class="wikitable"><tbody><tr><td>table</td></tr></tbody></table><hr />');
    }


    public function testBlock() {
        $this->try_parse("{{{
moge
}}}
", '<pre>moge
</pre>');
        $this->try_parse("{{{
moge
", '<pre>moge
</pre>');
    }

    public function testInlinePreformatted() {
      $this->try_parse("''ab''c''d{{{ef  g''}} }}}h''}}}i''j.",
                       "<p><em>ab</em>c<em>d<tt>ef  g''}} </tt>h</em>}}}i<em>j.</em></p>");
    }

    public function testInternalLink() {
      $this->try_parse("[http://www.yahoo.co.jp/ external]",
                       '<p><a href="http://www.yahoo.co.jp/" class="external" rel="nofollow">external</a></p>');
      $this->try_parse("[http://sourceforge.jp/ internal]",
                       '<p><a href="http://sourceforge.jp/">internal</a></p>');
      $this->try_parse("[http://sourceforge.jp/projects/test/moge internal]",
                       '<p><a href="http://sourceforge.jp/projects/test/moge">internal</a></p>');
      $this->try_parse("[http://test.sourceforge.jp/ external]",
                       '<p><a href="http://test.sourceforge.jp/" class="external" rel="nofollow">external</a></p>');
    }

    public function testHeadingCounter() {
      $this->try_parse("
== a
== b
== a
== a
",
                       '<h2 id="h2-a">a</h2><h2 id="h2-b">b</h2><h2 id="h2-a-2">a</h2><h2 id="h2-a-3">a</h2>');
    }

    public function testUselessParagraph() {
      $po_out = '<div class="pageoutline"><div class="pageoutline-title"><div class="action"><button type="button" onClick="javascript:togglePageOutline(this)"><img src="//static.sourceforge.jp/wiki/images/icons/roll-up.gif" border="0"></button></div>Outline</div></div>';
      $this->try_parse("[[PageOutline]]

a

[[PageOutline]]

b
",
                       "$po_out
<p>a
</p>$po_out
<p>b
</p>");

      $this->try_parse("

{{{
pre
}}}


",
                       '<pre>pre
</pre>');

      $this->try_parse("a

{{{ html
<i>html block</i>
}}}

b
",
                       '<p>a
</p><i>html block</i>
<p>b
</p>');

    }

    public function testTableAndHTMLBlockBug() {
        $this->try_parse('||table||
{{{ html
<ins>HTML</ins>
}}}', '<table class="wikitable"><tbody><tr><td>table</td></tr></tbody></table><ins>HTML</ins>
');
    }

    public function testSlashLinks() {
        $this->try_parse('[/path/to/]', '<p><a href="/path/to/">/path/to/</a></p>');
        $this->try_parse('[/path/to/ moge]', '<p><a href="/path/to/">moge</a></p>');
        $this->try_parse('[//server.com/path moge]', '<p><a href="//server.com/path">moge</a></p>');
    }

    public function testTicketComment() {
        $this->try_parse('comment:4:15142:1235097254',
            '<p><a href="http://sourceforge.jp/ticket/browse.php?group_id=4&amp;tid=15142#comment:4:15142:1235097254" class="ticket">comment:4:15142:1235097254</a></p>');
        $this->try_parse('comment:1 comment:foo', '<p>comment:1 comment:foo</p>');
        $this->try_parse('[comment:4:15142:1235097254 hiromichi-m] への返信',
            '<p><a href="http://sourceforge.jp/ticket/browse.php?group_id=4&amp;tid=15142#comment:4:15142:1235097254" class="ticket">hiromichi-m</a> への返信</p>');
	$this->try_parse('[comment::123:456 グループID省略]なコメント', '<p><a href="http://sourceforge.jp/ticket/detail.php?id=123&amp;cid=456" class="ticket">グループID省略</a>なコメント</p>');
	$this->try_parse('[comment::123:456 グループID0]なコメントリンク', '<p><a href="http://sourceforge.jp/ticket/detail.php?id=123&amp;cid=456" class="ticket">グループID0</a>なコメントリンク</p>');
	$this->try_parse('[comment:123:456 2引数]なコメントリンク', '<p><a href="http://sourceforge.jp/ticket/detail.php?id=123&amp;cid=456" class="ticket">2引数</a>なコメントリンク</p>');
    }

    public function testIRI() {
        $iri  = 'http://假定された有機交流電燈の.ひとつの青い照明です/';
        $url = 'http://%E5%81%87%E5%AE%9A%E3%81%95%E3%82%8C%E3%81%9F%E6%9C%89%E6%A9%9F%E4%BA%A4%E6%B5%81%E9%9B%BB%E7%87%88%E3%81%AE.%E3%81%B2%E3%81%A8%E3%81%A4%E3%81%AE%E9%9D%92%E3%81%84%E7%85%A7%E6%98%8E%E3%81%A7%E3%81%99/';
        $this->try_parse("$iri ひかりはたもち、その電燈は失はれ", "<p><a href=\"{$url}\" class=\"external\" rel=\"nofollow\">$iri</a> ひかりはたもち、その電燈は失はれ</p>");
        $this->try_parse("[$iri]", "<p><a href=\"$url\" class=\"external\" rel=\"nofollow\">$iri</a></p>");
        $this->try_parse("[$iri ひかりはたもち、その電燈は失はれ]", "<p><a href=\"$url\" class=\"external\" rel=\"nofollow\">ひかりはたもち、その電燈は失はれ</a></p>");
        $this->try_parse('[/納豆.html ひきわり]', '<p><a href="/%E7%B4%8D%E8%B1%86.html">ひきわり</a></p>');
    }

    public function testSVNRev() {
        $a_attr = 'href="http://svn/view?view=rev&amp;root=unittest&amp;revision=123" class="svn"';
        $this->try_parse('r123', "<p><a $a_attr>r123</a></p>");
        $this->try_parse('abcr123', '<p>abcr123</p>');
        $this->try_parse('r123abc', '<p>r123abc</p>');
        $this->try_parse('lead r123 trail', "<p>lead <a $a_attr>r123</a> trail</p>");
        $this->try_parse('日本語r123テキスト', "<p>日本語<a $a_attr>r123</a>テキスト</p>");
    }


    public function testWithUTF8() {
	    ini_set('default_charset', 'utf-8');
	    ini_set('mbstring.internal_encoding', 'utf-8');
	    ini_set('mbstring.detect_order', 'EUC-JP,UTF-8,SJIS,JIS');
	    ini_set('mbstring.language', 'Japanese');
	    $input = '!SourceForge.JPシステムのバグを見つけた場合には [/projects/sourceforge/ticket?type%5B%5D=113&status%5B%5D=1 バグ]、
CVSリポジトリの調整やアカウント削除等のサポート要求は [/projects/sourceforge/ticket?type%5B%5D=114&status%5B%5D=1 サポートリクエスト]へ、システムへの追加機能の要望等は [/projects/sourceforge/ticket?type%5B%5D=115&status%5B%5D=1 機能リクエスト]へ登録いただくようお願いいたします。その他の問い合わせについては、[/docs/SourceForge.JP%E3%81%AE%E9%80%A3%E7%B5%A1%E5%85%88 連絡先についての文書]をよくお読みください。';
	    $exp = '<p>SourceForge.JPシステムのバグを見つけた場合には <a href="/projects/sourceforge/ticket?type%5B%5D=113&amp;status%5B%5D=1">バグ</a>、
CVSリポジトリの調整やアカウント削除等のサポート要求は <a href="/projects/sourceforge/ticket?type%5B%5D=114&amp;status%5B%5D=1">サポートリクエスト</a>へ、システムへの追加機能の要望等は <a href="/projects/sourceforge/ticket?type%5B%5D=115&amp;status%5B%5D=1">機能リクエスト</a>へ登録いただくようお願いいたします。その他の問い合わせについては、<a href="/docs/SourceForge.JP%E3%81%AE%E9%80%A3%E7%B5%A1%E5%85%88">連絡先についての文書</a>をよくお読みください。</p>';
	    $this->try_parse($input, $exp);
    }

    public function testTracKeepNewLineMode() {
	    $this->p->setContext(array('trac.keep_newline' => true));
	    $this->try_parse('
改行を
全部
br に
マップする
モード




パラグラフは適切に分割される必要がある

> block
> quote
> text
さて
', '<p>改行を
<br />全部
<br />br に
<br />マップする
<br />モード
<br /></p><p>パラグラフは適切に分割される必要がある
<br /></p><blockquote class="citation"><p>block
<br />quote
<br />text
<br /></p></blockquote><p>さて
<br /></p>');
	    $this->try_parse('
  * リストでも改行保持
    のテスト
  * です
', '<ul><li>リストでも改行保持
<br />のテスト
<br /></li><li>です
<br /></li></ul>');
    }

    public function testInlinePluginAtStartOfLine() {
	    $this->try_parse('[[br]]test!', '<p><br />test!</p>');
    }

    public function testHashTrackerLink() {
	    $this->try_parse('#10', '<p><a href="http://sourceforge.jp/tracker.php?id=10" class="tracker">#10</a></p>');
	    $this->try_parse('#50010', '<p><a href="http://sourceforge.jp/ticket/detail.php?id=50010" class="tracker">#50010</a></p>');
    }


    public function testDisabledLinkWithExclamation() {
        $context = array(
            'sfjp.group_name' => 'unittest',
            'site_root_url' => 'http://sourceforge.jp',
            'internal_url_regex' => '^http://sourceforge\.jp/',
            'svn_base_url' => 'http://svn/view',
            'disable.link.CamelCase' => true,
        );
        $p = new \sfjp\Wiki\Parser($context);
        self::assertEquals('<p>CamelCase</p>', $p->parse('!CamelCase'));
    }


    protected function try_parse($text, $expect) {
        self::assertEquals($expect, $this->p->parse($text));
    }

}

// vim: set sts=4 sw=4 expandtab:
