package osm.jp.kiban25000;

import javax.xml.parsers.*;
import org.w3c.dom.*;
import org.xml.sax.*;
import java.io.*;
import java.util.Iterator;
import java.util.TreeMap;

/**
 * ====基盤地図情報のダウンロード====
 *
 * まず最初に、基盤地図情報をダウンロードします。
 *
 * 国土地理院の基盤地図情報ダウンロードページから、ログイン画面へ進みログインして、「基盤地図情報 縮尺レベル25000 JPGIS形式」を選択し、対象の地域のファイルをダウンロードします。(「JPGIS(GML)形式」ではありません)
 * http://fgd.gsi.go.jp/download/
 *
 * 対象地域と項目の選択方法は、ダウンロードページの指示に従ってください。
 *
 * 「基盤地図情報項目」では「行政区画の境界線及び代表点」を選択してください。
 * 		「FG-JPS-39-05-Z001.zip」というような「FG-JPS-xx-05-Z001.zip」という形式のZIPファイルがダウンロードされます。
 *
 * ダウンロードした圧縮ファイルを解凍しておきます。
 * 解凍すると「FG-JPS-44209-CommBdry-20091109-0001.xml」というような「FG-JPS-xxxx-CommBdry-xxxxxxxx-xxxx.xml」という形式のXMLファイルが出来上がります。
 * @author yuu
 *
 */
public class ConvAdmPt25000 {
	File inputFile;
	String admCodeStr = "";
	int iCounter = 0;
	boolean link = true;
	TreeMap<String, AdmPt> admMap = null;

	/**
	 *  メイン
	 *
	 * <GI>
	 * <dataset>
	 *
	 * <AdmPt id="K6_1" uuid="fgoid:10-00150-11-1002-1530">
		 <fid>fgoid:10-00150-11-1002-1530</fid>
		 <lfSpanFr>
		  <jps:position>
		   <jps:date8601>2008-03-31</jps:date8601>
		  </jps:position>
		 </lfSpanFr>
		 <lfSpanTo>
		  <jps:position>
		   <jps:date8601>2999-12-31</jps:date8601>
		  </jps:position>
		 </lfSpanTo>

		<devDate>
		  <jps:position>
		    <jps:date8601>2011-01-31</jps:date8601>
		  </jps:position>
		</devDate>

		<orgGILvl>25000</orgGILvl>
		<vis>表示</vis>

		<pos id="K6_2" uuid="fgoid:10-00150-11-1002-1530-g">
		  <jps:CRS uuidref="fguuid:jgd2000.bl"/>
		  <jps:position>
		    <jps:coordinate>37.055674 139.338021</jps:coordinate>
		  </jps:position>
		</pos>

		<type>その他</type>
		<name>福島県南会津郡檜枝岐村</name>
		<admCode>07364</admCode>

		</AdmPt>

		</dataset>

		</GI>

	 */
	public static void main(String[] argv)
	{
		boolean link = true;
		for (String arg : argv) {
			if (arg.equals("-nolink")) {
				link = false;
			}
		}

		File dir = new File(".");
		File[] files = dir.listFiles();
		int counter = 0;
		for (File iFile : files) {
			if (checkFile(iFile)) {
				counter++;
				ConvAdmPt25000 t = new ConvAdmPt25000(iFile, link);
				t.runner();
			}
		}
		System.out.println("["+ counter +"]つのファイルを処理しました。");
	}

	public ConvAdmPt25000 (File iFile, boolean link) {
		this.inputFile = iFile;
		this.link = link;
	}

	public void runner() {
		DocumentBuilderFactory factory;
		DocumentBuilder        builder;
		Node root;

		this.iCounter = 0;
		String iStr = this.inputFile.getName();
		String[] nameAry = iStr.split("-");
		String fileName = "";
		for (int i=0; i < nameAry.length; i++) {
			if (i != 0) {
				fileName = fileName + "-";
			}
			fileName = fileName + nameAry[i];
			if (i == 2) {
				this.admCodeStr = nameAry[2];
			}
		}

		File outputFile = new File(iStr.substring(0, iStr.length() - 4) +".gpx");
		BufferedWriter wGpx = null;

		File htmlFile = new File(iStr.substring(0, iStr.length() - 4) +".html");
		BufferedWriter wHtml = null;

		File wikiFile = new File(iStr.substring(0, iStr.length() - 4) +".txt");
		BufferedWriter wWiki = null;

		File csvFile = new File(iStr.substring(0, iStr.length() - 4) +".csv");
		BufferedWriter wCsv = null;

		try {
			wGpx = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(outputFile), "UTF-8"));
			wHtml = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(htmlFile), "UTF-8"));
			wWiki = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(wikiFile), "UTF-8"));
			wCsv = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(csvFile), "UTF-8"));

			wGpx.write("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>");
			wGpx.newLine();
			wGpx.write("<gpx xmlns=\"http://www.topografix.com/GPX/1/1\" version=\"1.1\" creator=\"osmtracker-android\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.topografix.com/GPX/1/1 http://www.topografix.com/GPX/1/1/gpx.xsd \">");
			wGpx.newLine();

			wWiki.write("=== "+ admCodeStr +": "+ getCodeName(Integer.parseInt(admCodeStr)) +" ===");
			wWiki.newLine();
			wWiki.write("{| class=\"wikitable sortable\"");
			wWiki.newLine();
			wWiki.write("|- ");
			wWiki.newLine();
			wWiki.write("! style=\"width:70px\" | admCode");
			wWiki.newLine();
			wWiki.write("! | name");
			wWiki.newLine();
			wWiki.write("! style=\"width:70px\" | type");
			wWiki.newLine();
			wWiki.write("! | 備考/最終更新日");
			wWiki.newLine();

			wCsv.write("\"都道府県コード\",\"都道府県名\",\"admCode\",\"Name\",\"type\",\"LAT\",\"LON\",\"備考/最終更新日\"");
			wCsv.newLine();

			wHtml.write("<!DOCTYPE html>");
			wHtml.newLine();
			wHtml.write("<html><head><meta charset=\"utf-8\" /></head>");
			wHtml.newLine();
			wHtml.write("<body><table border='1'>");
			wHtml.newLine();
			wHtml.write("<tr>");
			wHtml.write("<td>admCode</td>");
			wHtml.write("<td>name</td>");
			wHtml.write("<td></td>");
			wHtml.write("</tr>");
			wHtml.newLine();

			factory = DocumentBuilderFactory.newInstance();
			builder = factory.newDocumentBuilder();
			factory.setIgnoringElementContentWhitespace(true);
			factory.setIgnoringComments(true);
			factory.setValidating(true);
			root    = builder.parse(inputFile.getName());

			//---------------------------------------
			admMap = new TreeMap<String, AdmPt>();
			showNodes(root, "");
			//---------------------------------------

			Iterator<String> it = admMap.keySet().iterator();
	        while (it.hasNext()) {
	            String code = (String) it.next();
	            AdmPt adm = admMap.get(code);

	            wWiki.write("|- ");
				wWiki.newLine();
				wWiki.write("| " + adm.code);
				wWiki.newLine();
				wWiki.write("| " + adm.name);
				wWiki.newLine();
				wWiki.write("| " + adm.type);
				wWiki.newLine();
				wWiki.write("| ");
				wWiki.newLine();
				
				wCsv.write("\""+ admCodeStr +"\",\""+ getCodeName(Integer.parseInt(admCodeStr)) +"\",");
				wCsv.write("\""+ adm.code +"\",\""+ adm.name +"\",\""+ adm.type +"\",");
				wCsv.write("\""+ adm.latStr +"\",\""+ adm.lonStr +"\",\""+ adm.dateStr +"\"");
				wCsv.newLine();

				wHtml.write("<tr>");
				wHtml.write("<td>"+ adm.code +"</td>");
				wHtml.write("<td>"+ adm.name +"</td>");
				wHtml.write("<td>"+ adm.type +"</td>");
				wHtml.write("</tr>");
				wHtml.newLine();

				wGpx.write("<wpt lat=\""+ adm.latStr +"\" lon=\""+ adm.lonStr +"\">");
				wGpx.newLine();
				wGpx.write(" <time>"+ adm.dateStr +"T00:00:00Z</time>");
				wGpx.newLine();
				wGpx.write(" <name><![CDATA["+ adm.name +"]]></name>");
				wGpx.newLine();
				wGpx.write("</wpt>");
				wGpx.newLine();
	        }

			wGpx.write("</gpx>");
			wGpx.newLine();

			wHtml.write("</table></body></html>");
			wHtml.newLine();

			wWiki.write("|}");
			wWiki.newLine();

		} catch (ParserConfigurationException e0) {
			System.out.println(e0.getMessage());
		} catch (SAXException e1){
			System.out.println(e1.getMessage());
		} catch (IOException e2) {
			System.out.println(e2.getMessage());
		}
		finally {
			if (wGpx != null) {
				try {
					wGpx.close();
					wHtml.close();
				}
				catch (IOException e) {
					e.printStackTrace();
				}
			}
			if (wWiki != null) {
				try {
					wWiki.close();
				}
				catch (IOException e) {
					e.printStackTrace();
				}
			}
			if (wCsv != null) {
				try {
					wCsv.close();
				}
				catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
	}

	public void showNodes(Node node, String space) throws IOException {
		NodeList nodes = node.getChildNodes();
		for (int i=0; i<nodes.getLength(); i++) {
			Node node2 = nodes.item(i);
			if (node2.getNodeName().equals("AdmPt")) {
				showAdmPt(node2);
			}
			else {
				showNodes(node2, space + "    ");
			}
		}
	}

	public void showAdmPt(Node node) throws IOException {
		String positionStr = "";
		String typeStr = "";
		String nameStr = "";
		String latStr = "";
		String lonStr = "";
		String dateStr = "";
		String codeStr = "";

		NodeList nodes = node.getChildNodes();
		for (int i=0; i < nodes.getLength(); i++) {
			Node node2 = nodes.item(i);
			if (node2.getNodeName().equals("admCode")) {
				codeStr = node2.getTextContent();
			}
			else if (node2.getNodeName().equals("pos")) {
				NodeList nodes3 = node2.getChildNodes();
				for (int j=0; j < nodes3.getLength(); j++) {
					Node node3 = nodes3.item(j);
					if (node3.getNodeName().equals("jps:position")) {
						NodeList nodes4 = node3.getChildNodes();
						for (int k=0; k < nodes4.getLength(); k++) {
							Node node4 = nodes4.item(k);
							if (node4.getNodeName().equals("jps:coordinate")) {
								positionStr = node4.getTextContent();
								String[] str4Ary = positionStr.split(" ");
								latStr = str4Ary[0];
								lonStr = str4Ary[1];
							}
						}
					}
				}
			}
			else if (node2.getNodeName().equals("devDate")) {
				NodeList nodes3 = node2.getChildNodes();
				for (int j=0; j < nodes3.getLength(); j++) {
					Node node3 = nodes3.item(j);
					if (node3.getNodeName().equals("jps:position")) {
						NodeList nodes4 = node3.getChildNodes();
						for (int k=0; k < nodes4.getLength(); k++) {
							Node node4 = nodes4.item(k);
							if (node4.getNodeName().equals("jps:date8601")) {
								dateStr = node4.getTextContent();
							}
						}
					}
				}
			}
			else if (node2.getNodeName().equals("type")) {
				typeStr = node2.getTextContent();

			}
			else if (node2.getNodeName().equals("name")) {
				nameStr = node2.getTextContent();
			}
		}

		if (admMap.get(codeStr) == null) {
			AdmPt adm = new AdmPt();
			adm.code = codeStr;
			adm.name = nameStr;
			adm.type = typeStr;
			adm.latStr = latStr;
			adm.lonStr = lonStr;
			adm.dateStr = dateStr;
			admMap.put(codeStr, adm);
		}
	}

	class AdmPt {
		public String code;
		public String name;
		public String type;
		public String latStr;
		public String lonStr;
		public String dateStr;
	}

	/**
	 *
	 * 例　[FG-JPS-07-AdmPt25000-20110131-0001.xml]
	 *
	 * @param f
	 * @return
	 */
	static boolean checkFile(File f) {
		String name = f.getName();
		if (!name.startsWith("FG-JPS-")) {
			return false;
		}
		if (!name.toUpperCase().endsWith(".XML")) {
			return false;
		}
		if (name.toUpperCase().indexOf("-ADMPT25000-") < 0) {
			return false;
		}
		return true;
	}

	public static String getCodeName(int code) {
		switch (code) {
		case 1:
			return "北海道";
		case 2:
			return "青森県";
		case 3:
			return "岩手県";
		case 4:
			return "宮城県";
		case 5:
			return "秋田県";
		case 6:
			return "山形県";
		case 7:
			return "福島県";
		case 8:
			return "茨城県";
		case 9:
			return "栃木県";
		case 10:
			return "群馬県";
		case 11:
			return "埼玉県";
		case 12:
			return "千葉県";
		case 13:
			return "東京都";
		case 14:
			return "神奈川県";
		case 15:
			return "新潟県";
		case 16:
			return "富山県";
		case 17:
			return "石川県";
		case 18:
			return "福井県";
		case 19:
			return "山梨県";
		case 20:
			return "長野県";
		case 21:
			return "岐阜県";
		case 22:
			return "静岡県";
		case 23:
			return "愛知県";
		case 24:
			return "三重県";
		case 25:
			return "滋賀県";
		case 26:
			return "京都府";
		case 27:
			return "大阪府";
		case 28:
			return "兵庫県";
		case 29:
			return "奈良県";
		case 30:
			return "和歌山";
		case 31:
			return "鳥取県";
		case 32:
			return "島根県";
		case 33:
			return "岡山県";
		case 34:
			return "広島県";
		case 35:
			return "山口県";
		case 36:
			return "徳島県";
		case 37:
			return "香川県";
		case 38:
			return "愛媛県";
		case 39:
			return "高知県";
		case 40:
			return "福岡県";
		case 41:
			return "佐賀県";
		case 42:
			return "長崎県";
		case 43:
			return "熊本県";
		case 44:
			return "大分県";
		case 45:
			return "宮崎県";
		case 46:
			return "鹿児島";
		case 47:
			return "沖縄県";
		}
		return "";
	}

}