
/* --------------------------------------------- */
/*  H8-3069F File access (FAT16/32) function     */
/*                                               */
/*  CPU    : Renesus H8/3069F 25MHz              */
/*  Memory : ROM 512KB, RAM 16KB E-RAM 2MB       */
/*                (c) KAZ.Imamura                */
/* --------------------------------------------- */

#include "fat.h"
#include "serial.h"

// change directory
// read file

extern unsigned char eram_start;
extern unsigned char eram_end;

#define MAX_RETRY		20
// Keep file list on current directory
typedef struct {
	unsigned long	FirstCluster;
	unsigned long	FileSize;
	char			Name[13];
	unsigned char	Attributes;
} DIR_INFO, *P_DIR_INFO;

// Search status for DIR information
typedef struct {
	unsigned long	RootCluster;			// In case of static table, it should be 0.
	unsigned long	CurrentCluster;			// In case of static table, it should be 0.
//	unsigned long	RequestCluster;			// In case of static table, it should be 0.
	unsigned long	CurrentSector;
	unsigned int	ByteOffset;
} DIR_ENTRY, *P_DIR_ENTRY;

// Read file status
typedef struct {
	unsigned long ReadSize;
	unsigned long FileSize;
	unsigned long CurrentCluster;
	unsigned long CurrentSector;
} READ_FILE_STS, *P_READ_FILE_STS;

// FAT position
typedef struct {
	unsigned long Sector;
	unsigned long Offset;
} FAT_POSITION, *P_FAT_POSITION;

// Directory status
enum dir_status{
	DIR_ENTRY_EOD,
	DIR_ENTRY_EMPTY,
	DIR_LONG_FNAME,
	DIR_ENTRY_READY,
};

// -------------------------------------------
//  Macros
// -------------------------------------------
#define CLUSTER2SECTOR(X)	( ((X - 2) * BPB_info.SecPerClus) + FirstDataSector )

// -------------------------------------------
//  Proto type definitions
// -------------------------------------------
// Global
void fat_20ms_handler(void);
int  fat_initialize(void);
int  fat_process(void);
int ui_function_fat(UI_COMMAND uicmd);
unsigned char fat_change_directory( unsigned long cluster );
unsigned char fat_read_file( unsigned long cluster, unsigned long file_size );

// Locals
static FAT_POSITION get_fat_position(unsigned long cluster );
static SUB_PROC_RESULT move_window ( unsigned long sector );

static int ui_function_fat_debug (UI_COMMAND uicmd, int param);
static int ui_function_fat_mbr_bpb(UI_COMMAND uicmd, int param);
static int ui_function_fat_dir(UI_COMMAND uicmd, int param);
static int ui_function_fat_nop(UI_COMMAND uicmd, int param);
static void make_display_text_dirinfo( unsigned char* tempbuf, P_DIR_INFO pDirInfo  );

// -------------------------------------------
//  Variables
// -------------------------------------------
// Global
// Locals
static int fat_disable_timer;
static int window_disable_timer;
static unsigned char fat_proc;
static unsigned char fat_error;
static unsigned char window_proc;
static unsigned char ui_fat_proc;

static unsigned char fat_reset_request;
static unsigned char *p_buffer_addr;

static MBR_INFORMARION MBR_info;
static BPB_INFORMARION BPB_info;

static unsigned long RootDirSectors;
static unsigned long FATSectors;
static unsigned long FirstDataSector;
static unsigned long TotalSect;
static unsigned long DataSectors;
static unsigned long CountofClusters;

static unsigned long CurrentWindowSector;
static unsigned int  TotalFileNum;
static unsigned char IsFAT32;

static DIR_ENTRY dir_entry;
static DIR_INFO dir_info[MAX_FILE_IN_DIR];		// Keep file list on current directory
static READ_FILE_STS ReadFileStatus;
static unsigned char WindowBuffer[WINDOW_SIZE];

static unsigned long current_byte_cnt;

enum fat_error_code {
	FAT_ERR_NONE,
	FAT_ILLEGAL_SECTOR_SIZE,				// Corrently, olny 512 byte sector size can be supported.
	FAT_ZERO_SECTOR_SIZE,
	FAT_CAN_NOT_SUPPORT_FAT12,
	FAT_PROCESS_ERROR = 100,
};

enum fat_process_mode {
	FAT_READY_WAIT,				// 00
	FAT_READ_MBR_01,			// 01
	FAT_READ_MBR_02,			// 02
	FAT_READ_BPB_01,			// 03
	FAT_READ_BPB_02,			// 04
	FAT_CHANGE_DIR_01,			// 05
	FAT_CHANGE_DIR_02,			// 06
	FAT_CHANGE_DIR_03,			// 07
	FAT_CHANGE_DIR_04,			// 08
	FAT_CHANGE_DIR_05,			// 09
	FAT_IDLE = 0x80,			// Idle
	FAT_READ_FILE_01,			// 81
	FAT_READ_FILE_02,			// 82
	FAT_READ_FILE_03,			// 83
	FAT_READ_FILE_04,			// 84
	FAT_READ_FILE_05,			// 85
	FAT_READ_FILE_06,			// 86
	FAT_READ_FILE_07,			// 87
	FAT_RESET,
	FAT_ERROR_STOP = 0xFF,
};


static const MODULE_MENU_TABLE fat_app_table[] = {					// interval time will be ignored.
	{	 0,		0,	ui_function_fat_debug,			"  Debug         ",	},
	{	 1,		0,	ui_function_fat_mbr_bpb,		"  MBR/BPB       ",	},
	{	 2,		0,	ui_function_fat_dir,			"  Browser       ",	},
	{	-1,		0,	ui_function_fat_nop,			"0123456789ABCDEF",	},
};


static const DEBUG_VAR_TABLE debug_var_table[] = {
	{	 0,		 1,	"Process number: ",	&fat_proc,			},
	{	 1,		 1,	"Error Code:     ",	&fat_error,			},
	{	 1,		 3,	"RootDirSectors: ",	&RootDirSectors,	},
	{	 1,		 3,	"FATSectors:     ",	&FATSectors,		},
	{	 1,		 3,	"FirstDataSector:",	&FirstDataSector,	},
	{	 1,		 3,	"TotalSect:      ",	&TotalSect,			},
	{	 1,		 3,	"DataSectors:    ",	&DataSectors,		},
	{	 1,		 3,	"CountofClusters:",	&CountofClusters,	},
	{	 1,		 1,	"IsFAT32:        ",	&IsFAT32,			},
	{	 1,		 2,	"TotalFileNum:   ",	&TotalFileNum,		},
	{	 1,		 3,	"CurrentWinSect: ",	&CurrentWindowSector,},
	{	 1,		 3,	"current_bytecnt:",	&current_byte_cnt,},
	{	-1,		 1,	"0123456789ABCDEF", 0,					},
};

static const DEBUG_VAR_TABLE mbr_bpb_var_table[] = {
	{	 0,		 1,	"MBR.FAT type:   ",	&MBR_info.fat_type,			},
	{	 1,		 3,	"MBR.BPB sector: ",	&MBR_info.fat_BPB_sector,	},
	{	 2,		 2,	"BPB.ResvdSecCnt:",	&BPB_info.ResvdSecCnt,		},
	{	 3,		 2,	"BPB.BytsPerSec: ",	&BPB_info.BytsPerSec,		},
	{	 4,		 1,	"BPB.NumFATs:    ",	&BPB_info.NumFATs,			},
	{	 5,		 2,	"BPB.RootEntCnt: ",	&BPB_info.RootEntCnt,		},
	{	 6,		 2,	"BPB.FATSz16:    ",	&BPB_info.FATSz16,			},
	{	 7,		 3,	"BPB.FATSz32:    ",	&BPB_info.FATSz32,			},
	{	 8,		 1,	"BPB.SecPerClus: ",	&BPB_info.SecPerClus,		},
	{	-1,		 1,	"0123456789ABCDEF", 0,							},
};

// -------------------------------------------
//  Interrupt handlers
// -------------------------------------------
void fat_20ms_handler(void) {
	if( fat_disable_timer )		fat_disable_timer--;
	if( window_disable_timer )	window_disable_timer--;
}

// -------------------------------------------
//  Initialize
// -------------------------------------------
int  fat_initialize(void) {
	fat_proc = FAT_READY_WAIT;
	fat_error = FAT_ERR_NONE;
	window_proc = 0;
	fat_reset_request = 1;
	fat_disable_timer = 0;
	window_disable_timer = 0;
	p_buffer_addr = &eram_start;
	
	CurrentWindowSector = 0;
	TotalFileNum = 0;
	IsFAT32 = 0;
	
	dir_entry.RootCluster		= 0;
	dir_entry.CurrentCluster	= 0;
//	dir_entry.RequestCluster	= 0;
	dir_entry.CurrentSector		= 0;
	dir_entry.ByteOffset		= 0;
	
	// UI related
	ui_fat_proc = 0x00;
}

// -------------------------------------------
//  Change Directory
// -------------------------------------------
unsigned char fat_change_directory( unsigned long cluster ) {
	if( fat_proc != FAT_IDLE) return CLASS_REQ_BUSY;
	
	if( cluster == 0 ) {			// Root directory ?
		if(IsFAT32) {
			dir_entry.RootCluster   = 2;
			dir_entry.CurrentSector = FirstDataSector;
		} else {
			dir_entry.RootCluster   = 0;
			dir_entry.CurrentSector = BPB_info.ResvdSecCnt + (BPB_info.NumFATs * FATSectors);
		}
	} else {
		dir_entry.RootCluster   = cluster;
		dir_entry.CurrentSector = CLUSTER2SECTOR( cluster );
	}
	dir_entry.ByteOffset = 0;
	dir_entry.CurrentCluster = dir_entry.RootCluster;
	
	fat_proc = FAT_CHANGE_DIR_01;
	
	return CLASS_REQ_ACCEPTED;
}

// -------------------------------------------
//  Read File
// -------------------------------------------
unsigned char fat_read_file( unsigned long cluster, unsigned long file_size ) {
	if( fat_proc != FAT_IDLE) return CLASS_REQ_BUSY;
	
	ReadFileStatus.ReadSize				= 0;
	ReadFileStatus.FileSize 			= file_size;
	ReadFileStatus.CurrentCluster		= cluster;
	ReadFileStatus.CurrentSector		= CLUSTER2SECTOR(cluster);
	
	fat_proc = FAT_READ_FILE_01;
	return CLASS_REQ_ACCEPTED;
}

// -------------------------------------------
//  Main process
// -------------------------------------------
int fat_process(void) {
	unsigned long val;
	int i;
	READ10_REQ read10_req;
	SUB_PROC_RESULT res;
	FAT_POSITION fat;

	switch( fat_proc ) {
		case FAT_READY_WAIT:
			if( usbms_status(CLASS_REQ_NONE) == CLASS_STS_READY ) {
				fat_proc = FAT_READ_MBR_01;
			} else {
				fat_disable_timer = 1000/20; 		// 1000ms Wait
			}
			break;
			
		case FAT_READ_MBR_01:
			read10_req.logical_block_address = 0;
			read10_req.transfer_length = 1;
			read10_req.p_buffer = WindowBuffer;
			if( usbms_read10( &read10_req ) == CLASS_REQ_ACCEPTED ) {
				fat_proc = FAT_READ_MBR_02;
			} else {
				fat_disable_timer = 200/20; 		// 200ms Wait
			}
			break;
			
		case FAT_READ_MBR_02:
			if( usbms_status(CLASS_REQ_NONE) == CLASS_STS_READY ) {
				// **** Take in MBR information from buffer
				MBR_info.fat_type = p_buffer_addr[450];
				MBR_info.fat_BPB_sector =	WindowBuffer[454]
										+   WindowBuffer[455] * 0x100
										+   WindowBuffer[456] * 0x10000
										+   WindowBuffer[457] * 0x1000000;
										
				fat_proc = FAT_READ_BPB_01;
			}
			break;
		
		case FAT_READ_BPB_01:
			read10_req.logical_block_address = MBR_info.fat_BPB_sector;
			read10_req.transfer_length = 1;
			read10_req.p_buffer = WindowBuffer;
			if( usbms_read10( &read10_req ) == CLASS_REQ_ACCEPTED ) {
				fat_proc = FAT_READ_BPB_02;
			} else {
				fat_disable_timer = 200/20; 		// 200ms Wait
			}
			break;
			
		case FAT_READ_BPB_02:
			if( usbms_status(CLASS_REQ_NONE) == CLASS_STS_READY ) {
				// **** Take in BPB information from buffer
				BPB_info.BytsPerSec   = WindowBuffer[ 11] +   WindowBuffer[ 12] * 0x100;
				BPB_info.ResvdSecCnt  = WindowBuffer[ 14] +   WindowBuffer[ 15] * 0x100;
				BPB_info.RootEntCnt   = WindowBuffer[ 17] +   WindowBuffer[ 18] * 0x100;
				BPB_info.FATSz16      = WindowBuffer[ 22] +   WindowBuffer[ 23] * 0x100;
				BPB_info.NumFATs      = WindowBuffer[ 16];
				BPB_info.SecPerClus   = WindowBuffer[ 13];
				BPB_info.TotSec16     = WindowBuffer[ 19] +   WindowBuffer[ 20] * 0x100;
				BPB_info.TotSec32     = WindowBuffer[ 32]
									  + WindowBuffer[ 33] * 0x100
									  + WindowBuffer[ 34] * 0x10000
									  + WindowBuffer[ 35] * 0x1000000;
									  
				if( BPB_info.FATSz16 == 0 ) {				// AREA Only for FAT32
					BPB_info.FATSz32 = WindowBuffer[ 36]
									+  WindowBuffer[ 37] * 0x100
									+  WindowBuffer[ 38] * 0x10000
									+  WindowBuffer[ 39] * 0x1000000;
					BPB_info.RootClus = WindowBuffer[ 44]
									+   WindowBuffer[ 45] * 0x100
									+   WindowBuffer[ 46] * 0x10000
									+   WindowBuffer[ 47] * 0x1000000;
				}
				
				if( BPB_info.BytsPerSec == 0 ) {
					fat_error = FAT_ZERO_SECTOR_SIZE;
					fat_proc = FAT_ERROR_STOP;
				}else if( BPB_info.BytsPerSec != WINDOW_SIZE ) {
					fat_error = FAT_ILLEGAL_SECTOR_SIZE;
					fat_proc = FAT_ERROR_STOP;
				}
				
				// **** Calculate file access parameters
				FATSectors			= BPB_info.FATSz16 ? BPB_info.FATSz16 : BPB_info.FATSz32;
				RootDirSectors		= ((BPB_info.RootEntCnt * 32) + (BPB_info.BytsPerSec - 1)) / BPB_info.BytsPerSec;
				FirstDataSector		= BPB_info.ResvdSecCnt + (BPB_info.NumFATs * FATSectors) + RootDirSectors;
				TotalSect			= BPB_info.TotSec16 ? BPB_info.TotSec16 : BPB_info.TotSec32;
				DataSectors			= TotalSect - FirstDataSector;
				CountofClusters		= DataSectors / BPB_info.SecPerClus;
				
				// **** Judgement of FAT type
				if( CountofClusters < 4085 ) {
					// FAT12					// TODO: This system will not support FAT12
					IsFAT32 = 0;
					fat_error = FAT_CAN_NOT_SUPPORT_FAT12;
					fat_proc = FAT_ERROR_STOP;
					break;
				} else if( CountofClusters >= 65525 ) {
					IsFAT32 = 1;
				}
				
				// **** Root directory setting
				if(IsFAT32) {
					dir_entry.RootCluster   = 2;
					dir_entry.CurrentSector = FirstDataSector;
				} else {
					dir_entry.RootCluster   = 0;
					dir_entry.CurrentSector = BPB_info.ResvdSecCnt + (BPB_info.NumFATs * FATSectors);
				}
				dir_entry.ByteOffset = 0;
				dir_entry.CurrentCluster = dir_entry.RootCluster;
				
				fat_proc = FAT_CHANGE_DIR_01;
			}
			break;
		
		case FAT_CHANGE_DIR_01:										// Dir initialize
			TotalFileNum = 0;
			fat_proc = FAT_CHANGE_DIR_02;
			// break;
		case FAT_CHANGE_DIR_02:										// move_window()
			res = move_window(dir_entry.CurrentSector);
			if( res == SUB_PROC_DONE ) {
				fat_proc = FAT_CHANGE_DIR_03;
			} else if( res == SUB_PROC_ERROR ){
				fat_error = FAT_PROCESS_ERROR + fat_proc;
				fat_proc = FAT_ERROR_STOP;
			}
			break;
			
		case FAT_CHANGE_DIR_03:										// get_dir_entry()
			if(WindowBuffer[dir_entry.ByteOffset+  0] == 0x00) {
				fat_proc = FAT_IDLE;		// end of entries
				break;
			} else if (WindowBuffer[dir_entry.ByteOffset+  0] == 0xE5) {
				fat_proc = FAT_CHANGE_DIR_04;		// Empty  --> next entry
				break;
			} else if ( (WindowBuffer[dir_entry.ByteOffset+ 11] & 0x0F) == DIR_ATTR_LONG_NAME) {
				fat_proc = FAT_CHANGE_DIR_04;		// Long File name information   ---> next entry
				break;
			} else {
				dir_info[TotalFileNum].FirstCluster =
								  WindowBuffer[dir_entry.ByteOffset+ 20] * 0x10000
								+ WindowBuffer[dir_entry.ByteOffset+ 21] * 0x1000000
								+ WindowBuffer[dir_entry.ByteOffset+ 26]
								+ WindowBuffer[dir_entry.ByteOffset+ 27] * 0x100;
				dir_info[TotalFileNum].Attributes = WindowBuffer[dir_entry.ByteOffset+ 11];
				
				val = *( (unsigned long*) &WindowBuffer[dir_entry.ByteOffset+ 28] );
				dir_info[TotalFileNum].FileSize = (val<<24) | ((val<<8) & 0x00FF0000) | ((val>>8) & 0x0000FF00) | ((val>>24) & 0x000000FF);
				
				dir_info[TotalFileNum].Name[ 0] = WindowBuffer[dir_entry.ByteOffset+  0];	dir_info[TotalFileNum].Name[ 1] = WindowBuffer[dir_entry.ByteOffset+  1];
				dir_info[TotalFileNum].Name[ 2] = WindowBuffer[dir_entry.ByteOffset+  2];	dir_info[TotalFileNum].Name[ 3] = WindowBuffer[dir_entry.ByteOffset+  3];
				dir_info[TotalFileNum].Name[ 4] = WindowBuffer[dir_entry.ByteOffset+  4];	dir_info[TotalFileNum].Name[ 5] = WindowBuffer[dir_entry.ByteOffset+  5];
				dir_info[TotalFileNum].Name[ 6] = WindowBuffer[dir_entry.ByteOffset+  6];	dir_info[TotalFileNum].Name[ 7] = WindowBuffer[dir_entry.ByteOffset+  7];
				
				if(dir_info[TotalFileNum].Attributes & DIR_ATTR_DIRECTORY) {
					dir_info[TotalFileNum].Name[ 8] = WindowBuffer[dir_entry.ByteOffset+  8];
					dir_info[TotalFileNum].Name[ 9] = WindowBuffer[dir_entry.ByteOffset+  9];
					dir_info[TotalFileNum].Name[10] = WindowBuffer[dir_entry.ByteOffset+ 10];
					dir_info[TotalFileNum].Name[11] = ' ';
					dir_info[TotalFileNum].Name[12] = 0x00;
				} else {
					dir_info[TotalFileNum].Name[ 8] = '.';
					dir_info[TotalFileNum].Name[ 9] = WindowBuffer[dir_entry.ByteOffset+  8];
					dir_info[TotalFileNum].Name[10] = WindowBuffer[dir_entry.ByteOffset+  9];
					dir_info[TotalFileNum].Name[11] = WindowBuffer[dir_entry.ByteOffset+ 10];
					dir_info[TotalFileNum].Name[12] = 0x00;
				}
				
				if( TotalFileNum < MAX_FILE_IN_DIR ) {
					TotalFileNum++;
					fat_proc = FAT_CHANGE_DIR_04;
				} else {
					fat_proc = FAT_IDLE;				// Can not read any more
				}
			}
			break;
		
		case FAT_CHANGE_DIR_04:										// next_dir_entry()
			dir_entry.ByteOffset += 32;			// DIR entry is 32 byte
			if( dir_entry.ByteOffset >= BPB_info.BytsPerSec ) { // sector changed ?
				dir_entry.ByteOffset = 0;
				dir_entry.CurrentSector++;
				if( dir_entry.CurrentSector % BPB_info.SecPerClus == 0 ) {	// cluster changed?
					if( dir_entry.RootCluster != 0 ) {
						fat_proc = FAT_CHANGE_DIR_05; // get next cluster
					} else {
						fat_proc = FAT_CHANGE_DIR_02;	// nothing to do to go next sector because of static table
					}
				} else {
					fat_proc = FAT_CHANGE_DIR_02;	// nothing to do to go next sector
				}
			} else {
				fat_proc = FAT_CHANGE_DIR_02;	// nothing to do to go next offset 
			}
			break;
			
		case FAT_CHANGE_DIR_05:										// GetNextCluster() only for dynamic table
			fat = get_fat_position(dir_entry.CurrentCluster);
			res = move_window(fat.Sector);
			if( res == SUB_PROC_DONE ) {
				unsigned long EntryVal;
				if(IsFAT32) {
#if 1
					val = *( (unsigned long*) &WindowBuffer[fat.Offset] );
					EntryVal = (val<<24) | ((val<<8) & 0x00FF0000) | ((val>>8) & 0x0000FF00) | ((val>>24) & 0x000000FF);
					EntryVal &= 0x0FFFFFFF;
#else
					EntryVal = *(unsigned long*)  &WindowBuffer[fat.Offset] & 0x0FFFFFFF;
#endif
					if( (EntryVal > 0x00000002) && (EntryVal < 0x0FFFFFF6) ) {
						dir_entry.CurrentCluster = EntryVal;
						dir_entry.CurrentSector = CLUSTER2SECTOR(dir_entry.CurrentCluster);
					}
				} else {
#if 1
					val = *( (unsigned long*) &WindowBuffer[fat.Offset] );
					EntryVal = (unsigned short) (((val<<8) & 0xFF00) | ((val>>8) & 0x00FF));
#else
					EntryVal = *(unsigned short*) &WindowBuffer[fat.Offset];
#endif
					if( (EntryVal > 0x0002) && (EntryVal < 0xFFF6) ) {
						dir_entry.CurrentCluster = (unsigned long) EntryVal;
						dir_entry.CurrentSector = CLUSTER2SECTOR(dir_entry.CurrentCluster);
					}
				}
				fat_proc = FAT_CHANGE_DIR_02;
			} else if( res == SUB_PROC_ERROR ){
				fat_error = FAT_PROCESS_ERROR + fat_proc;
				fat_proc = FAT_ERROR_STOP;					// TODO: Error process
			}
			break;
			
		case FAT_IDLE:
			break;
			
		case FAT_READ_FILE_01:										// move_window()
			res = move_window(ReadFileStatus.CurrentSector);
			if( res == SUB_PROC_DONE ) {
				fat_proc = FAT_READ_FILE_02;
			} else if( res == SUB_PROC_ERROR ){
				fat_error = FAT_PROCESS_ERROR + fat_proc;
				fat_proc = FAT_ERROR_STOP;
			}
			break;
			
		case FAT_READ_FILE_02:										// Copy buffer
			for(i=0; i<WINDOW_SIZE; i++) {
				p_buffer_addr[ReadFileStatus.ReadSize++] = WindowBuffer[i];
				if( ReadFileStatus.ReadSize >= ReadFileStatus.FileSize ) break;
			}
			fat_proc = FAT_READ_FILE_03;
			break;
			
		case FAT_READ_FILE_03:										// next_dir_entry
			if( ReadFileStatus.ReadSize >= ReadFileStatus.FileSize ) {
				fat_proc = FAT_IDLE;
			} else {
				ReadFileStatus.CurrentSector++;
				if( ReadFileStatus.CurrentSector % BPB_info.SecPerClus == 0 ) {	// cluster changed?
					fat_proc = FAT_READ_FILE_04;		// Get next cluster
				} else {
					fat_proc = FAT_READ_FILE_01;		// nothing to do to go next sector
				}
			}
			break;
			
		case FAT_READ_FILE_04:										// GetNextCluster
			fat = get_fat_position(ReadFileStatus.CurrentCluster);
			res = move_window(fat.Sector);
			if( res == SUB_PROC_DONE ) {
				unsigned long EntryVal;
				if(IsFAT32) {
#if 1
					val = *( (unsigned long*) &WindowBuffer[fat.Offset] );
					EntryVal = (val<<24) | ((val<<8) & 0x00FF0000) | ((val>>8) & 0x0000FF00) | ((val>>24) & 0x000000FF);
					EntryVal &= 0x0FFFFFFF;
#else
					EntryVal = *(unsigned long*)  &WindowBuffer[fat.Offset] & 0x0FFFFFFF;
#endif
					if( (EntryVal > 0x00000002) && (EntryVal < 0x0FFFFFF6) ) {
						dir_entry.CurrentCluster = EntryVal;
						dir_entry.CurrentSector = CLUSTER2SECTOR(dir_entry.CurrentCluster);
					}
				} else {
#if 1
					val = *( (unsigned long*) &WindowBuffer[fat.Offset] );
					EntryVal = (unsigned short) (((val<<8) & 0xFF00) | ((val>>8) & 0x00FF));
#else
					EntryVal = *(unsigned short*) &WindowBuffer[fat.Offset];
#endif
					if( (EntryVal > 0x0002) && (EntryVal < 0xFFF6) ) {
						dir_entry.CurrentCluster = (unsigned long) EntryVal;
						dir_entry.CurrentSector = CLUSTER2SECTOR(dir_entry.CurrentCluster);
					}
				}
				fat_proc = FAT_READ_FILE_01;
			} else if( res == SUB_PROC_ERROR ){
				fat_error = FAT_PROCESS_ERROR + fat_proc;
				fat_proc = FAT_ERROR_STOP;					// TODO: Error process
			}
			break;
			
		case FAT_ERROR_STOP:
		default:
			break;
	}
}


// ------------------------------------------------------
//  Local function - [SUB PROC] Move window
// ------------------------------------------------------
static SUB_PROC_RESULT move_window ( unsigned long sector ) {
	READ10_REQ read10_req;
	SUB_PROC_RESULT res;
	static unsigned long last_sector;
	static unsigned char retry_count;
	
	if( last_sector != sector ) {
		window_proc = 0;
		window_disable_timer = 0;
		last_sector = sector;
	}
	
	if( window_disable_timer ) return SUB_PROC_BUSY;
	
	switch( window_proc ) {
		case 0x00:
			if( CurrentWindowSector == sector ) return SUB_PROC_DONE;
			retry_count = 0;
			window_proc = 0x01;
			// break;
			
		case 0x01:
			read10_req.logical_block_address = sector + MBR_info.fat_BPB_sector;
			read10_req.transfer_length = 1;
			read10_req.p_buffer = WindowBuffer;
			
			res = usbms_read10( &read10_req );
			if( res == CLASS_REQ_ACCEPTED )		window_proc = 0x02;
			else if( res == CLASS_REQ_REFUSED ) return SUB_PROC_ERROR;
			else {
				window_disable_timer = 200/20; 		// 200ms Wait
				if( ++retry_count > MAX_RETRY ) return SUB_PROC_ERROR;
			}
			break;
			
		case 0x02:
			if( usbms_status(CLASS_REQ_NONE) == CLASS_STS_READY ) {
				CurrentWindowSector = sector;
				window_proc = 0x00;
				return SUB_PROC_DONE;
			} else if( usbms_status(CLASS_REQ_NONE) == CLASS_STS_ERROR_STOP ) {
				return SUB_PROC_ERROR;
			}
			break;
		default:
			window_proc = 0x00;
			return SUB_PROC_DONE;
			break;
	}
	return SUB_PROC_BUSY;
}


// ------------------------------------------------------
//  Local function - Get FAT position (no wait)
// ------------------------------------------------------
static FAT_POSITION get_fat_position(unsigned long cluster ) {
	FAT_POSITION fat;
	
	if( BPB_info.BytsPerSec == 0 ) {
		fat.Sector = 0;
		return fat;
	}
	
	if(IsFAT32) {
		fat.Sector = BPB_info.ResvdSecCnt + (cluster * 4 / BPB_info.BytsPerSec);
		fat.Offset = (cluster * 4) % BPB_info.BytsPerSec;
	} else {
		fat.Sector = BPB_info.ResvdSecCnt + (cluster * 2 / BPB_info.BytsPerSec);
		fat.Offset = (cluster * 2) % BPB_info.BytsPerSec;
	}
	return fat;
}





// -------------------------------------------
//  UI Function - FAT function
// -------------------------------------------
int ui_function_fat(UI_COMMAND uicmd) {
	static unsigned int current_index;
	static unsigned int cursol_position;
	int ret_val;
	UI_COMMAND ui_cmd_sub;

	ret_val = UI_RET_READY;
	
	switch( ui_fat_proc ) {
		case 0x00:
			// Event check
			switch( uicmd.cmd ) {
				case UI_CMD_NOP:
				case UI_CMD_INTEVAL:
					break;
					
				case UI_CMD_STARTED:
					current_index = 0;
					cursol_position = 0;
					break;
					
				case UI_CMD_KEY_PRESS_BACK:
					if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
					ret_val = UI_RET_QUIT;
					break;
					
				case UI_CMD_KEY_PRESS_UP:
					if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
					if( cursol_position == 1 ) {
						cursol_position = 0;
						current_index--;
					} else {
						if( current_index != 0 ) current_index--;
					}
					break;
				case UI_CMD_KEY_PRESS_DOWN:
					if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
					if( cursol_position == 0 ) {
						cursol_position = 1;
						current_index++;
					} else {
						if( fat_app_table[current_index+1].num != -1 ) current_index++;
					}
					break;
				case UI_CMD_KEY_PRESS_OK:
					if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
					ui_cmd_sub.cmd = UI_CMD_STARTED;
					fat_app_table[current_index].pExecFunc(ui_cmd_sub, fat_app_table[current_index].param);	// Initialize
					ui_fat_proc = 0x01;
					break;
			}
			
			// Menu view
			if( cursol_position == 0) { // cursol in first line
				sc1602_set_buffer_cursol( 0, fat_app_table[current_index  ].display );
				sc1602_set_buffer       ( 1, fat_app_table[current_index+1].display );
			} else {
				sc1602_set_buffer       ( 0, fat_app_table[current_index-1].display );
				sc1602_set_buffer_cursol( 1, fat_app_table[current_index  ].display );
			}
			break;
		
		case 0x01:
			switch( fat_app_table[current_index].pExecFunc(uicmd, fat_app_table[current_index].param) ) {
				case UI_RET_QUIT:
					ui_fat_proc = 0x00;
					break;
				default:
					break;
			}
			break;
		default:
			break;
	}
	return ret_val;
}


// -------------------------------------------
//  UI Function - FAT debug
// -------------------------------------------
int ui_function_fat_debug(UI_COMMAND uicmd, int param) {
	static unsigned char current_index;
	static unsigned char ok_press;
	int ret_val;
	
	ret_val = UI_RET_READY;
	
	switch( uicmd.cmd ) {
	case UI_CMD_NOP:
		break;
		
	case UI_CMD_KEY_PRESS_OK:
		if( uicmd.param == OFF_EDGE )	ok_press = 0;
		else							ok_press = 1;
		break;
		
	case UI_CMD_INTEVAL:
		// Title
		sc1602_set_buffer( 0, debug_var_table[current_index].display );
		switch(debug_var_table[current_index].type) {
			case 1:
				sc1602_set_buffer_variable1 ( 1, debug_var_table[current_index].p_variable );
				break;
			case 2:
				sc1602_set_buffer_variable2 ( 1, debug_var_table[current_index].p_variable );
				break;
			case 3:
				sc1602_set_buffer_variable3 ( 1, debug_var_table[current_index].p_variable );
				break;
			case 4:
				sc1602_set_buffer_dump (1, debug_var_table[current_index].p_variable );
				break;
		}
		break;
		
	case UI_CMD_STARTED:
		current_index = 0;
		ok_press = 0;
		break;
		
	case UI_CMD_KEY_PRESS_UP:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( current_index != 0 ) current_index--;
		break;
	case UI_CMD_KEY_PRESS_DOWN:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( ok_press == 0 ) {
			if( debug_var_table[current_index+1].num !=  -1 ) current_index++;
		} else {
			// TODO
		}
		break;
	case UI_CMD_KEY_PRESS_BACK:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		ret_val = UI_RET_QUIT;
		break;
	}
	return ret_val;
}


// -------------------------------------------
//  UI Function - FAT debug
// -------------------------------------------
int ui_function_fat_mbr_bpb(UI_COMMAND uicmd, int param) {
	static unsigned char current_index;
	static unsigned char ok_press;
	int ret_val;
	
	ret_val = UI_RET_READY;
	
	switch( uicmd.cmd ) {
		case UI_CMD_NOP:
			break;
			
		case UI_CMD_KEY_PRESS_OK:
			if( uicmd.param == OFF_EDGE )	ok_press = 0;
			else							ok_press = 1;
			break;
			
		case UI_CMD_INTEVAL:
			// Title
			sc1602_set_buffer( 0, mbr_bpb_var_table[current_index].display );
			switch(mbr_bpb_var_table[current_index].type) {
				case 1:
					sc1602_set_buffer_variable1 ( 1, mbr_bpb_var_table[current_index].p_variable );
					break;
				case 2:
					sc1602_set_buffer_variable2 ( 1, mbr_bpb_var_table[current_index].p_variable );
					break;
				case 3:
					sc1602_set_buffer_variable3 ( 1, mbr_bpb_var_table[current_index].p_variable );
					break;
				case 4:
					sc1602_set_buffer_dump (1, mbr_bpb_var_table[current_index].p_variable );
					break;
			}
			break;
			
		case UI_CMD_STARTED:
			current_index = 0;
			ok_press = 0;
			break;
			
		case UI_CMD_KEY_PRESS_UP:
			if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
			if( current_index != 0 ) current_index--;
			break;
		case UI_CMD_KEY_PRESS_DOWN:
			if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
			if( ok_press == 0 ) {
				if( mbr_bpb_var_table[current_index+1].num !=  -1 ) current_index++;
			} else {
				// TODO
			}
			break;
		case UI_CMD_KEY_PRESS_BACK:
			if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
			ret_val = UI_RET_QUIT;
			break;
	}
	return ret_val;
}

enum {
	DIR_VIEW_MODE,
	FILE_VIEW_MODE,
	READ_FILE_MODE,
};

// -------------------------------------------
//  UI Function - DIR debug
// -------------------------------------------
int ui_function_fat_dir(UI_COMMAND uicmd, int param) {
	int ret_val,i;
	static unsigned char current_index, cursol_position, view_mode;
	char tempbuf[16];
	
	ret_val = UI_RET_READY;
	
	switch( uicmd.cmd ) {
		case UI_CMD_NOP:
			break;
			
		case UI_CMD_STARTED:
			current_index = 0;
			cursol_position = 0;
			view_mode = DIR_VIEW_MODE;
			current_byte_cnt = 0;
			break;
			
		case UI_CMD_KEY_PRESS_BACK:
			if( uicmd.param == OFF_EDGE ) break; // Ignore off edge

			switch ( view_mode ) {
				case FILE_VIEW_MODE:
					view_mode = DIR_VIEW_MODE;
					break;
				case READ_FILE_MODE:
					view_mode = FILE_VIEW_MODE;
					break;
				default:
				case DIR_VIEW_MODE:
					ret_val = UI_RET_QUIT;
					break;
			}
			break;
			
		case UI_CMD_KEY_PRESS_UP:
			if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
			switch ( view_mode ) {
				case FILE_VIEW_MODE:
					while( (p_buffer_addr[current_byte_cnt] != 0x0D) || (p_buffer_addr[current_byte_cnt+1] != 0x0A) ) {
						if( current_byte_cnt == 0 ) break;
						current_byte_cnt--;
					}
					current_byte_cnt--;
					while( (p_buffer_addr[current_byte_cnt] != 0x0D) || (p_buffer_addr[current_byte_cnt+1] != 0x0A) ) {
						if( current_byte_cnt == 0 ) break;
						current_byte_cnt--;
					}
					if( current_byte_cnt != 0 ) current_byte_cnt+=2;
					break;

				case DIR_VIEW_MODE:
					if( cursol_position == 1 ) {
						cursol_position = 0;
						current_index--;
					} else {
						if( current_index != 0 ) current_index--;
					}
					break;
					
				default:
				case READ_FILE_MODE:
					break;
			}
			break;
		case UI_CMD_KEY_PRESS_DOWN:
			if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
			switch ( view_mode ) {
				case FILE_VIEW_MODE:
					while( (p_buffer_addr[current_byte_cnt] != 0x0D) || (p_buffer_addr[current_byte_cnt+1] != 0x0A) ) {
						if( current_byte_cnt >= dir_info[current_index].FileSize ) break;
						current_byte_cnt++;
					}
					if( current_byte_cnt < dir_info[current_index].FileSize ) current_byte_cnt+=2;
					break;
					
				case DIR_VIEW_MODE:
					if( cursol_position == 0 ) {
						if( TotalFileNum != 0) {
							cursol_position = 1;
							current_index++;
						}
					} else {
						if( current_index < TotalFileNum ) current_index++;
					}
					
				default:
				case READ_FILE_MODE:
					break;
			}
			break;
		case UI_CMD_KEY_PRESS_OK:
			if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
			// change dir or file view
			switch ( view_mode ) {
				case FILE_VIEW_MODE:
					//TransferStartRequest(p_buffer_addr, dir_info[current_index].FileSize);
					view_mode = READ_FILE_MODE;
					break;
					
				case DIR_VIEW_MODE:
					if( dir_info[current_index].Attributes & DIR_ATTR_DIRECTORY ) {
						// Change dir
						fat_change_directory(dir_info[current_index].FirstCluster);
						current_index = 0;
					} else {
						// Open File
						fat_read_file(dir_info[current_index].FirstCluster, dir_info[current_index].FileSize );
						view_mode = READ_FILE_MODE;
						current_byte_cnt = 0;
					}
					break;
					
				case READ_FILE_MODE:
#if 1
					if( usbms_status( CLASS_REQ_NONE ) == CLASS_STS_READY ) 
						TransferStartRequest(p_buffer_addr, dir_info[current_index].FileSize);
#endif
					break;
					
				default:
					break;
			} 
			break;
			
		case UI_CMD_INTEVAL:
			switch ( view_mode ) {
				case FILE_VIEW_MODE:
				{
					unsigned char tempbuf[16];
					unsigned long offset;
					
					offset = 0;
					for(i=0; i<16; i++) tempbuf[i]=' ';
					for(i=0; i<16; i++) {
						if( current_byte_cnt+offset >= dir_info[current_index].FileSize ) break;
						if( (p_buffer_addr[current_byte_cnt+offset] == 0x0D) && (p_buffer_addr[current_byte_cnt+offset+1] == 0x0A) ) {
							offset += 2;
							break;
						} else {
							tempbuf[i] =  p_buffer_addr[current_byte_cnt+offset];
							offset++;
						}
					}
					sc1602_set_buffer(0,tempbuf);
					
					// If line length is over 16, need to skip until next CR.
					if(offset == 16) {
						while( (p_buffer_addr[current_byte_cnt+offset] != 0x0D) || (p_buffer_addr[current_byte_cnt+offset+1] != 0x0A) ) {
							offset++;
							if( current_byte_cnt+offset >= dir_info[current_index].FileSize ) break;
						}
					}
					
					for(i=0; i<16; i++) tempbuf[i]='.';
					for(i=0; i<16; i++) {
						if( current_byte_cnt+offset >= dir_info[current_index].FileSize ) break;
						if( (p_buffer_addr[current_byte_cnt+offset] == 0x0D) && (p_buffer_addr[current_byte_cnt+offset+1] == 0x0A) ) {
							offset += 2;
							break;
						} else {
							tempbuf[i] =  p_buffer_addr[current_byte_cnt+offset];
							offset++;
						}
					}
					sc1602_set_buffer(1,tempbuf);
				}
					break;
				case DIR_VIEW_MODE:
					// Directory
					if(TotalFileNum == 0) {
						sc1602_set_buffer(0,"--- NO FILE --- ");
						sc1602_set_buffer(1,"                ");
					} else {
						if( cursol_position == 0) { // cursol in first line
							make_display_text_dirinfo(tempbuf, &dir_info[current_index]);
							sc1602_set_buffer_cursol( 0, tempbuf );
							if( current_index < TotalFileNum) {
								make_display_text_dirinfo(tempbuf, &dir_info[current_index+1]);
								sc1602_set_buffer( 1, tempbuf );
							} else {
								sc1602_set_buffer(1,"                ");
							}
						} else {
							make_display_text_dirinfo(tempbuf, &dir_info[current_index-1]);
							sc1602_set_buffer       ( 0, tempbuf );
							make_display_text_dirinfo(tempbuf, &dir_info[current_index]);
							sc1602_set_buffer_cursol( 1, tempbuf );
						}
					}
					break;
					
				case READ_FILE_MODE:
					make_display_text_dirinfo(tempbuf, &dir_info[current_index]);
					sc1602_set_buffer( 0, tempbuf );
					sc1602_set_buffer_progress_kb(1, ReadFileStatus.ReadSize, ReadFileStatus.FileSize);
				default:
					break;
			}
			break;
	}
	return ret_val;
}
const char hex2_ascii[] = "0123456789ABCDEF";

static void make_display_text_dirinfo( unsigned char* tempbuf, P_DIR_INFO pDirInfo  ) {
	int i; 
	unsigned long Val;
	
	for(i=0;i<16;i++) tempbuf[i]=' ';
	if( pDirInfo->Attributes & DIR_ATTR_DIRECTORY ) {
		tempbuf[ 1]='[';
		tempbuf[14]=']';
	}
#if 0
	for(i=0;i<sizeof(pDirInfo->Name)-1;i++)
		tempbuf[i+2]=pDirInfo->Name[i];
#else
	for(i=0;i<sizeof(pDirInfo->Name)-6;i++)
		tempbuf[i+2]=pDirInfo->Name[i];
	Val = pDirInfo->FileSize / 1024;
	if( !(pDirInfo->Attributes & DIR_ATTR_DIRECTORY) ) {
		tempbuf[10] =  hex2_ascii[(Val /1000) % 10 ];
		tempbuf[11] =  hex2_ascii[(Val /100)  % 10 ];
		tempbuf[12] =  hex2_ascii[(Val /10)   % 10 ];
		tempbuf[13] =  hex2_ascii[(Val )      % 10 ];
		tempbuf[14]='K';
		tempbuf[15]='B';
	}
#endif

}



// -------------------------------------------
//  UI Sub Function - USB nop
// -------------------------------------------
int ui_function_fat_nop(UI_COMMAND uicmd, int param)
{
	return UI_RET_QUIT;
}

