﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using System.Diagnostics;
using Lugens.Utils;

namespace Lugens.Components
{
    /// <summary>
    /// ホットキーは2バイトで構成され、
    /// 上位1バイトにコントロールキー情報、下位1バイトにキーコードが入る
    /// RWin(右ウィンドウズキー)は考慮されていない
    /// 
    /// 0xC000...Win(LWin)
    /// 0x2000...LControll
    /// 0x1000...RControll
    /// 0x0800...LAlt(LMenu)
    /// 0x0400...RAlt(RMenu)
    /// 0x0200...LShift
    /// 0x0100...RShift
    /// 
    /// </summary>
    public partial class HotKeyTextBox : TextBox
    {

        [DllImport("user32.dll")]
        public static extern int GetKeyState(int nVirtKey);

        private static int[] KeyState = new int[256];

        //private int lastHotKeycode;
        protected List<int> acceptKeys = null;
        public List<int> AcceptKeys
        {
            get
            {
                return this.acceptKeys;
            }

            set
            {
                this.acceptKeys = value;
            }
        }

        public HotKeyTextBox()
        {
            this.ImeMode = ImeMode.Disable;
            this.TextAlign = System.Windows.Forms.HorizontalAlignment.Center;
            this.ContextMenu = new ContextMenu();
            this.GotFocus += this.OnGotFocus;
        }

        private int hotKeycode;
        public int HotKeycode
        {
            get { return hotKeycode; }
            set {
                hotKeycode = value;
                this.Text = GetKeyText();
            }
        }

        private string defaultText = "";
        public string DefaultText
        {
            get { return defaultText; }
            set { defaultText = value; }
        }

        public static bool IsControlKey(int keycode)
        {
            switch (keycode)
            {
                case (int)Keycode.Win:
                case (int)Keycode.LControl:
                case (int)Keycode.RControl:
                case (int)Keycode.LAlt:
                case (int)Keycode.RAlt:
                case (int)Keycode.LShift:
                case (int)Keycode.RShift:
                    return true;
            }
            return false;
        }

        public static bool IsNonDisplayKeycode(int keycode)
        {
            if (keycode < 47)
                return true;
            if(keycode >=112 && keycode <= 123) //F1-F12
                return true;
            if (keycode == 144) //NumLock
                return true;
            if (keycode == 145) //ScrollLock
                return true;
            return false;
        }


        public static int GetControlKeycode(int keycode)
        {
            switch (keycode)
            {
                case (int)Keycode.Win:
                    return 0xC000;
                case (int)Keycode.LControl:
                    return 0x2000;
                case (int)Keycode.RControl:
                    return 0x1000;
                case (int)Keycode.LAlt:
                    return 0x0800;
                case (int)Keycode.RAlt:
                    return 0x0400;
                case (int)Keycode.LShift:
                    return 0x0200;
                case (int)Keycode.RShift:
                    return 0x0100;
            }
            return 0;
        }

        public static string GetControlKeyText(int keycode)
        {
            switch (keycode)
            {
                case (int)Keycode.Win:
                    return "Win";
                case (int)Keycode.LControl:
                    return "LControl";
                case (int)Keycode.RControl:
                    return "RControl";
                case (int)Keycode.LAlt:
                    return "LAlt";
                case (int)Keycode.RAlt:
                    return "RAlt";
                case (int)Keycode.LShift:
                    return "LShift";
                case (int)Keycode.RShift:
                    return "RShift";
            }
            return "";
        }

        public static string GetKeyText(int hotKeyCode)
        {
            if (hotKeyCode == 0)
                return "";

            StringBuilder sb = new StringBuilder();

            if ((hotKeyCode & 0xC000) == 0xC000)
                sb.Append(Keycode.Win.ToString());

            if ((hotKeyCode & 0x2000) == 0x2000)
                Append(ref sb, Keycode.LControl.ToString());
            if ((hotKeyCode & 0x1000) == 0x1000)
                Append(ref sb, Keycode.RControl.ToString());

            if ((hotKeyCode & 0x0800) == 0x0800)
                Append(ref sb, Keycode.LAlt.ToString());
            if ((hotKeyCode & 0x0400) == 0x0400)
                Append(ref sb, Keycode.RAlt.ToString());

            if ((hotKeyCode & 0x0200) == 0x0200)
                Append(ref sb, Keycode.LShift.ToString());
            if ((hotKeyCode & 0x0100) == 0x0100)
                Append(ref sb, Keycode.RShift.ToString());

            char c = (char)(hotKeyCode & 0xFF);
            if (c != 0)
                Append(ref sb, KeycodeText.KeyText[c]);
            return sb.ToString();
        }

        private string GetKeyText()
        {
            if (hotKeycode == 0)
                return this.defaultText;

            StringBuilder sb = new StringBuilder();

            if ((this.hotKeycode & 0xC000) == 0xC000)
                sb.Append(Keycode.Win.ToString());

            if ((this.hotKeycode & 0x2000) == 0x2000)
                Append(ref sb, Keycode.LControl.ToString());
            if ((this.hotKeycode & 0x1000) == 0x1000)
                Append(ref sb, Keycode.RControl.ToString());

            if ((this.hotKeycode & 0x0800) == 0x0800)
                Append(ref sb, Keycode.LAlt.ToString());
            if ((this.hotKeycode & 0x0400) == 0x0400)
                Append(ref sb, Keycode.RAlt.ToString());

            if ((this.hotKeycode & 0x0200) == 0x0200)
                Append(ref sb, Keycode.LShift.ToString());
            if ((this.hotKeycode & 0x0100) == 0x0100)
                Append(ref sb, Keycode.RShift.ToString());

            char c = (char)(this.hotKeycode & 0xFF);
            if (c != 0)
                Append(ref sb, KeycodeText.KeyText[c]);
            return sb.ToString();
        }

        private static void Append(ref StringBuilder sb, string str)
        {
            if (sb.Length > 0)
                sb.Append(" + ");
            sb.Append(str);
        }

        protected override void OnKeyPress(KeyPressEventArgs e)
        {
            e.Handled = true;
        }

        protected override void OnKeyDown(KeyEventArgs e)
        {
            e.Handled = true;
        }

        protected override void OnTextChanged(EventArgs e)
        {
            base.OnTextChanged(e);
            this.SelectionStart = this.Text.Length;
        }
/*
        protected override void OnPreviewKeyDown(PreviewKeyDownEventArgs e)
        {
            switch (e.KeyCode)
            {
                case Keys.Tab:
                    e.IsInputKey = true;
                    break;
            }
            base.OnPreviewKeyDown(e);
        }
*/

        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {

                case 0x0101: //WM_KEYUP
                case 0x0105: //WM_SYSKEYUP
                    //KeyState[(int)m.WParam] = 0;
                    if ((this.hotKeycode & 0xFF) == 0)
                    {
                        this.hotKeycode = 0;
                        this.Text = this.defaultText;
                    }
                    break;

                case 0x0100: //WM_KEYDOWN
                case 0x0104: //WM_SYSKEYDOWN
                    //KeyState[(int)m.WParam] = 1;
                    this.hotKeycode = 0;
                    if (Win32.GetKeyState((int)Keycode.Win) < 0)
                        this.hotKeycode |= 0xC000;
                    if (Win32.GetKeyState((int)Keycode.LControl) < 0)
                        this.hotKeycode |= 0x2000;
                    if (Win32.GetKeyState((int)Keycode.RControl) < 0)
                        this.hotKeycode |= 0x1000;
                    if (Win32.GetKeyState((int)Keycode.LAlt) < 0)
                        this.hotKeycode |= 0x0800;
                    if (Win32.GetKeyState((int)Keycode.RAlt) < 0)
                        this.hotKeycode |= 0x0400;
                    if (Win32.GetKeyState((int)Keycode.LShift) < 0)
                        this.hotKeycode |= 0x0200;
                    if (Win32.GetKeyState((int)Keycode.RShift) < 0)
                        this.hotKeycode |= 0x0100;

                    if (this.acceptKeys == null || !this.acceptKeys.Contains((int)m.WParam))
                    {
                        this.Text = this.defaultText;
                        break;
                    }

                    switch ((int)m.WParam)
                    {
                        case (int)Keycode.Shift:
                        case (int)Keycode.Control:
                        case (int)Keycode.Alt:
                        case (int)Keycode.Win:
                            break;

                        default:
                            this.hotKeycode |= (char)m.WParam;
                            break;
                    }
                    if ((this.hotKeycode & 0xFF00) == 0)
                        this.hotKeycode |= 0x2000;
                    this.Text = GetKeyText();

                    return;
            }
            base.WndProc(ref m);
        }

        private void OnGotFocus(object sender, EventArgs e)
        {
            Array.Clear(KeyState, 0, 256);
        }
    
    }

}

