﻿using System;
using System.Collections.Generic;
using System.Text;
using Lugens.Utils;
using System.Drawing;
using System.Windows.Forms;
using System.Diagnostics;
using Lugens.Components;
using System.Runtime.InteropServices;
using System.Threading;
using Lugens.Passer.Macro;

namespace Lugens.Passer
{
    public enum HotKeyProcessType
    {
        /// <summary>
        /// メインフォームの表示
        /// </summary>
        OpenMainForm,

        /// <summary>
        /// 定型文フォームの表示
        /// </summary>
        OpenSentenceForm,

        /// <summary>
        /// 定型文のペースト
        /// </summary>
        PasteSentence,

        /// <summary>
        /// クリップボード文字列のペースト
        /// </summary>
        PasteClipboard,

        /// <summary>
        /// スクリプトの実行
        /// </summary>
        ExecScript,

        /// <summary>
        /// マクロの実行
        /// </summary>
        ExecMacro,

        /// <summary>
        /// キー操作記憶
        /// </summary>
        KeyRecord,

        /// <summary>
        /// 記憶したキー再生
        /// </summary>
        KeyPlay

    }

    public class HotKeyProcess
    {
        public HotKeyProcessType ProcessType;
        public int HotKey;
        public string Id;
        public string Value;
        public SentenceInfo SentenceInfo;
    }

    public static class HotKeyProcessExecuter
    {
        public static volatile HotKeyProcess Process;

        private static string MacroResult;
        
        private delegate void HotKeyProcessDelegate();

        public static void Execute(List<HotKeyProcess> processList)
        {
            HotKeyProcessExecuter.Process = processList[0];
            if (processList.Count == 1)
            {
                switch (HotKeyProcessExecuter.Process.ProcessType)
                {
                    case HotKeyProcessType.OpenMainForm:
                        OpenMainForm();
                        return;

                    case HotKeyProcessType.OpenSentenceForm:
                        OpenSentenceForm();
                        return;

                    case HotKeyProcessType.PasteSentence:
                        if (Program.Status == ProgramStatus.Waiting)
                        {
                            Program.Status = ProgramStatus.SendMessaging;
                            new Thread(new ThreadStart(InvokePasteSentence)).Start();
                        }
                        return;

                    case HotKeyProcessType.PasteClipboard:
                        if (Program.SentenceGroupList[0].Count > 0)
                        {
                            if (Program.Status == ProgramStatus.Waiting && !"".Equals(Program.SentenceGroupList[0][0].Value))
                            {
                                Program.Status = ProgramStatus.SendMessaging;
                                new Thread(new ThreadStart(InvokePasteClipboard)).Start();
                            }
                        }
                        return;

                    case HotKeyProcessType.ExecMacro:
                        if (Program.Status == ProgramStatus.Waiting)
                        {
                            if (Program.MacroDic.ContainsKey(HotKeyProcessExecuter.Process.Id))
                            {
                                Program.Status = ProgramStatus.SendMessaging;
                                new Thread(new ThreadStart(InvokeExecMacro)).Start();
                            }
                            else
                            {
                                string text = "マクロが見つかりません\nID:" + HotKeyProcessExecuter.Process.Id;
                                MessageBox.Show(Program.MainForm, text, "マクロ実行エラー", MessageBoxButtons.OK, MessageBoxIcon.Error);
                            }
                        }
                        return;

                    case HotKeyProcessType.ExecScript:
                        if (Program.Status == ProgramStatus.Waiting)
                        {
                            Program.Status = ProgramStatus.SendMessaging;
                            new Thread(new ThreadStart(InvokeExecScript)).Start();
                        }
                        return;

                    case HotKeyProcessType.KeyRecord:
                        KeyRecord();
                        return;

                    case HotKeyProcessType.KeyPlay:
                        if (Program.Status == ProgramStatus.Waiting)
                        {
                            Program.Status = ProgramStatus.SendMessaging;
                            new Thread(new ThreadStart(KeyPlay)).Start();
                        }
                        return;
                }
            }
            else
            {
                switch (HotKeyProcessExecuter.Process.ProcessType)
                {
                    case HotKeyProcessType.PasteSentence:
                    case HotKeyProcessType.ExecScript:
                    case HotKeyProcessType.ExecMacro:
                        OpenSentenceForm(processList);
                        return;
                }
            }
        }

        /// <summary>
        /// メインフォームの表示
        /// </summary>
        public static void OpenMainForm()
        {
            switch(Program.MainForm.Status)
            {
                case MainForm.MainFormStatus.Active:
                    Program.MainForm.ShowEngineList();
                    return;

                case MainForm.MainFormStatus.Hide:
                    Program.MainForm.BeforeHWnd = Win32.GetForegroundWindow();
                    Program.MainForm.DoVisibleChange(true);
                    break;
            }

            Program.MainForm.Status = MainForm.MainFormStatus.Active;
            Program.ForceForegroundWindow(Program.MainForm.Handle);
            Program.MainForm.Invalidate();
            Program.MainForm.SearchTextBox.Focus();
            Program.MainForm.Timer.Start();
        }

        /// <summary>
        /// 定型文フォームの表示
        /// </summary>
        public static void OpenSentenceForm()
        {
            if (Program.Status == ProgramStatus.Waiting)
            {
                Rectangle r = Program.GetSentenceFormBaseRectangle();
                Program.SentenceForm = new SentenceForm();

                Program.SentenceForm.FixData = false;
                Program.SentenceForm.SetData();
                Program.SentenceForm.DoVisibleChange(true);
                if (Settings.ContainsKey(Settings.PASSER_FORM_SENTENCE_WIDTH) && Settings.ContainsKey(Settings.PASSER_FORM_SENTENCE_HEIGHT))
                    Program.SentenceForm.Size = new Size(Settings.GetInt(Settings.PASSER_FORM_SENTENCE_WIDTH), Settings.GetInt(Settings.PASSER_FORM_SENTENCE_HEIGHT));
                Program.SentenceForm.Location = Util.CalcWindowLocate(r, Program.SentenceForm.Size, Util.Locate.Down);
                Program.SentenceForm.ShowToolTip();
                Program.Status = ProgramStatus.SentenceFormOpen;
            }
        }

        /// <summary>
        /// 定型文フォームの表示
        /// </summary>
        public static void OpenSentenceForm(List<HotKeyProcess> processList)
        {
            if (Program.Status == ProgramStatus.Waiting)
            {
                Rectangle r = Program.GetSentenceFormBaseRectangle();
                Program.SentenceForm = new SentenceForm();
                Program.SentenceForm.FixData = true;
                Program.SentenceForm.SetData(processList);
                Program.SentenceForm.DoVisibleChange(true);
                if (Settings.ContainsKey(Settings.PASSER_FORM_SENTENCE_WIDTH) && Settings.ContainsKey(Settings.PASSER_FORM_SENTENCE_HEIGHT))
                    Program.SentenceForm.Size = new Size(Settings.GetInt(Settings.PASSER_FORM_SENTENCE_WIDTH), Settings.GetInt(Settings.PASSER_FORM_SENTENCE_HEIGHT));
                Program.SentenceForm.Location = Util.CalcWindowLocate(r, Program.SentenceForm.Size, Util.Locate.Down);
                Program.SentenceForm.ShowToolTip();
                Program.Status = ProgramStatus.SentenceFormOpen;
            }
        }

        public static void InvokePasteSentence()
        {
            Program.MainForm.Invoke(new HotKeyProcessDelegate(PasteSentence));
        }

        public static void InvokePasteClipboard()
        {
            Program.MainForm.Invoke(new HotKeyProcessDelegate(PasteClipboard));
        }

        public static void InvokeExecMacro()
        {
            Program.MainForm.Invoke(new HotKeyProcessDelegate(ExecMacro));
        }

        public static void InvokeExecScript()
        {
            Program.MainForm.Invoke(new HotKeyProcessDelegate(ExecScript));
        }

        /// <summary>
        /// 定型文のペースト
        /// </summary>
        static void PasteSentence()
        {
            Clipboard.SetText(HotKeyProcessExecuter.Process.Value);
            Program.SendMessage("${LControl+}${V}${LControl-}", HotKeyProcessExecuter.Process.HotKey);
            Program.Status = ProgramStatus.Waiting;
        }

        /// <summary>
        /// クリップボード文字列のペースト
        /// </summary>
        static void PasteClipboard()
        {
            Clipboard.SetText(Program.SentenceGroupList[0][0].Value);
            Program.SendMessage("${LControl+}${V}${LControl-}", HotKeyProcessExecuter.Process.HotKey);
            Program.Status = ProgramStatus.Waiting;
        }
        
        /// <summary>
        /// マクロの実行
        /// </summary>
        static void ExecMacro()
        {
            IMacro macro = Program.MacroDic[HotKeyProcessExecuter.Process.Id].Macro;
            MacroExecuter.ExecuteAsync(HotKeyProcessExecuter.Process.HotKey, macro, Util.Split(HotKeyProcessExecuter.Process.Value, ' ', '"'));
        }
        

        /// <summary>
        /// スクリプトの実行
        /// </summary>
        public static void ExecScript()
        {
            Program.SendMessage(HotKeyProcessExecuter.Process.Value, HotKeyProcessExecuter.Process.HotKey);
            Program.Status = ProgramStatus.Waiting;
        }

        /// <summary>
        /// キーボード記憶
        /// </summary>
        public static void KeyRecord()
        {
            if (Program.Status == ProgramStatus.KeyRecoding)
            {
                //ストップ
                Program.Status = ProgramStatus.Waiting;
                Win32.PostMessage(Program.MainForm.Handle, Program.WM_APP_REC_END, (IntPtr)HotKeyProcessExecuter.Process.HotKey, IntPtr.Zero);
                Program.MainForm.notifyIcon1.Icon = Properties.Resources.App;
                if (Program.Main_ShowBalloonWhileRecoding)
                {
                    Win32.PostMessage(Program.BallonWindowHandle, Win32.WM_LBUTTONDOWN, (IntPtr)0, (IntPtr)0);
                    Program.BallonWindowHandle = IntPtr.Zero;
                    Program.MainForm.timer_keyRecoding.Enabled = false;
                }
            }
            else
            {
                //記録
                Program.KeyRecordList.Clear();
                Program.Status = ProgramStatus.KeyRecoding;
                NotifyIcon icon = Program.MainForm.notifyIcon1;
                icon.Icon = Properties.Resources.AppRec;
                if (Program.Main_ShowBalloonWhileRecoding)
                {
                    icon.BalloonTipText = "停止キー: " + HotKeyTextBox.GetKeyText(HotKeyProcessExecuter.Process.HotKey);
                    icon.ShowBalloonTip(Int32.MaxValue, icon.BalloonTipTitle, icon.BalloonTipText, ToolTipIcon.Info);
                    Win32.EnumWindows(Program.MainForm.EnumWindowsProc, IntPtr.Zero);
                    Program.MainForm.timer_keyRecoding.Enabled = true;
                }

            }
        }


        /// <summary>
        /// キーボード再生
        /// </summary>
        public static void KeyPlay()
        {
            Program.SendMessage(Program.KeyRecordString, HotKeyProcessExecuter.Process.HotKey);
            Program.Status = ProgramStatus.Waiting;
        }

        public static void OnMacroCompleted(string s)
        {
            HotKeyProcessExecuter.MacroResult = s;
        }
    }
}
