﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;
using Lugens.Components;

namespace Lugens.Passer
{
    public class KeyRecordParser
    {
        private int HotKey;
        private List<KeyRecordInfo> RecordList;

        public string ErrorString;

        public string Parse(List<KeyRecordInfo> recordList, int hotKey)
        {
            StringBuilder sb = new StringBuilder();
            this.HotKey = hotKey;
            this.RecordList = recordList;

            Parse_1();
            Parse_2();
            Parse_3(hotKey);
            Parse_4();
            Parse_5();
            Parse_6();

            foreach (KeyRecordInfo info in this.RecordList)
                if (info.Str != null && info.Str.Length > 0)
                    sb.Append(info.Str);

            return sb.ToString();
        }

        /// <summary>
        /// IMEの文字列があったら前後の余分なレコードを削除する
        /// 半角/全角のKEYDOWNレコードを削除する(229,243,244)
        /// </summary>
        /// <returns></returns>
        private bool Parse_1()
        {
            int i = 0;
            bool ime_in = false;
            KeyRecordInfo info;

            while (this.RecordList.Count > i)
            {
                info = this.RecordList[i];
                switch (info.Message)
                {
                    case Program.WM_APP_IME_START:
                        ime_in = true;
                        this.RecordList.RemoveAt(i);
                        continue;

                    case Program.WM_APP_IME_END:
                        ime_in = false;
                        this.RecordList.RemoveAt(i);
                        if (this.RecordList.Count > i)
                        {
                            info = this.RecordList[i];
                            if(info.Message == Program.WM_APP_KEYUP && info.Keycode == (int)Keycode.Enter)
                                this.RecordList.RemoveAt(i);
                        }
                        continue;

                    default:
                        if (ime_in && info.Message != Program.WM_APP_IME_COMP)
                        {
                            this.RecordList.RemoveAt(i);
                            continue;
                        }
                        else if (info.Message == Program.WM_APP_KEYDOWN && info.Keycode == 229)
                        {
                            this.RecordList.RemoveAt(i);
                            continue;
                        }
                        else if (info.Message == Program.WM_APP_KEYUP)
                        {
                            if (info.Keycode == 243 || info.Keycode == 244)
                            {
                                this.RecordList.RemoveAt(i);
                                continue;
                            }
                        }
                        break;
                }
                i++;
            }
            return !ime_in;
        }

        /// <summary>
        /// コントロールキーを左右区別したコードに変換する
        /// </summary>
        private bool Parse_2()
        {
            foreach (KeyRecordInfo info in this.RecordList)
            {
                if (info.Message == Program.WM_APP_KEYDOWN || info.Message == Program.WM_APP_KEYUP)
                {
                    switch (info.Keycode)
                    {
                        case (int)Keycode.Shift:
                            if ((info.VkCode & 0x2A0001) == 0x2A0001)
                                info.Keycode = 160;
                            else if ((info.VkCode & 0x360001) == 0x360001)
                                info.Keycode = 161;
                            else
                            {
                                this.ErrorString = "Shiftキーの値が異常(key:" + info.Keycode + ", vk=" + info.VkCode.ToString("X");
                                return false;
                            }
                            break;

                        case (int)Keycode.Control:
                            if ((info.VkCode & 0x11D0001) == 0x11D0001)
                                info.Keycode = 163;
                            else if ((info.VkCode & 0x1D0001) == 0x1D0001)
                                info.Keycode = 162;
                            else
                            {
                                this.ErrorString = "Controlキーの値が異常(key:" + info.Keycode + ", vk=" + info.VkCode.ToString("X");
                                return false;
                            }
                            break;

                        case (int)Keycode.Alt:
                            if ((info.VkCode & 0x1380001) == 0x1380001)
                                info.Keycode = 165;
                            else if ((info.VkCode & 0x380001) == 0x380001)
                                info.Keycode = 164;
                            else
                            {
                                this.ErrorString = "Altキーの値が異常(key:" + info.Keycode + ", vk=" + info.VkCode.ToString("X");
                                return false;
                            }
                            break;
                    }

                }

            }
            return true;
        }

        /// <summary>
        /// 開始、終了時のホットキー制御
        /// </summary>
        private bool Parse_3(int hotKey)
        {
            int keycode = 0;
            KeyRecordInfo info;
            hotKey &= 0xFF00;
            
            //開始時のホットキー入力削除
            while (this.RecordList.Count > 0 && hotKey != 0)
            {
                info = this.RecordList[0];
                if (info.Message == Program.WM_APP_KEYUP)
                {
                    switch (info.Keycode)
                    {
                        case (int)Keycode.Win:
                            hotKey &= 0x3FFF;
                            break;
                        case (int)Keycode.LControl:
                        case (int)Keycode.RControl:
                            hotKey &= 0xCFFF;
                            break;
                        case (int)Keycode.LAlt:
                        case (int)Keycode.RAlt:
                            hotKey &= 0xF3FF;
                            break;
                        case (int)Keycode.LShift:
                        case (int)Keycode.RShift:
                            hotKey &= 0xFCFF;
                            break;
                    }
                }
                this.RecordList.RemoveAt(0);
            }

            //終了時のホットキー入力削除
            while (this.RecordList.Count > 0)
            {
                info = this.RecordList[this.RecordList.Count-1];
                if (info.Message == Program.WM_APP_KEYDOWN)
                {
                    if (HotKeyTextBox.IsControlKey(info.Keycode))
                    {
                        this.RecordList.RemoveAt(this.RecordList.Count - 1);
                        continue;
                    }

                }
                break;
            }

            //Shiftキーが押されてるときにもう片方のShiftキーを押した場合、
            //片方のKEYUPレコードしか飛んでこない為、削除する
            hotKey = 0;
            for (int i = 0; i < this.RecordList.Count; i++)
            {
                info = this.RecordList[i];
                if (info.Message == Program.WM_APP_KEYDOWN)
                {
                    if (info.Keycode == (int)Keycode.LShift || info.Keycode == (int)Keycode.RShift)
                    {
                        if (hotKey > 0)
                        {
                            this.RecordList.RemoveAt(i);
                            i--;
                            continue;
                        }
                        hotKey |= HotKeyTextBox.GetControlKeycode(info.Keycode);
                        keycode = info.Keycode;
                    }
                }
                else if (info.Message == Program.WM_APP_KEYUP)
                {
                    if (info.Keycode == (int)Keycode.LShift || info.Keycode == (int)Keycode.RShift)
                    {
                        info.Keycode = keycode;
                        keycode = 0;
                        hotKey = 0;
                    }
                }
            }

            //ホットキーが入力されたまま終了した場合、KEYUPを加える
            hotKey = 0;
            for(int i = 0; i < this.RecordList.Count; i++)
            {
                info = this.RecordList[i];
                if (HotKeyTextBox.IsControlKey(info.Keycode))
                {
                    if (info.Message == Program.WM_APP_KEYDOWN)
                        hotKey |= HotKeyTextBox.GetControlKeycode(info.Keycode);
                    else if (info.Message == Program.WM_APP_KEYUP)
                        hotKey &= (~HotKeyTextBox.GetControlKeycode(info.Keycode) & 0xFFFF);
                }
            }

            if (hotKey > 0)
            {
                if ((hotKey & 0xC000) == 0xC000)
                {
                    info = new KeyRecordInfo();
                    info.Message = Program.WM_APP_KEYUP;
                    info.Keycode = (int)Keycode.Win;
                    this.RecordList.Add(info);
                }
                if ((hotKey & 0x2000) == 0x2000)
                {
                    info = new KeyRecordInfo();
                    info.Message = Program.WM_APP_KEYUP;
                    info.Keycode = (int)Keycode.LControl;
                    this.RecordList.Add(info);
                }
                if ((hotKey & 0x1000) == 0x1000)
                {
                    info = new KeyRecordInfo();
                    info.Message = Program.WM_APP_KEYUP;
                    info.Keycode = (int)Keycode.RControl;
                    this.RecordList.Add(info);
                }
                if ((hotKey & 0x0800) == 0x0800)
                {
                    info = new KeyRecordInfo();
                    info.Message = Program.WM_APP_KEYUP;
                    info.Keycode = (int)Keycode.LAlt;
                    this.RecordList.Add(info);
                }
                if ((hotKey & 0x0400) == 0x0400)
                {
                    info = new KeyRecordInfo();
                    info.Message = Program.WM_APP_KEYUP;
                    info.Keycode = (int)Keycode.RAlt;
                    this.RecordList.Add(info);
                }
                if ((hotKey & 0x0200) == 0x0200)
                {
                    info = new KeyRecordInfo();
                    info.Message = Program.WM_APP_KEYUP;
                    info.Keycode = (int)Keycode.LShift;
                    this.RecordList.Add(info);
                }
                if ((hotKey & 0x0100) == 0x0100)
                {
                    info = new KeyRecordInfo();
                    info.Message = Program.WM_APP_KEYUP;
                    info.Keycode = (int)Keycode.RShift;
                    this.RecordList.Add(info);
                }
            }

            return this.RecordList.Count > 0;
        }


        /// <summary>
        /// 重複した制御キーのKEYDOWN,KEYUPレコードを削除する
        /// </summary>
        /// <returns></returns>
        private bool Parse_4()
        {
            int i = 0;
            int hotKey = 0;
            KeyRecordInfo info;

            while (this.RecordList.Count > i)
            {
                info = this.RecordList[i];
                if (info.Message == Program.WM_APP_KEYUP && HotKeyTextBox.IsControlKey(info.Keycode))
                    hotKey &= (~HotKeyTextBox.GetControlKeycode(info.Keycode) & 0xFFFF);
                else if (info.Message == Program.WM_APP_KEYDOWN)
                {
                    if ((hotKey & HotKeyTextBox.GetControlKeycode(info.Keycode)) != 0)
                    {
                        this.RecordList.RemoveAt(i);
                        continue;
                    }
                    else
                        hotKey |= HotKeyTextBox.GetControlKeycode(info.Keycode);
                }
                i++;
            }

            i = 0;
            hotKey = 0;
            while (this.RecordList.Count > i)
            {
                info = this.RecordList[i];
                if (info.Message == Program.WM_APP_KEYUP)
                {
                    if (info.Keycode == hotKey)
                    {
                        this.RecordList.RemoveAt(i);
                        continue;
                    }
                    hotKey = info.Keycode;
                }
                else
                    hotKey = 0;
                i++;
            }

            return true;
        }

        /// <summary>
        /// 出力文字列を付与する
        /// 
        /// VkCode
        /// 0...通常キー
        /// 1...キーコード(Tab,Enter,etc)
        /// 2...コントロールキーを押した(Alt,Win)
        /// 3...コントロールキーを離した(Alt,Win)
        /// 8...コントロールキーを押した(Ctrl,Shift)
        /// 9...コントロールキーを離した(Ctrl,Shift)
        /// </summary>
        /// <returns></returns>
        private void Parse_5()
        {
            int i = 0;
            int hotKey = 0;
            KeyRecordInfo info;
            KeyRecordInfo infoNext;
            while (this.RecordList.Count > i)
            {
                info = this.RecordList[i];
                switch (info.Message)
                {
                    case Program.WM_APP_IME_COMP:
                        i++;
                        continue;

                    case Program.WM_APP_KEYDOWN:

                        //押されたキーがコントロールキー
                        if (HotKeyTextBox.IsControlKey(info.Keycode))
                        {
                            switch (info.Keycode)
                            {
                                case (int)Keycode.LAlt:
                                case (int)Keycode.RAlt:
                                case (int)Keycode.Win:
                                    info.VkCode = 2;
                                    break;
                                default:
                                    info.VkCode = 8;
                                    break;
                            }
                            info.Str = "${" + HotKeyTextBox.GetControlKeyText(info.Keycode) + "+}";
                            hotKey |= HotKeyTextBox.GetControlKeycode(info.Keycode);
                            i++;
                            continue;
                        }
                        else
                        {
                            //通常キーなので、次のCHARレコードを見る
                            if (this.RecordList.Count > i + 1)
                            {
                                infoNext = this.RecordList[i + 1];
                                if (infoNext.Message == Program.WM_APP_CHAR)
                                {
                                    if (info.Keycode == (int)Keycode.Tab && infoNext.Keycode == (int)Keycode.Tab)
                                    {
                                        //Ctrl+IでもTabになるので、KEYDOWNも見る
                                        infoNext.VkCode = 1;
                                        infoNext.Str = "${Tab}";
                                        this.RecordList.RemoveAt(i);
                                        i++;
                                        continue;
                                    }
                                    if (info.Keycode == (int)Keycode.Enter && infoNext.Keycode == (int)Keycode.Enter)
                                    {
                                        //Ctrl+MでもEnterになるので、KEYDOWNも見る
                                        infoNext.VkCode = 1;
                                        infoNext.Str = "${Enter}";
                                        this.RecordList.RemoveAt(i);
                                        i++;
                                        continue;
                                    }
                                    if (info.Keycode == (int)Keycode.BackSpace && infoNext.Keycode == (int)Keycode.BackSpace)
                                    {
                                        //Ctrl+HでもBackSpaceになるので、KEYDOWNも見る
                                        infoNext.VkCode = 1;
                                        infoNext.Str = "${BackSpace}";
                                        this.RecordList.RemoveAt(i);
                                        i++;
                                        continue;
                                    }
                                    if (info.Keycode == (int)Keycode.Space)
                                    {
                                        //Spaceもキーコードを送る
                                        infoNext.VkCode = 1;
                                        infoNext.Str = "${Space}";
                                        this.RecordList.RemoveAt(i);
                                        i++;
                                        continue;
                                    }
                                    //A-Z(I,Mを除く)
                                    if (infoNext.Keycode >= 1 && infoNext.Keycode <= 26)
                                    {
                                        infoNext.VkCode = 1;
                                        infoNext.Str = "${" + (char)((int)'A' + infoNext.Keycode - 1) + "}";
                                        this.RecordList.RemoveAt(i);
                                        i++;
                                        continue;
                                    }
                                    //タブとエンターはキーコードを送るので無視
                                    //$の場合はエスケープして、他はそのまま送信
                                    //if (infoNext.Keycode != (int)Keycode.Tab && infoNext.Keycode != (int)Keycode.Enter)
                                    //{
                                        if (infoNext.Keycode == (int)'$')
                                            infoNext.Str = "$$";
                                        else
                                            infoNext.Str = ((char)infoNext.Keycode).ToString();
                                        infoNext.VkCode = 0;
                                        this.RecordList.RemoveAt(i);
                                        i++;
                                        continue;
                                    //}
                                }
                                else if (infoNext.Message == Program.WM_APP_KEYUP)
                                {
                                    if (info.Keycode == infoNext.Keycode)
                                    {
                                        info.VkCode = 1;
                                        info.Str = "${" + (Keycode)info.Keycode + "}";
                                        i += 2;
                                        continue;
                                    }
                                }


                                //TAB,ENTERを含む表示されないキーの場合(INSERT,DELETE等)
                                /*
                                if (HotKeyTextBox.IsNonDisplayKeycode(info.Keycode))
                                {
                                    info.VkCode = 1;
                                    info.Str = "${" + (Keycode)info.Keycode + "}";
                                    i++;
                                    continue;
                                }
                                */
                            }
                        }
                        break;

                    case Program.WM_APP_KEYUP:
                        //離されたキーがコントロールキー
                        if (HotKeyTextBox.IsControlKey(info.Keycode))
                        {
                            switch (info.Keycode)
                            {
                                case (int)Keycode.LAlt:
                                case (int)Keycode.RAlt:
                                case (int)Keycode.Win:
                                    info.VkCode = 3;
                                    break;
                                default:
                                    info.VkCode = 9;
                                    break;
                            }
                            info.Str = "${" + HotKeyTextBox.GetControlKeyText(info.Keycode) + "-}";
                            hotKey &= (~HotKeyTextBox.GetControlKeycode(info.Keycode) & 0xFFFF);
                            i++;
                            continue;
                        }
                        if ((hotKey & 0x0C00) != 0)
                        {
                            info.VkCode = 1;
                            info.Str = "${" + (Keycode)info.Keycode + "}";
                            i++;
                            continue;
                        }
                        break;
                }
                this.RecordList.RemoveAt(i);
            }
        }

        /// <summary>
        /// 不要な制御キーを削除する
        /// ${RShift+}ab${RControl+}${RControl-}${RShift-}
        /// ↓
        /// ${RShift+}ab${RShift-}
        /// </summary>
        private void Parse_6()
        {
            int i = 0;
            while (this.RecordList.Count > i)
            {
                if (this.RecordList[i].VkCode == 8)
                {
                    i = Parse_6_1(i);
                    continue;
                }
                i++;
            }
            Parse_6_2();
        }

        private int Parse_6_1(int si)
        {
            int i = si + 1;
            int keycode = this.RecordList[si].Keycode;
            KeyRecordInfo info;
            bool remove = true;
            while (this.RecordList.Count > i)
            {
                info = this.RecordList[i];
                if (info.VkCode == 8)
                {
                    i = Parse_6_1(i);
                    continue;
                }
                if (info.Keycode == keycode && info.VkCode == 9)
                    break;

                if (info.VkCode != 0)
                    remove = false;
                i++;

            }
            if (!remove)
                i++;
            else
            {
                if (this.RecordList.Count > i)
                {
                    this.RecordList.RemoveAt(i);
                    this.RecordList.RemoveAt(si);
                    i -= 2;
                }
            }
            if (i < si)
                return si;
            return i;

        }

        /// <summary>
        /// ${LAlt+}${LAlt-}などの情報を${LAlt}にまとめる
        /// </summary>
        private void Parse_6_2()
        {
            int i = 0;
            KeyRecordInfo info;
            KeyRecordInfo infoNext;
            while (this.RecordList.Count > i + 1)
            {
                info = this.RecordList[i];
                if (info.VkCode == 2)
                {
                    infoNext = this.RecordList[i + 1];
                    if (infoNext.VkCode == 3 && info.Keycode == infoNext.Keycode)
                    {
                        info.Str = "${" + (Keycode)info.Keycode + "}";
                        this.RecordList.RemoveAt(i + 1);
                    }
                }
                i++;
            }
        }

        private void DebugPrint(string str)
        {
            Debug.WriteLine(str);
            for (int i = 0; i < this.RecordList.Count; i++)
            {
                Debug.Write(i + ":");
                this.RecordList[i].DebugPrint();
            }
            Debug.WriteLine("");
        }
    }
}
