package jp.sourceforge.pdt_tools.callhierarchy;

import java.io.IOException;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.dltk.ast.Modifiers;
import org.eclipse.dltk.ast.references.SimpleReference;
import org.eclipse.dltk.core.DLTKCore;
import org.eclipse.dltk.core.IMethod;
import org.eclipse.dltk.core.IModelElement;
import org.eclipse.dltk.core.ISourceModule;
import org.eclipse.dltk.core.ModelException;
import org.eclipse.dltk.core.search.IDLTKSearchScope;
import org.eclipse.dltk.internal.core.Model;
import org.eclipse.dltk.internal.core.ProjectFragment;
import org.eclipse.dltk.internal.core.ScriptFolder;
import org.eclipse.dltk.internal.core.ScriptProject;
import org.eclipse.dltk.internal.core.SourceModule;
import org.eclipse.dltk.internal.core.SourceType;
import org.eclipse.dltk.internal.core.search.DLTKSearchScope;
import org.eclipse.php.internal.core.ast.nodes.Expression;
import org.eclipse.php.internal.core.ast.nodes.FunctionInvocation;
import org.eclipse.php.internal.core.ast.nodes.IFunctionBinding;
import org.eclipse.php.internal.core.ast.nodes.IMethodBinding;
import org.eclipse.php.internal.core.ast.nodes.MethodInvocation;
import org.eclipse.php.internal.core.ast.nodes.Program;
import org.eclipse.php.internal.core.ast.nodes.StaticMethodInvocation;
import org.eclipse.php.internal.core.ast.visitor.AbstractVisitor;
import org.eclipse.php.internal.core.project.PHPNature;
import org.eclipse.php.internal.core.typeinference.PHPModelUtils;
import org.eclipse.php.ui.editor.SharedASTProvider;

@SuppressWarnings("restriction")
public class FindCallVisitor extends AbstractVisitor {

	private Map<SimpleReference, IModelElement> map;
	private IMethod method;
	private ISourceModule sourceModule;
	private boolean isFunction;

	public FindCallVisitor() {
		map = new TreeMap<SimpleReference, IModelElement>(
				new Comparator<SimpleReference>() {
					public int compare(SimpleReference o1, SimpleReference o2) {
						return o1.sourceStart() - o2.sourceStart();
					}
				});
	}

	/**
	 * 
	 * @param parent
	 * @param member
	 * @param scope
	 * @param monitor
	 * @return
	 */
	public Map<SimpleReference, IModelElement> find(IModelElement parent,
			IModelElement member, IDLTKSearchScope scope,
			IProgressMonitor monitor) {
		method = (IMethod) member;
		isFunction = false;
		IModelElement p = method.getParent();
		if (p instanceof SourceModule) {
			isFunction = true;
		} else if (p instanceof SourceType) {
			try {
				int flags = ((SourceType) p).getFlags();
				if ((flags & Modifiers.AccNameSpace) != 0) {
					isFunction = true;
				}
			} catch (ModelException e) {
				CallHierarchyPlugin.log(e);
			}
		}
		try {
			IResource[] resources = getResources(scope, member);
			for (IResource resource : resources) {
				if (monitor.isCanceled()) {
					map.clear();
					return map;
				}
				sourceModule = (ISourceModule) DLTKCore.create(resource);
				try {
					Program program = SharedASTProvider.getAST(sourceModule,
							SharedASTProvider.WAIT_YES, null);
					if (program != null) {
						program.accept(this);
					}
				} catch (ModelException e) {
					CallHierarchyPlugin.log(e);
				} catch (IOException e) {
					CallHierarchyPlugin.log(e);
				}
			}
		} catch (ModelException e) {
			CallHierarchyPlugin.log(e);
		}
		return map;
	}

	private IResource[] getResources(IDLTKSearchScope scope,
			IModelElement element) throws ModelException {
		List<IResource> list = new LinkedList<IResource>();
		if (scope instanceof DLTKSearchScope) {
			IPath[] paths = ((DLTKSearchScope) scope)
					.enclosingProjectsAndZips();
			if (paths.length == 1 && !scope.encloses(paths[0].toString())) {
				return new IResource[] { element.getResource() };
			}
			IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
			for (IPath path : paths) {
				IResource member = root.findMember(path);
				if (member != null && member instanceof IProject) {
					IProject project = (IProject) member;
					try {
						if (project.getNature(PHPNature.ID) != null) {
							IModelElement scriptProject = DLTKCore
									.create(project);
							list.addAll(getResources(scriptProject, scope));
						}
					} catch (CoreException e) {
						CallHierarchyPlugin.log(e);
					}
				}
			}
		}
		return list.toArray(new IResource[list.size()]);
	}

	private List<IResource> getResources(IModelElement element,
			IDLTKSearchScope scope) throws ModelException {
		List<IResource> list = new LinkedList<IResource>();
		if (scope.encloses(element)) {
			IModelElement[] elements = null;
			if (element instanceof Model) {
				elements = ((Model) element).getChildren();
			} else if (element instanceof ScriptProject) {
				if (((ScriptProject) element).isOpen()) {
					elements = ((ScriptProject) element).getChildren();
				}
			} else if (element instanceof ProjectFragment) {
				elements = ((ProjectFragment) element).getChildren();
			} else if (element instanceof ScriptFolder) {
				if (((ScriptFolder) element).containsScriptResources()) {
					elements = ((ScriptFolder) element).getChildren();
				}
			} else if (element instanceof SourceModule) {
				list.add(element.getResource());
			} else if (element instanceof SourceType) {
				list.add(element.getResource());
			}
			if (elements != null) {
				for (IModelElement element2 : elements) {
					list.addAll(getResources(element2, scope));
				}
			}
		}
		return list;
	}

	@Override
	public boolean visit(FunctionInvocation functionInvocation) {
		if (isFunction) {
			IFunctionBinding functionBinding = functionInvocation
					.resolveFunctionBinding();
			if (functionBinding != null) {
				if (functionBinding.getPHPElement().equals(method)) {
					Expression funcName = functionInvocation.getFunctionName()
							.getName();
					SimpleReference ref = new SimpleReference(
							functionInvocation.getStart(),
							functionInvocation.getEnd(),
							method.getElementName());
					IModelElement elm = PHPModelUtils.getCurrentMethod(
							sourceModule, funcName.getStart());
					if (elm == null) {
						elm = sourceModule;
					}
					map.put(ref, elm);
				}
			}
		}
		return super.visit(functionInvocation);
	}

	@Override
	public boolean visit(MethodInvocation methodInvocation) {
		IMethodBinding methodBinding = methodInvocation.resolveMethodBinding();
		if (methodBinding != null) {
			if (methodBinding.getPHPElement().equals(method)) {
				SimpleReference ref = new SimpleReference(methodInvocation
						.getMethod().getStart(), methodInvocation.getMethod()
						.getEnd(), method.getElementName());
				IModelElement elm = PHPModelUtils.getCurrentMethod(
						sourceModule, methodInvocation.getStart());
				if (elm == null) {
					elm = sourceModule;
				}
				map.put(ref, elm);
			}
		}
		return super.visit(methodInvocation);
	}

	@Override
	public boolean visit(StaticMethodInvocation staticMethodInvocation) {
		IMethodBinding methodBinding = staticMethodInvocation
				.resolveMethodBinding();
		if (methodBinding != null) {
			if (methodBinding.getPHPElement().equals(method)) {
				SimpleReference ref = new SimpleReference(
						staticMethodInvocation.getMethod().getStart(),
						staticMethodInvocation.getMethod().getEnd(),
						method.getElementName());
				IModelElement elm = PHPModelUtils.getCurrentMethod(
						sourceModule, staticMethodInvocation.getStart());
				if (elm == null) {
					elm = sourceModule;
				}
				map.put(ref, elm);
			}
		}
		return super.visit(staticMethodInvocation);
	}

}
