package jp.sourceforge.pdt_tools.external.assist.handlers;

import jp.sourceforge.pdt_tools.external.assist.Activator;
import jp.sourceforge.pdt_tools.external.assist.filters.ProjectFilter;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.filesystem.EFS;
import org.eclipse.core.filesystem.IFileInfo;
import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IProjectNature;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.osgi.util.NLS;
import org.eclipse.php.internal.core.PHPLanguageToolkit;
import org.eclipse.php.internal.core.project.PHPNature;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.handlers.HandlerUtil;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.progress.UIJob;

@SuppressWarnings("restriction")
public class OpenFileHandler extends AbstractHandler {

	public OpenFileHandler() {
	}

	public Object execute(ExecutionEvent event) throws ExecutionException {
		IWorkbenchWindow window = HandlerUtil
				.getActiveWorkbenchWindowChecked(event);

		IProject project = ResourcesPlugin.getWorkspace().getRoot()
				.getProject(ProjectFilter.PROJECT_NAME);
		if (!project.exists()) {
			project = createProject(project);
		}
		if (project == null) {
			String msgFmt = Messages.OpenFileHandler_CreationFailure;
			String msg = NLS.bind(msgFmt, ProjectFilter.PROJECT_NAME);
			MessageDialog.open(MessageDialog.ERROR, window.getShell(),
					Messages.OpenLocalFileAction_title, msg, SWT.SHEET);
			return null;
		}
		if (!isValidProject(project)) {
			String msgFmt = Messages.OpenFileHandler_InvalidProject;
			String msg = NLS.bind(msgFmt, ProjectFilter.PROJECT_NAME);
			MessageDialog.open(MessageDialog.ERROR, window.getShell(),
					Messages.OpenLocalFileAction_title, msg, SWT.SHEET);
			return null;
		}

		FileDialog dialog = new FileDialog(window.getShell(), SWT.OPEN
				| SWT.MULTI);
		dialog.setText(Messages.OpenLocalFileAction_title);
		String[] specs = ((PHPLanguageToolkit) PHPLanguageToolkit.getDefault())
				.getLanguageFileExtensions();
		String extensions = new String();
		for (String spec : specs) {
			extensions = extensions.concat(";*.").concat(spec); //$NON-NLS-1$
		}
		if (extensions.length() > 0) {
			extensions = extensions.substring(1);
		}
		dialog.setFilterExtensions(new String[] { extensions, "*.*" }); //$NON-NLS-1$
		dialog.open();
		final String[] names = dialog.getFileNames();
		if (names == null || names.length == 0) {
			return null;
		}
		final String filterPath = dialog.getFilterPath();
		final IProject projectx = project;

		UIJob job = new UIJob("") {
			@Override
			public IStatus runInUIThread(IProgressMonitor monitor) {
				IWorkbenchWindow window = PlatformUI.getWorkbench()
						.getActiveWorkbenchWindow();
				int numberOfFilesNotFound = 0;
				StringBuffer notFound = new StringBuffer();
				IFileStore dirStore = EFS.getLocalFileSystem().getStore(
						new Path(filterPath));
				for (int i = 0; i < names.length; i++) {
					IFileStore fileStore = dirStore.getChild(names[i]);
					IFileInfo fetchInfo = fileStore.fetchInfo();
					if (!fetchInfo.isDirectory() && fetchInfo.exists()) {
						IPath path = new Path(filterPath + Path.SEPARATOR
								+ fetchInfo.getName());
						openFile(path.toOSString(), projectx, window);
					} else {
						if (++numberOfFilesNotFound > 1)
							notFound.append('\n'); //$NON-NLS-1$
						notFound.append(fileStore.getName());
					}
				}

				if (numberOfFilesNotFound > 0) {
					String msgFmt = numberOfFilesNotFound == 1 ? Messages.OpenLocalFileAction_message_fileNotFound
							: Messages.OpenLocalFileAction_message_filesNotFound;
					String msg = NLS.bind(msgFmt, notFound.toString());
					MessageDialog.open(MessageDialog.ERROR, window.getShell(),
							Messages.OpenLocalFileAction_title, msg, SWT.SHEET);
				}
				return Status.OK_STATUS;
			}
		};
		job.schedule();

		return null;
	}

	private IProject createProject(IProject project) {
		try {
			IProgressMonitor monitor = new NullProgressMonitor();
			project.create(monitor);
			project.open(monitor);
			IProjectDescription desc = project.getDescription();
			desc.setNatureIds(new String[] { PHPNature.ID });
			project.setDescription(desc, monitor);
			project.build(IncrementalProjectBuilder.AUTO_BUILD, monitor);
		} catch (CoreException e) {
			Activator.log(e);
			if (project.exists()) {
				try {
					project.close(null);
					project.delete(true, null);
				} catch (CoreException e1) {
					Activator.log(e1);
				}
			}
			project = null;
		}
		return project;
	}

	private boolean isValidProject(IProject project) {
		try {
			IProjectNature nature = project.getNature(PHPNature.ID);
			if (nature != null) {
				return true;
			}
		} catch (CoreException e) {
			Activator.log(e);
		}
		return false;
	}

	private void openFile(String name, IProject project, IWorkbenchWindow window) {
		IPath location = new Path(name);
		String fileExt = location.getFileExtension();
		if (fileExt == null) {
			fileExt = ""; //$NON-NLS-1$
		} else if (fileExt.length() > 0) {
			fileExt = "." + fileExt; //$NON-NLS-1$
		}
		String fileName = location.removeFileExtension().lastSegment();
		IFile link = project.getFile(fileName + fileExt);
		int number = 0;
		while (link.exists()) {
			link = project.getFile(fileName
					+ String.format(" (%d)" + fileExt, ++number)); //$NON-NLS-1$
		}
		IWorkspace workspace = ResourcesPlugin.getWorkspace();
		IStatus status = workspace.validateLinkLocation(link, location);
		if (status.isOK()) {
			try {
				IProgressMonitor monitor = new NullProgressMonitor();
				link.createLink(location, IResource.NONE, monitor);
				IWorkbenchPage page = window.getActivePage();
				IEditorPart editor = IDE.openEditor(page, link);
				page.addPartListener(new EditorPartListener(editor, link, page));
				return;
			} catch (CoreException e) {
				Activator.log(e);
			}
		} else {
			Activator.logError(new RuntimeException(status.getMessage()));
		}
		String msg = NLS.bind(Messages.OpenLocalFileAction_message_errorOnOpen,
				name);
		MessageDialog.open(MessageDialog.ERROR, window.getShell(),
				Messages.OpenLocalFileAction_title, msg, SWT.SHEET);
	}

	private class EditorPartListener implements IPartListener {
		private IEditorPart editorPart;
		private IResource resource;
		private IWorkbenchPage page;

		public EditorPartListener(IEditorPart editorPart, IResource resource,
				IWorkbenchPage page) {
			this.editorPart = editorPart;
			this.resource = resource;
			this.page = page;
		}

		@Override
		public void partOpened(IWorkbenchPart part) {
		}

		@Override
		public void partDeactivated(IWorkbenchPart part) {
		}

		@Override
		public void partClosed(IWorkbenchPart part) {
			if (part.equals(editorPart)) {
				try {
					resource.delete(true, null);
				} catch (CoreException e) {
					Activator.log(e);
				}
				page.removePartListener(this);
				editorPart = null;
				resource = null;
			}
		}

		@Override
		public void partBroughtToTop(IWorkbenchPart part) {
		}

		@Override
		public void partActivated(IWorkbenchPart part) {
		}
	}
}
