﻿using System;
using System.Collections.Generic;
using System.Net;
using System.Text;
using System.Threading.Tasks;
using Progressive.PecaStarter.Common;
using Progressive.PecaStarter.Model;
using Progressive.PecaStarter.Model.Internal;
using Progressive.PecaStarter.Util;

namespace Progressive.PecaStarter.Dao
{
    class PeercastDao : IDisposable
    {
        private Progressive.PecaStarter.Commons.Daos.WebRequest webDao;
        private Peercast4Net.Peercast4Net peercast4Net;

        public string Address
        {
            get { return peercast4Net.Address; }
            set { peercast4Net.Address = value; }
        }
        public bool IsBusy { get { return String.IsNullOrEmpty(Address) || webDao.IsBusy; } }

        public PeercastDao()
        {
            webDao = new Progressive.PecaStarter.Commons.Daos.WebRequest();
            peercast4Net = new Peercast4Net.Peercast4Net();
        }

        #region IDisposable メンバー
        public void Dispose()
        {
            webDao.Dispose();
        }
        #endregion

        public async Task<Result> Fetch(string url, string name, string genre, string description, string contactUrl, string type)
        {
            try
            {
                await Task.Factory.FromAsync(
                        peercast4Net.BeginBroadcast(url, name, genre, description, contactUrl, type, null, null),
                        peercast4Net.EndBroadcast);
            }
            catch (WebException e)
            {
                if (e.Status == WebExceptionStatus.ConnectFailure)
                {
                    return Result.ConnectFailure;
                }
                throw e;
            }
            return Result.Success;
        }

        public async Task<Result> SetMeta(
            string name, string genre, string description, string url, string comment,
            string trackArtist, string trackTitle, string trackAlbum, string trackGenre, string trackContact)
        {
            try
            {
                await Task.Factory.FromAsync(
                        peercast4Net.BeginUpdate(name, genre, description, url, comment,
                                trackArtist, trackTitle, trackAlbum, trackGenre, trackContact, null, null),
                        peercast4Net.EndUpdate);
            }
            catch (WebException e)
            {
                if (e.Status == WebExceptionStatus.ConnectFailure)
                {
                    return Result.ConnectFailure;
                }
                throw e;
            }
            return Result.Success;
        }

        public async Task<Result> Stop(string id)
        {
            try
            {
                await Task.Factory.FromAsync(
                        peercast4Net.BeginStop(id, null, null),
                        peercast4Net.EndStop);
            }
            catch (WebException e)
            {
                if (e.Status == WebExceptionStatus.ConnectFailure)
                {
                    return Result.ConnectFailure;
                }
                throw e;
            }
            return Result.Success;
        }

        public async Task<Result> Keep(string id)
        {
            try
            {
                await Task.Factory.FromAsync(
                        peercast4Net.BeginKeep(id, null, null),
                        peercast4Net.EndKeep);
            }
            catch (WebException e)
            {
                if (e.Status == WebExceptionStatus.ConnectFailure)
                {
                    return Result.ConnectFailure;
                }
                throw e;
            }
            return Result.Success;
        }

        public async Task<Result> Apply(IList<string[]> parameters)
        {
            var sb = new StringBuilder(PeercastServerDefine.ApplyUrl);
            foreach (var item in parameters)
            {
                sb.Append('&').Append(item[0]).Append('=').Append(Encode(item[1]));
            }
            return await Access(sb.ToString());
        }

        public async Task<ResultInfo<string>> ViewXml()
        {
            try
            {
                return ResultInfo.Create(Result.Success,
                        await Task.Factory.FromAsync<string>(
                                peercast4Net.BeginGetStatus(null, null),
                                peercast4Net.EndGetStatus));
            }
            catch (WebException e)
            {
                if (e.Status == WebExceptionStatus.ConnectFailure)
                {
                    return ResultInfo.Create(Result.ConnectFailure, "");
                }
                throw e;
            }
        }

        public async Task<ResultInfo<string>> GetSettingsHtml()
        {
            return await Download(PeercastServerDefine.SettingsHtmlUrl);
        }

        private string UTF8Encode(string str)
        {
            string sjis = HttpUtils.UrlEncode(str, Encoding.UTF8);
            if (sjis.Length > PeercastDefine.MaxParameterLength)
            {
                throw new ArgumentException();
            }
            return sjis;
        }
        private string Encode(string str)
        {
            string sjis = HttpUtils.UrlEncode(str, PeercastDefine.Encoding);
            if (sjis.Length > PeercastDefine.MaxParameterLength)
            {
                throw new ArgumentException();
            }
            return sjis;
        }

        private async Task<Result> Access(string path)
        {
            try
            {
                await webDao.Access("http://" + Address + path);
            }
            catch (WebException e)
            {
                if (e.Status == WebExceptionStatus.ConnectFailure)
                {
                    return Result.ConnectFailure;
                }
                throw e;
            }
            return Result.Success;
        }

        private async Task<ResultInfo<string>> Download(string path)
        {
            // return ResultInfo.Create(Result.Success, await client.DownloadStringTaskAsync("http://" + address + path));
            try
            {
                return ResultInfo.Create(Result.Success, await webDao.GetText("http://" + Address + path));
            }
            catch (WebException e)
            {
                if (e.Status == WebExceptionStatus.ConnectFailure)
                {
                    return ResultInfo.Create(Result.ConnectFailure, "");
                }
                throw e;
            }
        }
    }
}
