﻿using System;
using System.Linq;
using System.Collections.Specialized;
using System.Threading.Tasks;
using Progressive.PecaStarter.Util;
using Progressive.PecaStarter.Model.Internal;
using System.Threading;
using Progressive.PecaStarter.Dao;
using System.Collections.Generic;

namespace Progressive.PecaStarter.Model
{
    public class Peercast : IDisposable
    {
        private PeercastDao dao;
        private string name;
        private string id;

        public string Address { get { return dao.Address; } set { dao.Address = value; } }
        public bool IsBusy { get { return dao.IsBusy; } }
        public bool IsBroadcast { get; private set; }

        public Peercast()
        {
            dao = new PeercastDao();
            IsBroadcast = false;
            name = "";
        }

        public async Task<ResultInfo<Tuple<int, string>>> Broadcast(string yellowPages,
            string streamUrl, string name, string genre, string description, string type,
            string contactUrl, string comment,
            string trackArtist, string trackTitle, string trackAlbum, string trackGenre, string trackContact)
        {
            if (IsBusy || IsBroadcast)
            {
                throw new ApplicationException();
            }
            var exists = await Exists(name);
            if (!Try(exists)) return ResultInfo.Create(exists.Result, default(Tuple<int, string>));
            if (exists.Value)
            {
                return ResultInfo.Create(Result.ChannelDuplicated, default(Tuple<int, string>));
            }
            var setResult = await SetYellowPages(yellowPages);
            if (!Try(setResult)) return ResultInfo.Create(setResult, default(Tuple<int, string>));
            var fetchResult = await dao.Fetch(streamUrl, name, genre, description, contactUrl, type);
            if (!Try(fetchResult)) return ResultInfo.Create(fetchResult, default(Tuple<int, string>));
            IsBroadcast = true;
            this.name = name;
            var id = await GetChannelId(name);
            if (!Try(id)) return ResultInfo.Create(id.Result, default(Tuple<int, string>));
            if (id.Value == "")
            {
                return ResultInfo.Create(Result.CreatingChannelFailure, default(Tuple<int, string>));
            }
            if (id.Value == PeercastServerDefine.NullId)
            {
                await Stop();
                return ResultInfo.Create(Result.CreatingChannelFailure, default(Tuple<int, string>));
            }
            this.id = id.Value;
            var info = await Update(genre, description, contactUrl, comment,
                trackArtist, trackTitle, trackAlbum, trackGenre, trackContact);
            if (!Try(info))
            {
                await Stop();
                return ResultInfo.Create(Result.CreatingChannelFailure, default(Tuple<int, string>));
            }
            return ResultInfo.Create(info.Result, Tuple.Create(info.Value, id.Value));
        }

        public async Task<ResultInfo<int>> Update(string genre, string description,
            string contactUrl, string comment,
            string trackArtist, string trackTitle, string trackAlbum, string trackGenre, string trackContact)
        {
            if (IsBusy || !IsBroadcast)
            {
                throw new ApplicationException();
            }
            var result = await dao.SetMeta(name, genre, description, contactUrl, comment,
                    trackArtist, trackTitle, trackAlbum, trackGenre, trackContact);
            if (!Try(result)) return ResultInfo.Create(result, default(int));
            var xmlInfo = await GetXmlStatus();
            if (!Try(xmlInfo)) return ResultInfo.Create(xmlInfo.Result, default(int));
            var xml = xmlInfo.Value;
            var equals = xml.EqualsChannel(name, genre, description, contactUrl, comment,
                    trackTitle, trackArtist, trackAlbum, trackGenre, trackContact);
            if (!equals)
            {
                return ResultInfo.Create(Result.UpdatingChannelFailure, default(int));
            }
            return ResultInfo.Create(Result.Success, xml.GetBitrate(name));
        }

        public async Task<Result> Stop()
        {
            if (IsBusy || !IsBroadcast)
            {
                throw new ApplicationException();
            }
            for (int i = 0; i < 5; i++)
            {
                var stopResult = await dao.Stop(id);
                if (!Try(stopResult)) return stopResult;
                var exists = await Exists(name);
                if (!Try(exists.Result)) return exists.Result;
                if (!exists.Value)
                {
                    IsBroadcast = false;
                    return Result.Success;
                }
                await dao.Keep(id);
            }
            return Result.DeletingChannelFailure;
        }

        public async Task<ResultInfo<Tuple<int, int>>> GetListeners()
        {
            if (IsBusy)
            {
                throw new ApplicationException();
            }
            var info = await GetXmlStatus();
            if (info.Result != Result.Success)
            {
                return ResultInfo.Create(info.Result, Tuple.Create(0, 0));
            }
            return ResultInfo.Create(Result.Success, info.Value.GetHits(name));
        }

        public async Task<ResultInfo<IList<Channel>>> GetChannels()
        {
            var info = await GetXmlStatus();
            if (info.Result != Result.Success)
            {
                return ResultInfo.Create(info.Result, default(IList<Channel>));
            }
            return ResultInfo.Create(Result.Success, info.Value.Channels);
        }

        public void SetChannel(string name, string id)
        {
            if (IsBusy)
            {
                throw new ApplicationException();
            }
            this.name = name;
            this.id = id;
            IsBroadcast = true;
        }

        private bool Try(Result result)
        {
            return result == Result.Success;
        }

        private bool Try<T>(ResultInfo<T> info)
        {
            return info.Result == Result.Success;
        }

        private async Task<ResultInfo<bool>> Exists(string name)
        {
            var result = await GetXmlStatus();
            if (result.Result != Result.Success)
            {
                return ResultInfo.Create(result.Result, default(bool));
            }
            return ResultInfo.Create(Result.Success, result.Value.Exists(name));
        }

        private async Task<ResultInfo<string>> GetChannelId(string name)
        {
            string id = "";
            for (int i = 0; i < 5; i++)
            {
                var result = await GetXmlStatus();
                if (result.Result != Result.Success)
                {
                    return ResultInfo.Create(result.Result, default(string));
                }
                id = result.Value.GetChannelId(name);
                if (id != "" && id != PeercastServerDefine.NullId)
                {
                    return ResultInfo.Create(Result.Success, id);
                }
                await Wait(1000);
            }
            return ResultInfo.Create(Result.Success, id);
        }

        private async Task Wait(int ms)
        {
            await Task.Factory.StartNew(() => Thread.Sleep(ms));
        }

        private async Task<Result> SetYellowPages(string host)
        {
            var gettingResult = await dao.GetSettingsHtml();
            if (gettingResult.Result != Result.Success)
            {
                return gettingResult.Result;
            }
            IList<string[]> nvc;
            if (!PeercastUtils.ToSettings(gettingResult.Value, out nvc))
            {
                return Result.GettingSettingsHtmlFailure;
            }
            if (host == nvc.Where(p => p[0] == "yp").Single()[1])
            {
                return Result.Success;
            }
            nvc.Where(p => p[0] == "yp").Single()[1] = host;
            var applyingResult = await dao.Apply(nvc);
            if (applyingResult != Result.Success)
            {
                return applyingResult;
            }
            return Result.Success;
        }

        private async Task<ResultInfo<XmlStatus>> GetXmlStatus()
        {
            var result = await dao.ViewXml();
            if (result.Result != Result.Success)
            {
                return ResultInfo.Create(result.Result, (XmlStatus)null);
            }
            return ResultInfo.Create(Result.Success, new XmlStatus(result.Value));
        }

        #region IDisposable メンバー
        public void Dispose()
        {
            dao.Dispose();
        }
        #endregion
    }
}
