﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Dynamic;

namespace Progressive.PecaStarter.ViewModel
{
    public class DynamicObjectViewModel : DynamicObject, INotifyPropertyChanged, IDisposable
    {
        public event PropertyChangedEventHandler PropertyChanged;
        private readonly Dictionary<string, string> dictionary;

        public string this[string key]
        {
            get
            {
                string result;
                dictionary.TryGetValue(key, out result);
                return result;
            }
            set
            {
                dictionary[key] = value;
                NotifyDynamicPropertyChanged(key);
            }
        }

        public static dynamic Create()
        {
            return new DynamicObjectViewModel();
        }

        public DynamicObjectViewModel()
        {
            dictionary = new Dictionary<string, string>();
        }

        public override bool TryGetMember(GetMemberBinder binder, out object result)
        {
            bool boolResult;
            string stringResult;
            boolResult = dictionary.TryGetValue(binder.Name, out stringResult);
            result = stringResult;
            return boolResult;
        }

        public override bool TrySetMember(SetMemberBinder binder, object value)
        {
            if (!dictionary.ContainsKey(binder.Name) || !Equals(dictionary[binder.Name], value))
            {
                dictionary[binder.Name] = value.ToString();
                NotifyDynamicPropertyChanged(binder.Name);
            }
            return true;
        }

        public void Clear()
        {
            dictionary.Clear();
        }

        protected void NotifyDynamicPropertyChanged(string name)
        {
            if (PropertyChanged != null)
            {
                PropertyChanged(this, new PropertyChangedEventArgs(name));
            }
        }

        protected void NotifyPropertyChanged(string name)
        {
            if (GetType().GetProperty(name) == null)
            {
                throw new ArgumentException();
            }
            if (PropertyChanged == null)
            {
                return;
            }
            PropertyChanged(this, new PropertyChangedEventArgs(name));
        }

        #region From ViewModelBase
        private IDictionary<Tuple<INotifyPropertyChanged, string, string>, PropertyChangedEventHandler> relations;
        protected void AddRelation(INotifyPropertyChanged source, string destinationPropertyName)
        {
            AddRelation(source, null, destinationPropertyName);
        }
        protected void AddRelation(string sourcePropertyName, string destinationPropertyName)
        {
            AddRelation(this, sourcePropertyName, destinationPropertyName);
        }
        protected void AddRelation(INotifyPropertyChanged source, string sourcePropertyName, string destinationPropertyName)
        {
            if (sourcePropertyName != null && source.GetType().GetProperty(sourcePropertyName) == null)
            {
                throw new ArgumentException();
            }
            if (GetType().GetProperty(destinationPropertyName) == null)
            {
                throw new ArgumentException();
            }
            if (relations == null)
            {
                relations = new Dictionary<Tuple<INotifyPropertyChanged, string, string>, PropertyChangedEventHandler>();
            }
            var tuple = Tuple.Create(source, sourcePropertyName, destinationPropertyName);
            if (relations.ContainsKey(tuple))
            {
                return;
            }
            PropertyChangedEventHandler handler = (sender, e) =>
            {
                if (sourcePropertyName != null)
                {
                    if (e.PropertyName != sourcePropertyName)
                    {
                        return;
                    }
                }
                else
                {
                    if (source == this && e.PropertyName == destinationPropertyName)
                    {
                        return;
                    }
                }
                NotifyPropertyChanged(destinationPropertyName);
            };
            source.PropertyChanged += handler;
            relations.Add(tuple, handler);
        }
        protected void RemoveRelation(INotifyPropertyChanged source, string destinationPropertyName)
        {
            RemoveRelation(this, null, destinationPropertyName);
        }
        protected void RemoveRelation(string sourcePropertyName, string destinationPropertyName)
        {
            RemoveRelation(this, sourcePropertyName, destinationPropertyName);
        }
        protected void RemoveRelation(INotifyPropertyChanged source, string sourcePropertyName, string destinationPropertyName)
        {
            var tuple = Tuple.Create(source, sourcePropertyName, destinationPropertyName);
            tuple.Item1.PropertyChanged -= relations[tuple];
            relations.Remove(tuple);
            if (relations.Count == 0)
            {
                relations = null;
            }
        }
        protected void RemoveAllRelation()
        {
            if (relations != null)
            {
                foreach (var relation in relations)
                {
                    relation.Key.Item1.PropertyChanged -= relation.Value;
                }
                relations.Clear();
                relations = null;
            }
        }

        #region IDisposable メンバー

        public virtual void Dispose()
        {
            RemoveAllRelation();
        }

        #endregion
        #endregion
    }
}
