<?php
/*
 * This tag helper function misappropriated a tag helper function of symfony and changed it.
 * (c) 2004-2006 Fabien Potencier <fabien.potencier@symfony-project.com>
 */

/*
 * This file is part of the petitwork package.
 * (c) 2007-2008 Exbridge,inc. <info@exbridge.jp>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

/**
 * Project: petitwork: the PHP lightweight web framework
 * File:    function.tagHelpers.php
 *
 * @link http://exbridge.jp/
 * @author Y.Shimizu
 *         S.Tajima <tajima@exbridge.jp> バグFIX etc...
 * @version svn:$Id: function.tagHelpers.php 17 2009-05-09 06:21:54Z exbridge $
 * @copyright 2007-2008 Exbridge,Inc.
 */

/**
 * null2br
 * @return string
 */
function null2br($str)
{
    if (isNULL($str)) {
        $str = '<br/>';
    }
    return $str;
}

/**
 * br2null
 * @return string
 */
function br2null($str)
{
    if (strtolower($str)=='<br>' || strtolower($str)=='<br/>' || strtolower($str)=='<br />') {
        $str = '';
    }
    return $str;
}

/**
 * input タグヘルパー（数字）
 */
function input_s_tag($nm, $val, $options, $zeroFlg=true)
{
    if (!isNULL($options)) {
        if (is_array($options)) {
            $options['style'] = $options['style'].';ime-mode:disabled;text-align:right;padding-right:1px;';
        }
    }
    else {
        $options = array('style'=>'ime-mode:disabled;text-align:right;padding-right:1px;');
    }
    if ($val == 0 && $zeroFlg) {
        $val = '';
    }
    return input_tag($nm, $val, $options);
}

/**
 * input タグヘルパー（英数字）
 */
function input_a_tag($nm, $val, $options)
{
    if (!isNULL($options)) {
        if (is_array($options)) {
            $options['style'] = $options['style'].';ime-mode:disabled;text-align:left;';
        }
    }
    else {
        $options = array('style'=>'ime-mode:disabled;text-align:left;');
    }
    return input_tag($nm, $val, $options);
}

/**
 * input タグヘルパー（日本語入力）
 *
 * @param $nm   id,name
 * @param $val  value
 * @param $vals list
 * @param $ev   event
 */
function input_j_tag($nm, $val, $options) {
    if (!isNULL($options)) {
        if (is_array($options)) {
            $options['style'] = $options['style'].';ime-mode:active;';
        }
    }
    else {
        $options = array('style'=>'ime-mode:active');
    }
    return input_tag($nm, $val, $options);
}

/**
 * Returns an HTML <form> tag that points to a valid action, route or URL as defined by <i>$url_for_options</i>.
 *
 * By default, the form tag is generated in POST format, but can easily be configured along with any additional
 * HTML parameters via the optional <i>$options</i> parameter. If you are using file uploads, be sure to set the
 * <i>multipart</i> option to true.
 *
 * <b>Options:</b>
 * - multipart - When set to true, enctype is set to "multipart/form-data".
 *
 * <b>Examples:</b>
 *     <code><?php echo form_tag('@myroute'); ?></code>
 *     <code><?php echo form_tag('/module/action', array('name' => 'myformname', 'multipart' => true)); ?></code>
 *
 * @param    string valid action, route or URL
 * @param    array optional HTML parameters for the <form> tag
 * @return string opening HTML <form> tag with options
 */
function form_tag($url_for_options = '', $options = array())
{
    $options = _parse_attributes($options);

    $html_options = $options;
    if (!isset($html_options['method'])){
        $html_options['method'] = 'post';
    }
    if (_get_option($html_options, 'multipart')){
        $html_options['enctype'] = 'multipart/form-data';
    }
    $html_options['action'] = convert_uri($url_for_options);
    return tag('form', $html_options, true);
}

/**
 * Returns an XHTML compliant <input> tag with type="text".
 *
 * The input_tag helper generates your basic XHTML <input> tag and can utilize any standard <input> tag parameters
 * passed in the optional <i>$options</i> parameter.
 *
 * <b>Examples:</b>
 * <code>
 *  echo input_tag('name');
 * </code>
 *
 * <code>
 *  echo input_tag('amount', $sf_params->get('amount'), array('size' => 8, 'maxlength' => 8));
 * </code>
 *
 * @param  string field name
 * @param  string selected field value
 * @param  array  additional HTML compliant <input> tag parameters
 * @return string XHTML compliant <input> tag with type="text"
 */
function input_tag($name, $value = null, $options = array())
{
    if (isset($options['editable'])) {
        if ($options['editable']===false) {
            $options['readonly'] = true;
            $options['tabindex'] = -1;
            $options['style'] = $options['style'] . 'background-color:#C0C0C0;';
        }
    }
    return tag('input', array_merge(array('type' => 'text', 'name' => $name, 'id' => get_id_from_name($name, $value), 'value' => $value), _convert_options($options)));
}

/**
 * Returns an XHTML compliant <input> tag with type="password".
 *
 * Similar to the input_tag helper, the input_hidden_tag helper generates your basic XHTML <input> tag and can utilize
 * any standard <input> tag parameters passed in the optional <i>$options</i> parameter.  The only difference is that it
 * creates the tag with type="password", meaning that the text entered into this field will not be visible to the end user.
 * In most cases it is replaced by  * * * * * * * *.  Even though this text is not readable, it is recommended that you do not
 * populate the optional <i>$value</i> option with a plain-text password or any other sensitive information, as this is a
 * potential security risk.
 *
 * <b>Examples:</b>
 * <code>
 *  echo input_password_tag('password');
 *  echo input_password_tag('password_confirm');
 * </code>
 *
 * @param  string field name
 * @param  string populated field value
 * @param  array  additional HTML compliant <input> tag parameters
 * @return string XHTML compliant <input> tag with type="password"
 * @see input_tag
 */
function input_password_tag($name = 'password', $value = null, $options = array())
{
    $options = _parse_attributes($options);
    $options['type'] = 'password';
    return input_tag($name, $value, $options);
}

/**
 * Returns an XHTML compliant <input> tag with type="hidden".
 *
 * Similar to the input_tag helper, the input_hidden_tag helper generates an XHTML <input> tag and can utilize
 * any standard <input> tag parameters passed in the optional <i>$options</i> parameter.  The only difference is
 * that it creates the tag with type="hidden", meaning that is not visible on the page.
 *
 * <b>Examples:</b>
 * <code>
 *  echo input_hidden_tag('id', $id);
 * </code>
 *
 * @param  string field name
 * @param  string populated field value
 * @param  array  additional HTML compliant <input> tag parameters
 * @return string XHTML compliant <input> tag with type="hidden"
 */
function input_hidden_tag($name, $value = null, $options = array())
{
    $options = _parse_attributes($options);
    $options['type'] = 'hidden';
    return input_tag($name, $value, $options);
}

/**
 * Returns an XHTML compliant <input> tag with type="file".
 *
 * Similar to the input_tag helper, the input_hidden_tag helper generates your basic XHTML <input> tag and can utilize
 * any standard <input> tag parameters passed in the optional <i>$options</i> parameter.  The only difference is that it
 * creates the tag with type="file", meaning that next to the field will be a "browse" (or similar) button.
 * This gives the user the ability to choose a file from there computer to upload to the web server.  Remember, if you
 * plan to upload files to your website, be sure to set the <i>multipart</i> option form_tag helper function to true
 * or your files will not be properly uploaded to the web server.
 *
 * <b>Examples:</b>
 * <code>
 *  echo input_file_tag('filename', array('size' => 30));
 * </code>
 *
 * @param  string field name
 * @param  array  additional HTML compliant <input> tag parameters
 * @return string XHTML compliant <input> tag with type="file"
 * @see input_tag, form_tag
 */
function input_file_tag($name, $options = array())
{
    $options = _parse_attributes($options);
    $options['type'] = 'file';
    return input_tag($name, null, $options);
}

/**
 * Returns an XHTML compliant <input> tag with type="button".
 *
 * The input_tag helper generates your basic XHTML <input> tag and can utilize any standard <input> tag parameters
 * passed in the optional <i>$options</i> parameter.
 *
 * <b>Examples:</b>
 * <code>
 *  echo button_tag('name');
 * </code>
 *
 * <code>
 *  echo button_tag('amount', $sf_params->get('amount'), array('size' => 8, 'maxlength' => 8));
 * </code>
 *
 * @param  string field name
 * @param  string selected field value
 * @param  array  additional HTML compliant <input> tag parameters
 * @return string XHTML compliant <input> tag with type="button"
 */
function button_tag($name, $value = null, $options = array())
{
    return tag('input', array_merge(array('type' => 'button', 'name' => $name, 'id' => get_id_from_name($name, $value), 'value' => $value), _convert_options($options)));
}

/**
 * Returns an XHTML compliant <input> tag with type="checkbox".
 *
 * When creating multiple checkboxes with the same name, be sure to use an array for the
 * <i>$name</i> parameter (i.e. 'name[]').    The checkbox_tag is smart enough to create unique ID's
 * based on the <i>$value</i> parameter like so:
 *
 * <samp>
 *    <input type="checkbox" name="status[]" id="status_3" value="3" />
 *    <input type="checkbox" name="status[]" id="status_4" value="4" />
 * </samp>
 *
 * <b>Examples:</b>
 * <code>
 *    echo checkbox_tag('newsletter', 1, $sf_params->get('newsletter'));
 * </code>
 *
 * <code>
 *    echo checkbox_tag('option_a', 'yes', true, array('class' => 'style_a'));
 * </code>
 *
 * <code>
 *    // one request variable with an array of checkbox values
 *    echo checkbox_tag('choice[]', 1);
 *    echo checkbox_tag('choice[]', 2);
 *    echo checkbox_tag('choice[]', 3);
 *    echo checkbox_tag('choice[]', 4);
 * </code>
 *
 * <code>
 *    // assuming you have Prototype.js enabled, you could do this
 *    echo checkbox_tag('show_tos', 1, false, array('onclick' => "Element.toggle('tos'); return false;"));
 * </code>
 *
 * @param    string field name
 * @param    string checkbox value (if checked)
 * @param    bool     is the checkbox checked? (1 or 0)
 * @param    array    additional HTML compliant <input> tag parameters
 * @return string XHTML compliant <input> tag with type="checkbox"
 */
function checkbox_tag($name, $value = '1', $checked = false, $options = array())
{
    $html_options = array_merge(array('type' => 'checkbox', 'name' => $name, 'id' => get_id_from_name($name, $value), 'value' => $value), _convert_options($options));
    if ($checked) {
        $html_options['checked'] = 'checked';
    }
    return tag('input', $html_options);
}

/**
 * Returns an XHTML compliant <input> tag with type="radio".
 *
 * <b>Examples:</b>
 * <code>
 *    echo ' Yes '.radiobutton_tag('newsletter', 1);
 *    echo ' No '.radiobutton_tag('newsletter', 0);
 * </code>
 *
 * @param    string field name
 * @param    string radio button value (if selected)
 * @param    bool     is the radio button selected? (1 or 0)
 * @param    array    additional HTML compliant <input> tag parameters
 * @return string XHTML compliant <input> tag with type="radio"
 */
function radiobutton_tag($name, $value, $checked = false, $options = array())
{
    $html_options = array_merge(array('type' => 'radio', 'name' => $name, 'id' => get_id_from_name($name, $value), 'value' => $value), _convert_options($options));
    if ($checked) {
        $html_options['checked'] = 'checked';
    }
    return tag('input', $html_options);
}

/**
 * Returns an XHTML compliant <input> tag with type="submit".
 *
 * By default, this helper creates a submit tag with a name of <em>commit</em> to avoid
 * conflicts with other parts of the framework.    It is recommended that you do not use the name
 * "submit" for submit tags unless absolutely necessary. Also, the default <i>$value</i> parameter
 * (title of the button) is set to "Save changes", which can be easily overwritten by passing a
 * <i>$value</i> parameter.
 *
 * <b>Examples:</b>
 * <code>
 *    echo submit_tag();
 * </code>
 *
 * <code>
 *    echo submit_tag('Update Record');
 * </code>
 *
 * @param    string field value (title of submit button)
 * @param    array    additional HTML compliant <input> tag parameters
 * @return string XHTML compliant <input> tag with type="submit"
 */
function submit_tag($value = 'Save changes', $options = array())
{
    return tag('input', array_merge(array('type' => 'submit', 'name' => 'commit', 'value' => $value), $options));
}

/**
 * Returns an XHTML compliant <input> tag with type="image".
 *
 * The submit_image_tag is very similar to the submit_tag, the only difference being that it uses an image
 * for the submit button instead of the browser-generated default button. The image is defined by the
 * <i>$source</i> parameter and must be a valid image, either local or remote (URL). By default, this
 * helper creates a submit tag with a name of <em>commit</em> to avoid conflicts with other parts of the
 * framework.  It is recommended that you do not use the name "submit" for submit tags unless absolutely necessary.
 *
 * <b>Examples:</b>
 * <code>
 *  // Assuming your image is in the /web/images/ directory
 *  echo submit_image_tag('my_submit_button.gif');
 * </code>
 *
 * <code>
 *  echo submit_image_tag('http://mydomain.com/my_submit_button.gif');
 * </code>
 *
 * @param  string path to image file
 * @param  array  additional HTML compliant <input> tag parameters
 * @return string XHTML compliant <input> tag with type="image"
 */
function submit_image_tag($source, $options = array())
{
    if (!isset($options['alt'])) {
        $path_pos = strrpos($source, '/');
        $dot_pos = strrpos($source, '.');
        $begin = $path_pos ? $path_pos + 1 : 0;
        $nb_str = ($dot_pos ? $dot_pos : strlen($source)) - $begin;
        $options['alt'] = ucfirst(substr($source, $begin, $nb_str));
    }
    return tag('input', array_merge(array('type' => 'image', 'name' => 'commit', 'src' => image_path($source)), _convert_options_to_javascript(_convert_options($options))));
}

/**
 * Returns a <textarea> tag, optionally wrapped with an inline rich-text JavaScript editor.
 *
 * The texarea_tag helper generates a standard HTML <textarea> tag and can be manipulated with
 * any number of standard HTML parameters via the <i>$options</i> array variable.    However, the
 * textarea tag also has the unique capability of being transformed into a WYSIWYG rich-text editor
 * such as TinyMCE (http://tinymce.moxiecode.com) very easily with the use of some specific options:
 *
 * <b>Options:</b>
 *    - rich: A rich text editor class (for example sfRichTextEditorTinyMCE for TinyMCE).
 *
 * <b>Examples:</b>
 * <code>
 *    echo textarea_tag('notes');
 * </code>
 *
 * <code>
 *    echo textarea_tag('description', 'This is a description', array('rows' => 10, 'cols' => 50));
 * </code>
 *
 * @param    string field name
 * @param    string populated field value
 * @param    array    additional HTML compliant <textarea> tag parameters
 *
 * @return string <textarea> tag optionally wrapped with a rich-text WYSIWYG editor
 */
function textarea_tag($name, $content = null, $options = array())
{
    $options = _parse_attributes($options);
    if ($size = _get_option($options, 'size')){
        list($options['cols'], $options['rows']) = split('x', $size, 2);
    }
    return content_tag('textarea', escape_once((is_object($content)) ? $content->__toString() : $content), array_merge(array('name' => $name, 'id' => get_id_from_name(_get_option($options, 'id', $name), null)), _convert_options($options)));
}

/**
 * Creates an <input> button tag of the given name pointing to a routed URL
 * based on the module/action passed as argument and the routing configuration.
 * The syntax is similar to the one of link_to.
 *
 * <b>Options:</b>
 * - 'absolute' - if set to true, the helper outputs an absolute URL
 * - 'query_string' - to append a query string (starting by ?) to the routed url
 * - 'confirm' - displays a javascript confirmation alert when the button is clicked
 * - 'popup' - if set to true, the button opens a new browser window
 * - 'post' - if set to true, the button submits a POST request instead of GET (caution: do not use inside a form)
 *
 * <b>Examples:</b>
 * <code>
 *    echo button_to('Delete this page', 'my_module/my_action');
 *        => <input value="Delete this page" type="button" onclick="document.location.href='/path/to/my/action';" />
 * </code>
 *
 * @param    string name of the button
 * @param    string 'module/action' or '@rule' of the action
 * @param    array additional HTML compliant <input> tag parameters
 * @return string XHTML compliant <input> tag
 * @see        url_for, link_to
 */
function button_to($name, $internal_uri, $options = array())
{
    $html_options = _convert_options($options);
    $html_options['value'] = $name;
    $html_options['type'] = 'button';
    $html_options['onclick'] = "document.location.href='".convert_uri($internal_uri)."';";
    return tag('input', $html_options);
}

/**
 * Creates a <a> link tag of the given name using a routed URL
 * based on the module/action passed as argument and the routing configuration.
 * It's also possible to pass a string instead of a module/action pair to
 * get a link tag that just points without consideration.
 * If null is passed as a name, the link itself will become the name.
 * If an object is passed as a name, the object string representation is used.
 * One of the options serves for for creating javascript confirm alerts where
 * if you pass 'confirm' => 'Are you sure?', the link will be guarded
 * with a JS popup asking that question. If the user accepts, the link is processed,
 * otherwise not.
 *
 * <b>Options:</b>
 * - 'absolute' - if set to true, the helper outputs an absolute URL
 * - 'query_string' - to append a query string (starting by ?) to the routed url
 * - 'confirm' - displays a javascript confirmation alert when the link is clicked
 * - 'popup' - if set to true, the link opens a new browser window
 * - 'post' - if set to true, the link submits a POST request instead of GET (caution: do not use inside a form)
 *
 * <b>Note:</b> The 'popup' and 'post' options are not compatible with each other.
 *
 * <b>Examples:</b>
 * <code>
 *  echo link_to('Delete this page', 'my_module/my_action');
 *    => <a href="/path/to/my/action">Delete this page</a>
 *  echo link_to('Visit Hoogle', 'http://www.hoogle.com');
 *    => <a href="http://www.hoogle.com">Visit Hoogle</a>
 *  echo link_to('Delete this page', 'my_module/my_action', array('id' => 'myid', 'confirm' => 'Are you sure?', 'absolute' => true));
 *    => <a href="http://myapp.example.com/path/to/my/action" id="myid" onclick="return confirm('Are you sure?');">Delete this page</a>
 * </code>
 *
 * @param  string name of the link, i.e. string to appear between the <a> tags
 * @param  string 'module/action' or '@rule' of the action
 * @param  array additional HTML compliant <a> tag parameters
 * @return string XHTML compliant <a href> tag
 * @see    url_for
 */
function link_to($name = '', $internal_uri = '', $options = array())
{
    $html_options = _parse_attributes($options);
    $html_options['href'] = convert_uri($internal_uri);
    if (!strlen($name)) {
        $name = $html_options['href'];
    }
    return content_tag('a', $name, $html_options);
}

/**
 * If the condition passed as first argument is true,
 * creates a <a> link tag of the given name using a routed URL
 * based on the module/action passed as argument and the routing configuration.
 * If the condition is false, the given name is returned between <span> tags
 *
 * <b>Options:</b>
 * - 'tag' - the HTML tag that must enclose the name if the condition is false, defaults to <span>
 * - 'absolute' - if set to true, the helper outputs an absolute URL
 * - 'query_string' - to append a query string (starting by ?) to the routed url
 * - 'confirm' - displays a javascript confirmation alert when the link is clicked
 * - 'popup' - if set to true, the link opens a new browser window
 * - 'post' - if set to true, the link submits a POST request instead of GET (caution: do not use inside a form)
 *
 * <b>Examples:</b>
 * <code>
 *  echo link_to_if($user->isAdministrator(), 'Delete this page', 'my_module/my_action');
 *    => <a href="/path/to/my/action">Delete this page</a>
 *  echo link_to_if(!$user->isAdministrator(), 'Delete this page', 'my_module/my_action');
 *    => <span>Delete this page</span>
 * </code>
 *
 * @param  bool condition
 * @param  string name of the link, i.e. string to appear between the <a> tags
 * @param  string 'module/action' or '@rule' of the action
 * @param  array additional HTML compliant <a> tag parameters
 * @return string XHTML compliant <a href> tag or name
 * @see    link_to
 */
function link_to_if($condition, $name = '', $internal_uri = '', $options = array())
{
    if ($condition) {
        return link_to($name, $internal_uri, $options);
    }
    else {
        $html_options = _parse_attributes($options);
        $tag = _get_option($html_options, 'tag', 'span');
        return content_tag($tag, $name, $html_options);
    }
}

/**
 * If the condition passed as first argument is false,
 * creates a <a> link tag of the given name using a routed URL
 * based on the module/action passed as argument and the routing configuration.
 * If the condition is true, the given name is returned between <span> tags
 *
 * <b>Options:</b>
 * - 'tag' - the HTML tag that must enclose the name if the condition is true, defaults to <span>
 * - 'absolute' - if set to true, the helper outputs an absolute URL
 * - 'query_string' - to append a query string (starting by ?) to the routed url
 * - 'confirm' - displays a javascript confirmation alert when the link is clicked
 * - 'popup' - if set to true, the link opens a new browser window
 * - 'post' - if set to true, the link submits a POST request instead of GET (caution: do not use inside a form)
 *
 * <b>Examples:</b>
 * <code>
 *  echo link_to_unless($user->isAdministrator(), 'Delete this page', 'my_module/my_action');
 *    => <span>Delete this page</span>
 *  echo link_to_unless(!$user->isAdministrator(), 'Delete this page', 'my_module/my_action');
 *    => <a href="/path/to/my/action">Delete this page</a>
 * </code>
 *
 * @param  bool condition
 * @param  string name of the link, i.e. string to appear between the <a> tags
 * @param  string 'module/action' or '@rule' of the action
 * @param  array additional HTML compliant <a> tag parameters
 * @return string XHTML compliant <a href> tag or name
 * @see    link_to
 */
function link_to_unless($condition, $name = '', $url = '', $options = array())
{
    return link_to_if(!$condition, $name, $url, $options);
}

/**
 * Returns a link that'll trigger a javascript function using the
 * onclick handler and return false after the fact.
 *
 * Examples:
 *   <?php echo link_to_function('Greeting', "alert('Hello world!')") ?>
 *   <?php echo link_to_function(image_tag('delete'), "if confirm('Really?'){ do_delete(); }") ?>
 */
function link_to_function($name, $function, $html_options = array())
{
    $html_options = _parse_attributes($html_options);
    $html_options['href'] = isset($html_options['href']) ? convert_uri($html_options['href']) : '#';
    $html_options['onclick'] = $function.'; return false;';
    return content_tag('a', $name, $html_options);
}

/**
 * Returns a <select> tag, optionally comprised of <option> tags.
 *
 * The select tag does not generate <option> tags by default.
 * To do so, you must populate the <i>$option_tags</i> parameter with a string of valid HTML compliant <option> tags.
 * Fortunately, Symfony provides a handy helper function to convert an array of data into option tags (see options_for_select).
 * If you need to create a "multiple" select tag (ability to select multiple options), set the <i>multiple</i> option to true.
 * Doing so will automatically convert the name field to an array type variable (i.e. name="name" becomes name="name[]").
 *
 * <b>Options:</b>
 * - multiple - If set to true, the select tag will allow multiple options to be selected at once.
 *
 * <b>Examples:</b>
 * <code>
 *  $person_list = array(1 => 'Larry', 2 => 'Moe', 3 => 'Curly');
 *  echo select_tag('person', options_for_select($person_list, $sf_params->get('person')), array('class' => 'full'));
 * </code>
 *
 * <code>
 *  echo select_tag('department', options_for_select($department_list), array('multiple' => true));
 * </code>
 *
 * <code>
 *  echo select_tag('url', options_for_select($url_list), array('onChange' => 'Javascript:this.form.submit();'));
 * </code>
 *
 * @param  string field name
 * @param  mixed contains a string of valid <option></option> tags, or an array of options that will be passed to options_for_select
 * @param  array  additional HTML compliant <select> tag parameters
 * @return string <select> tag optionally comprised of <option> tags.
 * @see options_for_select, content_tag
 */
function select_tag($name, $option_tags = null, $options = array())
{
    $options = _convert_options($options);
    $id = $name;
    if (isset($options['multiple']) && $options['multiple'] && substr($name, -2) !== '[]') {
        $name .= '[]';
    }
    if (is_array($option_tags)) {
        $option_tags = options_for_select($option_tags);
    }
    return content_tag('select', $option_tags, array_merge(array('name' => $name, 'id' => get_id_from_name($id)), $options));
}

/**
 * Returns a formatted set of <option> tags based on optional <i>$options</i> array variable.
 *
 * The options_for_select helper is usually called in conjunction with the select_tag helper, as it is relatively
 * useless on its own. By passing an array of <i>$options</i>, the helper will automatically generate <option> tags
 * using the array key as the value and the array value as the display title. Additionally the options_for_select tag is
 * smart enough to detect nested arrays as <optgroup> tags.  If the helper detects that the array value is an array itself,
 * it creates an <optgroup> tag with the name of the group being the key and the contents of the <optgroup> being the array.
 *
 * <b>Options:</b>
 * - include_blank  - Includes a blank <option> tag at the beginning of the string with an empty value
 * - include_custom - Includes an <option> tag with a custom display title at the beginning of the string with an empty value
 *
 * <b>Examples:</b>
 * <code>
 *  echo select_tag('person', options_for_select(array(1 => 'Larry', 2 => 'Moe', 3 => 'Curly')));
 * </code>
 *
 * <code>
 *  $card_list = array('VISA' => 'Visa', 'MAST' => 'MasterCard', 'AMEX' => 'American Express', 'DISC' => 'Discover');
 *  echo select_tag('cc_type', options_for_select($card_list, 'AMEX', array('include_custom' => '-- Select Credit Card Type --')));
 * </code>
 *
 * <code>
 *  $optgroup_array = array(1 => 'Joe', 2 => 'Sue', 'Group A' => array(3 => 'Mary', 4 => 'Tom'), 'Group B' => array(5 => 'Bill', 6 =>'Andy'));
 *  echo select_tag('employee', options_for_select($optgroup_array, null, array('include_blank' => true)), array('class' => 'mystyle'));
 * </code>
 *
 * @param  array dataset to create <option> tags and <optgroup> tags from
 * @param  string selected option value
 * @param  array  additional HTML compliant <option> tag parameters
 * @return string populated with <option> tags derived from the <i>$options</i> array variable
 * @see select_tag
 */
function options_for_select($options = array(), $selected = '', $html_options = array())
{
    $html_options = _parse_attributes($html_options);
    if (is_array($selected)){
        $selected = array_map('strval', array_values($selected));
    }
    $html = '';
    if ($value = _get_option($html_options, 'include_custom')) {
        $html .= content_tag('option', $value, array('value' => ''))."\n";
    }
    else if (_get_option($html_options, 'include_blank')) {
        $html .= content_tag('option', '', array('value' => ''))."\n";
    }
    foreach ($options as $key => $value) {
        if (is_array($value)) {
            $html .= content_tag('optgroup', options_for_select($value, $selected, $html_options), array('label' => $key))."\n";
        }
        else {
            $option_options = array('value' => $key);
            if ((is_array($selected) && in_array(strval($key), $selected, true)) || (strval($key) === strval($selected))) {
                $option_options['selected'] = 'selected';
            }
            foreach ($html_options as $k => $v) {
                if (is_array($v)) {
                    $a = $v[$key];
                    $option_options[$k] = $a;
                }
            }
            $html .= content_tag('option', $value, $option_options)."\n";
        }
    }
    return $html;
}

function _get_option(&$options, $name, $default = null)
{
    if (array_key_exists($name, $options)) {
        $value = $options[$name];
        unset($options[$name]);
    }
    else {
        $value = $default;
    }
    return $value;
}

/**
 * Returns the path to an image asset.
 *
 * <b>Example:</b>
 * <code>
 *  echo image_path('foobar');
 *    => /images/foobar.png
 * </code>
 *
 * <b>Note:</b> The asset name can be supplied as a...
 * - full path, like "/my_images/image.gif"
 * - file name, like "rss.gif", that gets expanded to "/images/rss.gif"
 * - file name without extension, like "logo", that gets expanded to "/images/logo.png"
 *
 * @param  string asset name
 * @param  bool return absolute path ?
 * @return string file path to the image file
 * @see    image_tag
 */
function image_path($source)
{
    $template_dir = PW_LAYOUT_TEMPLATE;
    if ($template_dir!='') {
        $template_dir .= '/';
    }
    return _compute_public_path($source, $template_dir . 'images', 'png');
}

/**
 * Returns an <img> image tag for the asset given as argument.
 *
 * <b>Options:</b>
 * - 'absolute' - to output absolute file paths, useful for embedded images in emails
 * - 'alt'  - defaults to the file name part of the asset (capitalized and without the extension)
 * - 'size' - Supplied as "XxY", so "30x45" becomes width="30" and height="45"
 *
 * <b>Examples:</b>
 * <code>
 *  echo image_tag('foobar');
 *    => <img src="images/foobar.png" alt="Foobar" />
 *  echo image_tag('/my_images/image.gif', array('alt' => 'Alternative text', 'size' => '100x200'));
 *    => <img src="/my_images/image.gif" alt="Alternative text" width="100" height="200" />
 * </code>
 *
 * @param  string image asset name
 * @param  array additional HTML compliant <img> tag parameters
 * @return string XHTML compliant <img> tag
 * @see    image_path
 */
function image_tag($source, $options = array())
{
    if (!$source) {
        return '';
    }
    $options = _parse_attributes($options);
    $options['src'] = convert_uri(image_path($source));

    if (!isset($options['alt'])) {
        $path_pos = strrpos($source, '/');
        $dot_pos = strrpos($source, '.');
        $begin = $path_pos ? $path_pos + 1 : 0;
        $nb_str = ($dot_pos ? $dot_pos : strlen($source)) - $begin;
        $options['alt'] = ucfirst(substr($source, $begin, $nb_str));
    }
    if (isset($options['size'])) {
        list($options['width'], $options['height']) = split('x', $options['size'], 2);
        unset($options['size']);
    }
    return tag('img', $options);
}

function _compute_public_path($source, $dir, $ext)
{
    if (strpos($source, '://')) {
        return $source;
    }
    if (strpos($source, '/') !== 0) {
        $source = $dir.'/'.$source;
    }
    if (strpos(basename($source), '.') === false) {
        $source .= '.'.$ext;
    }
    return $source;
}

/**
 * Returns a formatted ID based on the <i>$name</i> parameter and optionally the <i>$value</i> parameter.
 *
 * This function determines the proper form field ID name based on the parameters. If a form field has an
 * array value as a name we need to convert them to proper and unique IDs like so:
 * <samp>
 *  name[] => name (if value == null)
 *  name[] => name_value (if value != null)
 *  name[bob] => name_bob
 *  name[item][total] => name_item_total
 * </samp>
 *
 * <b>Examples:</b>
 * <code>
 *  echo get_id_from_name('status[]', '1');
 * </code>
 *
 * @param  string field name
 * @param  string field value
 * @return string <select> tag populated with all the languages in the world.
 */
function get_id_from_name($name, $value = null)
{
    // check to see if we have an array variable for a field name
    if (strstr($name, '[')){
        $name = str_replace(array('[]', '][', '[', ']'), array((($value != null) ? '_'.$value : ''), '_', '_', ''), $name);
    }
    return $name;
}

/**
 * Converts specific <i>$options</i> to their correct HTML format
 * @param  array options
 * @return array returns properly formatted options
 */
function _convert_options($options)
{
    $options = _parse_attributes($options);
    foreach (array('disabled', 'readonly', 'multiple') as $attribute) {
        if (array_key_exists($attribute, $options)) {
            if ($options[$attribute]) {
                $options[$attribute] = $attribute;
            }
            else {
                unset($options[$attribute]);
            }
        }
    }
    return $options;
}

/**
 *
 * @param  string/array
 * @return string
 */
function _parse_attributes($string)
{
    return is_array($string) ? $string : stringToArray($string);
}

function stringToArray($string)
{
    preg_match_all('/
      \s*(\w+)              # key                               \\1
      \s*=\s*               # =
      (\'|")?               # values may be included in \' or " \\2
      (.*?)                 # value                             \\3
      (?(2) \\2)            # matching \' or " if needed        \\4
      \s*(?:
        (?=\w+\s*=) | \s*$  # followed by another key= or the end of the string
      )
    /x', $string, $matches, PREG_SET_ORDER);

    $attributes = array();
    foreach ($matches as $val) {
        $attributes[$val[1]] = literalize($val[3]);
    }
    return $attributes;
}

/**
 * Finds the type of the passed value, returns the value as the new type.
 * @param  string
 * @return mixed
 */
function literalize($value)
{
    // lowercase our value for comparison
    $value  = trim($value);
    $lvalue = strtolower($value);

    if (in_array($lvalue, array('null', '~', ''))) {
      $value = null;
    }
    else if (in_array($lvalue, array('true', 'on', '+', 'yes'))) {
      $value = true;
    }
    else if (in_array($lvalue, array('false', 'off', '-', 'no'))) {
      $value = false;
    }
    else if (ctype_digit($value)) {
      $value = (int) $value;
    }
    else if (is_numeric($value)) {
      $value = (float) $value;
    }
    return $value;
}

/**
 * Constructs an html tag.
 * @param  $name    string  tag name
 * @param  $options array   tag options
 * @param  $open    boolean true to leave tag open
 * @return string
 */
function tag($name, $options = array(), $open = false)
{
    if (!$name) {
        return '';
    }
    return '<'.$name._tag_options($options).(($open) ? '>' : ' />');
}

/**
 * Escapes an HTML string.
 *
 * @param  string HTML string to escape
 * @return string escaped string
 */
function escape_once($html)
{
    return fix_double_escape(htmlspecialchars($html));
}

/**
 * Fixes double escaped strings.
 *
 * @param  string HTML string to fix
 * @return string escaped string
 */
function fix_double_escape($escaped)
{
    return preg_replace('/&amp;([a-z]+|(#\d+)|(#x[\da-f]+));/i', '&$1;', $escaped);
}

function _tag_options($options = array())
{
    $options = _parse_attributes($options);

    $html = '';
    foreach ($options as $key => $value){
        $html .= ' '.$key.'="'.escape_once($value).'"';
    }
    return $html;
}

function content_tag($name, $content = '', $options = array())
{
    if (!$name) {
        return '';
    }
    return '<'.$name._tag_options($options).'>'.$content.'</'.$name.'>';
}

function _convert_options_to_javascript($html_options, $internal_uri = '')
{
    // post
    $post = isset($html_options['post']) ? $html_options['post'] : '';
    unset($html_options['post']);

    $onclick = isset($html_options['onclick']) ? $html_options['onclick'] : '';

    if ($post) {
        $html_options['onclick'] = $onclick._post_javascript_function().'return false;';
    }
    return $html_options;
}

function _post_javascript_function()
{
  return "f = document.createElement('form'); document.body.appendChild(f); f.method = 'POST'; f.action = this.href; f.submit();";
}
