/*
 * Created on 2005/07/01
 * 
 * Copyright (C) 2005- Monami software, LP.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Monami software, LP. - Initial API and implementation
 */
package jp.pizzafactory.debug.core;

import org.eclipse.cdt.core.IBinaryParser.IBinaryObject;
import org.eclipse.cdt.core.model.ICProject;
import org.eclipse.cdt.debug.core.CDIDebugModel;
import org.eclipse.cdt.debug.core.CDebugCorePlugin;
import org.eclipse.cdt.debug.core.ICDTLaunchConfigurationConstants;
import org.eclipse.cdt.debug.core.ICDebugConfiguration;
import org.eclipse.cdt.debug.core.cdi.CDIException;
import org.eclipse.cdt.debug.core.cdi.ICDISession;
import org.eclipse.cdt.debug.core.cdi.model.ICDITarget;
import org.eclipse.cdt.launch.AbstractCLaunchDelegate;
import org.eclipse.cdt.launch.internal.ui.LaunchMessages;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.debug.core.model.IDebugTarget;
import org.eclipse.debug.core.model.IProcess;

public class LaunchConfigurationDelegate extends AbstractCLaunchDelegate {

	public void launch(ILaunchConfiguration config, String mode, ILaunch launch, IProgressMonitor monitor) throws CoreException {
		if (monitor == null) {
			monitor = new NullProgressMonitor();
		}
		monitor.beginTask(LaunchMessages.getString("LocalRunLaunchDelegate.Launching_Local_C_Application"), 10); //$NON-NLS-1$
		// check for cancellation
		if (monitor.isCanceled()) {
			return;
		}
		if (!mode.equals(ILaunchManager.DEBUG_MODE)) {
			String id = CorePlugin.getUniqueIdentifier();
			throw new CoreException(new Status(Status.ERROR, id, 1000, "Supports debug mode only.", null));
		}

		try {
			monitor.worked(1);
			IPath exePath = verifyProgramPath(config);
			ICProject project = verifyCProject(config);
			IBinaryObject exeFile = verifyBinary(project, exePath);

			// set the default source locator if required
			setDefaultSourceLocator(launch, config);

			ICDebugConfiguration debugConfig = getDebugConfig(config);
			ICDISession dsession = null;
			String debugMode = config.getAttribute(ICDTLaunchConfigurationConstants.ATTR_DEBUGGER_START_MODE,
					ICDTLaunchConfigurationConstants.DEBUGGER_MODE_RUN);
			if (debugMode.equals(ICDTLaunchConfigurationConstants.DEBUGGER_MODE_CORE)) {
				String id = CorePlugin.getUniqueIdentifier();
				throw new CoreException(new Status(Status.ERROR, id, 1000, "Doesn't support core mode.", null));
			}
			dsession = debugConfig.createDebugger().createDebuggerSession(launch, exeFile,
					new SubProgressMonitor(monitor, 8));
			try {
				monitor.worked(1);

				ICDITarget[] targets = dsession.getTargets();
				for (int i = 0; i < targets.length; i++) {
					Process process = targets[i].getProcess();
					IProcess iprocess = null;
					boolean isSimulator = config.getAttribute(IGDBStubConfigurationConstants.ATTR_SIMULATOR, false);
					if (process != null) {
						iprocess = DebugPlugin.newProcess(launch, process, renderProcessLabel(exePath.toOSString()));
					}
					if (isSimulator) {
						/*            Terminate  Disconnect  Resume
						 * Simulator  o(true)    x(false)    x(false means run)
						 * Stub       x          o           o(true  means continue)
						 */
						CDIDebugModel.newDebugTarget(launch, project.getProject(), targets[i], renderTargetLabel(debugConfig),
								iprocess, exeFile, true, false, null, false);
						try {
							targets[i].restart();
						} catch (CDIException e2) {
							//TODO implement it
						}
					} else {
						/* For stub */
						newDebugTarget(launch, project.getProject(), targets[i], renderTargetLabel(debugConfig),
								iprocess, exeFile, false, true, null);
					}
				}
			} catch (CoreException e) {
				try {
					dsession.terminate();
				} catch (CDIException e1) {
					// ignore
				}
				throw e;
			}
		}
		finally {
			monitor.done();
		}
	}

	protected String getPluginID() {
		return CorePlugin.getUniqueIdentifier();
	}

	/**
	 * Creates and returns a debug target for the given CDI target, with the specified name, and associates it with the given process for console I/O. The debug
	 * target is added to the given launch.
	 * 
	 * @param launch the launch the new debug target will be contained in
	 * @param project the project to use to persist breakpoints.
	 * @param cdiTarget the CDI target to create a debug target for
	 * @param name the name to associate with this target, which will be returned from <code>IDebugTarget.getName</code>.
	 * @param debuggeeProcess the process to associate with the debug target, which will be returned from <code>IDebugTarget.getProcess</code>
	 * @param file the executable to debug.
	 * @param allowTerminate allow terminate().
	 * @param allowDisconnect allow disconnect().
	 * @param stopSymbol place temporary breakpoint at <code>stopSymbol</code>, ignore if <code>null</code> or empty.
	 * @param resumeTarget resume target.
	 * @return a debug target
	 * @throws DebugException
	 * @since 3.1
	 */
	public static IDebugTarget newDebugTarget( final ILaunch launch, final IProject project, final ICDITarget cdiTarget, final String name, final IProcess debuggeeProcess, final IBinaryObject file, final boolean allowTerminate, final boolean allowDisconnect, final String stopSymbol) throws DebugException {
		final IDebugTarget[] target = new IDebugTarget[1];
		IWorkspaceRunnable r = new IWorkspaceRunnable() {

			public void run( IProgressMonitor m ) throws CoreException {
				target[0] = new StubRemoteDebugTarget( launch, project, cdiTarget, name, debuggeeProcess, file, allowTerminate, allowDisconnect );
				((StubRemoteDebugTarget)target[0]).ready( stopSymbol);
			}
		};
		try {
			ResourcesPlugin.getWorkspace().run( r, null );
		}
		catch( CoreException e ) {
			CDebugCorePlugin.log( e );
			throw new DebugException( e.getStatus() );
		}
		return target[0];
	}
}