﻿/**
 *	アプリケーションのクラス。
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module outland.dmajor.application;

import win32.windows;
import outland.dmajor.exception;
import outland.dmajor.tstring;
import outland.dmajor.trace;

/// モジュール基本クラス。リソース管理などを行う。
class Module {
	
	/// ハンドル型。
	alias HINSTANCE Handle;
	
	/// インスタンスハンドル。
	Handle handle() {return handle_;}
	
	/// リソースの検索。
	void* findResource(LPCTSTR type, LPCTSTR name, WORD lang) {
		HRSRC rsrc = FindResourceEx(handle, type, name, lang);
		checkApi(rsrc);
		HGLOBAL ptr = LoadResource(handle, rsrc);
		checkApi(rsrc);
		return cast(void*) ptr;
	}
	
	/// ditto
	void* findResource(LPCTSTR type, LPCTSTR name) {
		return findResource(type, name, MAKELANGID(LANG_NEUTRAL, SUBLANG_NEUTRAL));
	}
	
	/// ditto
	void* findResource(LPCTSTR type, char[] name, WORD lang) {
		return findResource(type, toTString(name), lang);
	}
	
	/// ditto
	void* findResource(LPCTSTR type, char[] name) {return findResource(type, toTString(name));}
	
	/// ditto
	void* findResource(LPCTSTR type, uint id, WORD lang) {return findResource(type, MAKEINTRESOURCE(id), lang);}
	
	/// ditto
	void* findResource(LPCTSTR type, uint id) {return findResource(type, MAKEINTRESOURCE(id));}
	
	/// リソースのサイズを得る。
	size_t getResourceSize(LPCTSTR type, LPCTSTR name, WORD lang) {
		HRSRC rsrc = FindResourceEx(handle, type, name, lang);
		if(!rsrc) {
			return 0;
		}
		return SizeofResource(handle, rsrc);
	}
	
	/// ditto
	size_t getResourceSize(LPCTSTR type, LPCTSTR name) {
		return getResourceSize(type, name, MAKELANGID(LANG_NEUTRAL, SUBLANG_NEUTRAL));
	}
	
	/// ditto
	size_t getResourceSize(LPCTSTR type, uint id, WORD lang) {
		return getResourceSize(type, MAKEINTRESOURCE(id), lang);
	}
	
	/// ditto
	size_t getResourceSize(LPCTSTR type, uint id) {
		return getResourceSize(type, MAKEINTRESOURCE(id));
	}
	
	/// リソースの取得。
	int loadTString(uint id, TCHAR[] buf) {
		return LoadString(handle, id, buf.ptr, buf.length);
	}
	
	/// ditto
	TCHAR[] loadTString(uint id) {
		TCHAR[] buf;
		int len = 32;
		while(buf.length <= len) {
			buf.length = len * 2;
			len = loadTString(id, buf);
		}
		buf.length = (len > 0) ? len : 0;
		return buf;
	}
	
	/// ditto
	char[] loadString(uint id) {return fromTString(loadTString(id).ptr);}
	
	/// ditto
	HBITMAP loadBitmap(char[] name) {return LoadBitmap(handle, toTString(name));}
	
	/// ditto
	HBITMAP loadBitmap(uint id) {return LoadBitmap(handle, MAKEINTRESOURCE(id));}
	
	/// ditto
	HICON loadIcon(char[] name) {return LoadIcon(handle, toTString(name));}
	
	/// ditto
	HICON loadIcon(uint id) {return LoadIcon(handle, MAKEINTRESOURCE(id));}
	
	/// ditto
	HCURSOR loadCursor(char[] name) {return LoadCursor(handle, toTString(name));}
	
	/// ditto
	HCURSOR loadCursor(uint id) {return LoadCursor(handle, MAKEINTRESOURCE(id));}
	
protected:
	
	/// ハンドルを指定して生成する。
	this(HINSTANCE h) {handle_ = h;}
	
private:
	
	/// デフォルトコンストラクタ。
	this() {}
	
	/// インスタンスハンドル。
	Handle handle_;
}

/// DLL。
class DynamicLinkLibrary : Module {
	
	/// ロードする。参照カウントが増える。
	static DynamicLinkLibrary loadLibrary(char[] path) {
		HINSTANCE h = LoadLibrary(toTString(path));
		checkApi(h);
		scope(failure) FreeLibrary(h);
		DynamicLinkLibrary dll = new DynamicLinkLibrary(h);
		dll.isFree_ = true;
		return dll;
	}
	
	/// 既にロード済みのDLLを取得する。
	static DynamicLinkLibrary getLibrary(char[] path) {
		HINSTANCE h = GetModuleHandle(toTString(path));
		if(!h) {
			return null;
		} else {
			return new DynamicLinkLibrary(h);
		}
	}
	
	/// 破棄する。
	~this() {
		if(isFree_) {
			FreeLibrary(handle);
		}
	}
	
	/// 関数を検索する。
	FARPROC findProcedure(char[] name) {
		return GetProcAddress(handle, toTString(name));
	}
	
	/// 関数を検索する。
	FARPROC findProcedure(uint i) {
		return GetProcAddress(handle, MAKEINTRESOURCE(i));
	}
	
	/// 関数を検索する。
	FARPROC getProcedure(char[] name) {
		FARPROC proc = findProcedure(name);
		checkApi(proc);
		return proc;
	}
	
	/// 関数を検索する。
	FARPROC getProcedure(uint i) {
		FARPROC proc = findProcedure(i);
		checkApi(proc);
		return proc;
	}
	
protected:
	
	/// 生成する。
	this(HINSTANCE h) {super(h);}
	
private:
	
	/// 破棄するかどうか。
	bool isFree_ = false;
}

/// アプリケーション基本クラス。
class Application : Module {
	
	/** アプリケーション初期化。
	 *
	 *	デフォルトでは何もしない。
	 */
	void initialize() {}
	
	/** アプリケーション終了。
	 *
	 *	デフォルトでは何もしない。
	 *	initializeで失敗した場合も呼び出される。
	 */
	void finalize() {}
	
	/// インスタンスを返す。
	static Application instance() {return application_;}
	
	/// メイン関数の実行。
	int main(Handle inst, Handle prev, LPSTR cmd, int show) {
		handle_ = inst;
		preview_ = prev;
		command_ = cmd;
		show_ = show;
		
		// 初期化・終了処理。
		scope(exit) finalize();
		initialize();
		
		// 実行。
		return run();
	}
	
	/// メッセージループ実行。
	int run() {		
		// メッセージループ。
		for(;;) {
			// メッセージの有無を確かめる。
			if(!peekMessage()) {
				// メッセージなし。アイドル処理を行う。
				// アイドル処理もなくなればブロックを伴うメッセージ待ちに入る。
				if(onIdle(++idle_)) {
					continue;
				}
			}
			
			// アイドル処理リセット。
			idle_ = 0;
			
			// メッセージを受信して処理する。
			if(!processMessage()) {
				// 終了メッセージが来た。
				break;
			}
		}
		
		// 終了コードを返す。
		return exitCode_;
	}
	
	/// アプリケーション終了。
	void quit(int code) {PostQuitMessage(code);}
	
	/// 以前のインスタンスハンドル。
	Handle preview() {return preview_;}
	
	/// コマンドライン文字列。
	LPSTR command() {return command_;}
	
	/// 表示フラグ。
	int showFlag() {return show_;}
	
	/// 現在実行中のメッセージ。
	MSG* currentMessage() {return &msg_;}
	
	/// メッセージが来ているか確認する。
	bool peekMessage() {return PeekMessage(&msg_, null, 0, 0, PM_NOREMOVE) != FALSE;}
	
	/**	メッセージが来るまで待ち、処理を行う。
	 *
	 *	Returns:
	 *		trueならばアプリケーション続行。falseならばアプリケーション終了。
	 *	Throws:
	 *		ApiException	メッセージ取得失敗時に投げられる。
	 */
	bool processMessage() {
		// メッセージを取得する。
		BOOL result = GetMessage(&msg_, null, 0, 0);
		if(result > 0) {
			// 通常のメッセージ。ディスパッチ。
			TranslateMessage(&msg_); 
			DispatchMessage(&msg_);
		} else if(result < 0) {
			// エラー発生。
			throwLastError();
		} else {
			// WM_QUIT。終了する。
			exitCode_ = msg_.wParam;
			return false;
		}
		
		// 処理は続く。
		return true;
	}
	
protected:
	
	/** アイドル処理。
	 *
	 *	Windowsメッセージが来なくなった時に繰り返し呼び出される。
	 *
	 *	Params:
	 *		cnt = アイドリング時にアイドル処理が呼び出された回数。
	 *
	 *	Returns:
	 *		trueならばメッセージがあるかどうか確かめ、なければアイドル処理を再実行する。
	 *		falseならばアイドル処理の続行を止め、次のメッセージを待つ。
	 */
	bool onIdle(uint cnt) {
		// そのうち何かするかもしれない。
		return false;
	}
	
private:
	
	/// 唯一のアプリケーションオブジェクト。
	static Application application_;
	
	/// 以前のインスタンスハンドル。
	Handle preview_;
	
	/// コマンドライン。
	LPSTR command_;
	
	/// 表示フラグ。
	int show_;
	
	/// 処理中メッセージ。
	MSG msg_;
	
	/// アイドルカウンタ。
	uint idle_;
	
	/// 終了コード。
	int exitCode_ = -1;
}

/// メッセージボックス。
int messageBox(char[] msg, char[] title = "", int type = MB_OK) {
	return MessageBoxA(null, toTString(msg), toTString(title), type);
}

/// メッセージボックス。
int messageBoxTString(LPTSTR msg, LPTSTR title = null, int type = MB_OK) {
	return MessageBoxA(null, msg, title, type);
}

/** アプリケーション基本クラス。
 *
 *	アプリケーションを新たに作成する場合はこのクラスの派生クラスを作成する。
 *	型Tを指定して派生クラスを定義した場合、アプリケーション実行時に自動的にインスタンスが生成されるようになる。
 *
 *	機能を拡張したアプリケーションを定義したい場合、引数Tを取るテンプレートクラスとして定義すること。
 *  引数Tはそのまま基本クラスのApplicationBaseに渡す。
 *
 *	Params:
 *		T = アプリケーション派生クラスの型。派生時に自分自身のクラスを指定すること。
 */
class ApplicationBase(T) : Application {
	
	/** 静的コンストラクタ。
	 *
	 *	派生クラスのアプリケーションオブジェクトが生成される。
	 */
	static this()
	in {
		assert(application_ is null);
	} out {
		assert(application_ !is null);
	} body {
		application_ = new T();
	}
}

// アプリケーション初期化用の関数。
extern (C) {
	void gc_init();
	void gc_term();
	void _minit();
	void _moduleCtor();
	void _moduleDtor();
	void _moduleUnitTests();
}

/// WinMain関数。
extern(Windows)
int WinMain(HINSTANCE inst, HINSTANCE prev, LPSTR cmd, int show) {
	// GC初期化。
	gc_init();
	
	// モジュール初期化。
	_minit();
	
	// 終了コード。
	int result = -1;
	
	try {
		// モジュールコンストラクタ。
		_moduleCtor();
		
		// 単体テスト実行。
		debug _moduleUnitTests();
		
		assert(Application.instance !is null);
		result = Application.instance.main(inst, prev, cmd, show);
		
		// モジュールデストラクタ。
		_moduleDtor();
	} catch(ApiException e) {
		messageBox(e.toString(), "API error");
	} catch(Object o) {
		messageBox(o.toString(), "Unknown error");
	}
	
	// GC終了。
	gc_term();
	
	return result;
}
