﻿/**
 *	コンボボックス関連モジュール。
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module outland.dmajor.combobox;

import win32.windows;
import outland.dmajor.application;
import outland.dmajor.exception;
import outland.dmajor.shape;
import outland.dmajor.tstring;
import outland.dmajor.window;

/// コンボボックスコントロール。
class ComboBox : Window {
	
	/// ウィンドウクラス名。
	const TCHAR[] WINDOW_CLASS = "COMBOBOX";
	
	/// デフォルトのスタイル。
	const DWORD STYLE = CBS_AUTOHSCROLL | CBS_DROPDOWN | WS_VISIBLE;
	
	/// デフォルトの拡張スタイル。
	const DWORD STYLE_EX = 0;
	
	/// コンボボックスを生成する。
	static ComboBox create(Rect r, Window parent, uint id, DWORD style = STYLE, DWORD styleEx = STYLE_EX) {
		ComboBox c = new ComboBox();
		c.createWindow(r, parent, id, style, styleEx);
		return c;
	}
	
	/// サブクラス化する。
	static ComboBox subclassWindow(HWND wnd) {
		ComboBox c = new ComboBox();
		c.subclass(wnd);
		return c;
	}
	
public:
	
	/// 項目の追加。追加項目のインデックスが返る。
	size_t addString(char[] str) {
		return checkResult(sendMessage(CB_ADDSTRING, 0, cast(LPARAM) toTString(str)));
	}
	
	/// 項目の削除。
	void deleteString(size_t i) {checkResult(sendMessage(CB_DELETESTRING, i));}
	
	/// ディレクトリ項目の表示。
	size_t dir(uint attr, char[] path) {
		return checkResult(sendMessage(CB_DIR, attr, cast(LPARAM) toTString(path)));
	}
	
	/// 項目の検索。指定文字列の一部にでもマッチするものを検索する。
	int findString(char[] str, int start = -1) {
		return sendMessage(CB_FINDSTRING, start, cast(LPARAM) toTString(str));
	}
	
	/// 項目の検索。指定文字列全体にマッチするものを検索する。
	int findStringExact(char[] str, int start = -1) {
		return sendMessage(CB_FINDSTRINGEXACT, start, cast(LPARAM) toTString(str));
	}
	
	/// 項目数を返す。  
	size_t length() {return checkResult(sendMessage(CB_GETCOUNT));}
	
	/// 現在の選択項目。CB_ERRならば選択なし。
	int currentSelect() {return sendMessage(CB_GETCURSEL);}
	
	/// ditto
	int currentSelect(int i) {return sendMessage(CB_SETCURSEL, i);}
	 
	/// ドロップダウン時の領域をスクリーン座標で返す。
	Rect getDroppedControlRect() {
		Rect r;
		if(!sendMessage(CB_GETDROPPEDCONTROLRECT, 0, cast(LPARAM) r.ptr)) {
			throwLastError();
		}
		return r;
	}
	
	/// ドロップダウン状態。
	bool droppedState() {return sendMessage(CB_GETDROPPEDSTATE) != FALSE;}
	
	/// ditto
	void droppedState(bool b) {sendMessage(CB_SHOWDROPDOWN, b ? TRUE : FALSE);}
	
	/// ドロップダウンリストの幅。
	size_t droppedWidth() {return sendMessage(CB_GETDROPPEDWIDTH);}
	
	/// ditto
	void droppedWidth(size_t w) {sendMessage(CB_SETDROPPEDWIDTH, w);}
	
	/// エディットボックス内の選択範囲。
	void getEditSelection(out DWORD start, out DWORD end) {
		sendMessage(CB_GETEDITSEL, cast(WPARAM) &start, cast(WPARAM) &end);
	}
	
	/// ditto
	void setEditSelection(DWORD start, DWORD end) {
		sendMessage(CB_SETEDITSEL, start, end);
	}
	
	/// 拡張ユーザインターフェイス有無。
	bool extendedUserInterface() {return sendMessage(CB_GETEXTENDEDUI) != FALSE;}
	
	/// ditto
	void extendedUserInterface(bool b) {sendMessage(CB_SETEXTENDEDUI, b ? TRUE : FALSE);}
	
	/// リストボックスの幅。
	size_t horizontalExtent() {return sendMessage(CB_GETHORIZONTALEXTENT);}
	
	/// ditto
	void horizontalExtent(size_t w) {return sendMessage(CB_SETHORIZONTALEXTENT, w);}
	
	/// 追加データ。
	DWORD getItemData(size_t i) {return sendMessage(CB_GETITEMDATA, i);}
	
	/// ditto
	void setItemData(size_t i, DWORD data) {
		checkResult(sendMessage(CB_SETITEMDATA, i, data));
	}
	
	/// 項目の高さ。-1を指定した場合はエディット部分の高さ。オーナードローコントロールの場合は指定項目の高さ。
	size_t getItemHeight(size_t i = 0) {return sendMessage(CB_GETITEMHEIGHT, i);}
	
	/// ditto
	void setItemHeight(size_t i, size_t h) {checkResult(sendMessage(CB_SETITEMHEIGHT, i, h));}
	
	/// 項目部分の高さ。
	void setItemHeight(size_t h) {setItemHeight(0, h);}
	
	/// 指定項目の文字列を得る。バッファは自動的に拡張される。
	void getListBoxText(size_t i, inout TCHAR[] buf) {
		buf.length = getListBoxTextLength(i) + 1;
		checkResult(sendMessage(CB_GETLBTEXT, i, cast(LPARAM) buf.ptr));
	}
	
	/// 指定項目の文字数を得る。
	size_t getListBoxTextLength(size_t i) {return checkResult(sendMessage(CB_GETLBTEXTLEN, i));}
	
	/// ロケール。
	DWORD locale() {return sendMessage(CB_GETLOCALE);}
	
	/// ditto
	void locale(DWORD loc) {checkResult(sendMessage(CB_SETLOCALE, loc));}
	
	/// 最上位に表示されている項目。
	size_t topIndex() {return checkResult(sendMessage(CB_GETTOPINDEX));}
	
	/// ditto
	void topIndex(size_t i) {checkResult(sendMessage(CB_SETTOPINDEX, i));}
	
	/// 記憶領域を確保する。
	void initializeStorage(size_t items, size_t len) {
		checkResult(sendMessage(CB_INITSTORAGE, items, len));
	}
	
	/// 文字列を挿入する。
	void insertString(size_t i, char[] str) {
		checkResult(sendMessage(CB_INSERTSTRING, i, cast(LPARAM) toTString(str)));
	}
	
	/// 最大文字数を設定する。
	void limitText(size_t cnt) {sendMessage(CB_LIMITTEXT, cnt);}
	
	/// 内容を破棄する。
	void resetContent() {sendMessage(CB_RESETCONTENT);}
	
	/// 指定文字列の項目を選択する。検索失敗の場合はCB_ERRが返る。
	int selectString(char[] str, int start = -1) {
		return sendMessage(CB_SELECTSTRING, start, cast(LPARAM) toTString(str));
	}

	/// リストボックスが閉じられた時の処理。
	void onCloseUp(CommandDelegate dg) {onReflectCommand(CBN_CLOSEUP, dg);}
	
	/// ダブルクリック時の処理。
	void onDoubleClick(CommandDelegate dg) {onReflectCommand(CBN_DBLCLK, dg);}
	
	/// リストボックスが開かれた時の処理。
	void onDropDown(CommandDelegate dg) {onReflectCommand(CBN_DROPDOWN, dg);}
	
	/// エディットボックスの内容が変更された時の処理。画面更新後に通知される。
	void onEditChange(CommandDelegate dg) {onReflectCommand(CBN_EDITCHANGE, dg);}
	
	/// エディットボックスの内容が変更された時の処理。画面更新前に通知される。
	void onEditUpdate(CommandDelegate dg) {onReflectCommand(CBN_EDITUPDATE, dg);}
	
	/// メモリ不足時の処理。
	void onErrorSpace(CommandDelegate dg) {onReflectCommand(CBN_ERRSPACE, dg);}
	
	/// 選択項目が変更された時の処理。
	void onSelectChange(CommandDelegate dg) {onReflectCommand(CBN_SELCHANGE, dg);}
	
	/// 項目選択がキャンセルされた時の処理。
	void onSelectCancel(CommandDelegate dg) {onReflectCommand(CBN_SELENDCANCEL, dg);}
	
	/// 項目選択が実行された時の処理。
	void onSelectOk(CommandDelegate dg) {onReflectCommand(CBN_SELENDOK, dg);}
	
	// 不要？
	//CBN_KILLFOCUS  
	//CBN_SETFOCUS 

protected:
	
	/// ウィンドウ生成。
	void createWindow(Rect r, Window parent, uint id, DWORD style, DWORD styleEx) {
		HINSTANCE inst = (parent) ? parent.getInstanceHandle() : Application.instance.handle;
		super.createWindow(
			WINDOW_CLASS.ptr,
			"",
			style | WS_CHILD,
			styleEx,
			r,
			parent ? parent.handle : cast(HWND) null,
			cast(HMENU) id,
			inst,
			null);
	}
	
private:
	
	/// 戻り値のチェック。
	int checkResult(int result) {
		if(result == CB_ERR || result == CB_ERRSPACE) {
			throwLastError();
		}
		return result;
	}
}
