﻿/**
 *	共通ダイアログ関連モジュール。
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module outland.dmajor.commdlg;

import win32.windows;

import outland.dmajor.application;
import outland.dmajor.exception;
import outland.dmajor.trace;
import outland.dmajor.tstring;
import outland.dmajor.window;

/** 共通ダイアログエラー時の例外。
 *
 *	エラーコードを指定して生成する。エラーコードに応じたメッセージを取り出せる。
 */
class CommonDialgoException : ApiException {
	
	/// ラーコードを指定して生成する。
	this(DWORD code) {super(code);}
	
	/// 文字列に変換する。
	override char[] toString() {
		switch(code) {
 		case CDERR_DIALOGFAILURE: return "CDERR_DIALOGFAILURE";
		case CDERR_GENERALCODES: return "CDERR_GENERALCODES";
		case CDERR_STRUCTSIZE: return "CDERR_STRUCTSIZE";
		case CDERR_INITIALIZATION: return "CDERR_INITIALIZATION";
		case CDERR_NOTEMPLATE: return "CDERR_NOTEMPLATE";
		case CDERR_NOHINSTANCE: return "CDERR_NOHINSTANCE";
		case CDERR_LOADSTRFAILURE: return "CDERR_LOADSTRFAILURE";
		case CDERR_FINDRESFAILURE: return "CDERR_FINDRESFAILURE";
		case CDERR_LOADRESFAILURE: return "CDERR_LOADRESFAILURE";
		case CDERR_LOCKRESFAILURE: return "CDERR_LOCKRESFAILURE";
		case CDERR_MEMALLOCFAILURE: return "CDERR_MEMALLOCFAILURE";
		case CDERR_MEMLOCKFAILURE: return "CDERR_MEMLOCKFAILURE";
		case CDERR_NOHOOK: return "CDERR_NOHOOK";
		case CDERR_REGISTERMSGFAIL: return "CDERR_REGISTERMSGFAIL";
		case PDERR_PRINTERCODES: return "PDERR_PRINTERCODES";
		case PDERR_SETUPFAILURE: return "PDERR_SETUPFAILURE";
		case PDERR_PARSEFAILURE: return "PDERR_PARSEFAILURE";
		case PDERR_RETDEFFAILURE: return "PDERR_RETDEFFAILURE";
		case PDERR_LOADDRVFAILURE: return "PDERR_LOADDRVFAILURE";
		case PDERR_GETDEVMODEFAIL: return "PDERR_GETDEVMODEFAIL";
		case PDERR_INITFAILURE: return "PDERR_INITFAILURE";
		case PDERR_NODEVICES: return "PDERR_NODEVICES";
		case PDERR_NODEFAULTPRN: return "PDERR_NODEFAULTPRN";
		case PDERR_DNDMMISMATCH: return "PDERR_DNDMMISMATCH";
		case PDERR_CREATEICFAILURE: return "PDERR_CREATEICFAILURE";
		case PDERR_PRINTERNOTFOUND: return "PDERR_PRINTERNOTFOUND";
		case PDERR_DEFAULTDIFFERENT: return "PDERR_DEFAULTDIFFERENT";
		case CFERR_CHOOSEFONTCODES: return "CFERR_CHOOSEFONTCODES";
		case CFERR_NOFONTS: return "CFERR_NOFONTS";
		case CFERR_MAXLESSTHANMIN: return "CFERR_MAXLESSTHANMIN";
		case FNERR_FILENAMECODES: return "FNERR_FILENAMECODES";
		case FNERR_SUBCLASSFAILURE: return "FNERR_SUBCLASSFAILURE";
		case FNERR_INVALIDFILENAME: return "FNERR_INVALIDFILENAME";
		case FNERR_BUFFERTOOSMALL: return "FNERR_BUFFERTOOSMALL";
		case FRERR_FINDREPLACECODES: return "FRERR_FINDREPLACECODES";
		case FRERR_BUFFERLENGTHZERO: return "FRERR_BUFFERLENGTHZERO";
		case CCERR_CHOOSECOLORCODES: return "CCERR_CHOOSECOLORCODES";
		default:
			break;
		}
		return "Common dialog unknown error!";
	}
	
	/// Win文字列に変換する。
	override LPCTSTR toTString() {return outland.dmajor.tstring.toTString(toString());}
}

/// ファイルダイアログ。
class FileDialog : Window {
	
	/// デフォルトコンストラクタ。
	this(Window owner = null) {
		with(ofn_) {
			static if (WINVER >= 0x0500) {
				lStructSize = OPENFILENAME_SIZE_VERSION_400;
			} else {
				lStructSize = OPENFILENAME.sizeof;
			}
			
			filePath_[] = cast(TCHAR) '\0';
		
			lpstrFile = filePath_.ptr;
			nMaxFile = filePath_.length;
			
			if(owner) {
				hwndOwner = owner.handle;
				hInstance = owner.getInstanceHandle;
			} else {
				hInstance = Application.instance.handle;
			}
		}
	}
	
	/// ファイルを開くダイアログの表示。
	bool selectOpenFileName() {
		if(!.GetOpenFileName(&ofn_)) {
			if(DWORD err = .CommDlgExtendedError()) {
				throw new CommonDialgoException(err);
			}
			return false;
		}
		return true;
	}
	
	/// ファイルを保存ダイアログの表示。
	bool selectSaveFileName() {
		if(!.GetSaveFileName(&ofn_)) {
			if(DWORD err = .CommDlgExtendedError()) {
				throw new CommonDialgoException(err);
			}
			return false;
		}
		return true;
	}
	
	/// フィルタの追加。
	void addFilter(char[] caption, char[] pattern) {
		LPTSTR tcap = toTString(caption);
		LPTSTR tpat = toTString(pattern);
		
		if(filter_.length) {
			filter_.length = filter_.length - 1;
		}
		
		filter_ ~= tcap[0 .. lstrlen(tcap)];
		filter_ ~= '\0';
		filter_ ~= tpat[0 .. lstrlen(tpat)];
		filter_ ~= '\0';
		filter_ ~= '\0';
		
		ofn_.lpstrFilter = filter_.ptr;
	}
	
	/// フィルタインデックス。
	void filterIndex(size_t i) {ofn_.nFilterIndex = i;}
	
	/// ditto
	size_t filterIndex() {return ofn_.nFilterIndex;}
	
	/// カスタムリソース。
	void dialogTemplate(LPCTSTR name) {
		ofn_.lpTemplateName = name;
		ofn_.Flags |= OFN_ENABLETEMPLATE;
	}
	
	/// ditto
	void dialogTemplate(UINT id) {dialogTemplate(MAKEINTRESOURCE(id));}
	
	/// 初期ディレクトリ。
	void initialDirectory(char[] path) {ofn_.lpstrInitialDir = toTString(path);}
	
	/// ダイアログタイトル。
	void title(char[] title) {ofn_.lpstrTitle = toTString(title);}
	
	/// デフォルトの拡張子。
	void defaultExtent(char[] ext) {ofn_.lpstrDefExt = toTString(ext);}
	
	/// 複数選択を許すかどうか。
	void allowMultiSelect(bool b) {changeFlags(b, OFN_ALLOWMULTISELECT);}
	
	/// ファイル作成を問い合わせるか。
	void createPrompt(bool b) {changeFlags(b, OFN_CREATEPROMPT);}
	
	/// エクスプローラ型かどうか。
	void explorer(bool b) {changeFlags(b, OFN_EXPLORER);}
	
	/// 別の拡張子を許すかどうか。
	void extensionDifferent(bool b) {changeFlags(b, OFN_EXTENSIONDIFFERENT);}
	
	/// 存在するファイルだけを対象にするかどうか。
	void fileMustExist(bool b) {changeFlags(b, OFN_FILEMUSTEXIST);}
	
	/// 読み込み専用を非表示にするかどうか。
	void hideReadOnly(bool b) {changeFlags(b, OFN_HIDEREADONLY);}
	
	/// 選択されたファイルパスの取得。
	char[] filePath() {return fromTString(filePath_.ptr);}
	
	/// 選択されたファイルメイの取得。
	char[] fileName() {return fromTString(filePath_[ofn_.nFileOffset .. $].ptr);}
	
	/// 選択されたファイル拡張子の取得。
	char[] fileExtent() {
		return fromTString(filePath_[ofn_.nFileExtension .. $].ptr);
	}
	
private:
	
	/// フラグの変更。
	void changeFlags(bool b, DWORD flag) {
		if(b) {
			ofn_.Flags |= flag;
		} else {
			ofn_.Flags &= ~flag;
		}
	}
	
	/// 設定構造体。
	OPENFILENAME ofn_;
	
	/// フィルタ配列。
	TCHAR[] filter_;
	
	/// ファイルパス。
	TCHAR[MAX_PATH + 1] filePath_;
}
