﻿/**
 *	エディットボックス関連モジュール。
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module outland.dmajor.edit;

import std.bind;

import win32.windows;
import outland.dmajor.application;
import outland.dmajor.shape;
import outland.dmajor.window;

/// エディットボックスコントロール。
class Edit : Window {
	
	/// ウィンドウクラス名。
	const TCHAR[] WINDOW_CLASS = "EDIT";
	
	/// デフォルトのスタイル。
	const DWORD DEFAULT_STYLE = ES_MULTILINE | ES_WANTRETURN | ES_AUTOVSCROLL | WS_VISIBLE;
	
	/// デフォルトの拡張スタイル。
	const DWORD DEFAULT_STYLE_EX = 0;
	
	/// エディットボックスを生成する。
	static Edit create(char[] text, Rect r, Window parent, uint id, DWORD style = DEFAULT_STYLE, DWORD styleEx = DEFAULT_STYLE_EX) {
		auto e = new Edit();
		e.createWindow(text, r, parent, id, style, styleEx);
		return e;
	}
	
	/// サブクラス化する。
	static Edit subclassWindow(HWND wnd) {
		auto e = new Edit();
		e.subclass(wnd);
		return e;
	}
	
public:
	
	/// コピー。
	void copy() {sendMessage(WM_COPY);}
	
	/// ペースト。
	void paste() {sendMessage(WM_PASTE);}
	
	/// 切り取り。
	void cut() {sendMessage(WM_CUT);}
	
	/// 削除。
	void clear() {sendMessage(WM_CLEAR);}
	
	/// Undoできるかどうか返す。
	bool canUndo() {return sendMessage(EM_CANUNDO) != FALSE;}
	
	/// 指定座標の文字インデックス・行インデックスを返す。
	Tuple!(WORD, WORD) charFromPosition(Point pos) {
		auto result = sendMessage(EM_CHARFROMPOS, 0, MAKELPARAM(pos.x, pos.y));
		return tuple(LOWORD(result), HIWORD(result));
	}

	/// Undoバッファをクリアする。
	void emptyUndoBuffer() {sendMessage(EM_EMPTYUNDOBUFFER);}
	
	/// 折り返し位置にソフト改行(CRCRLF)を挿入するか設定する。
	void formatLines(bool b) {sendMessage(EM_FMTLINES, b ? TRUE : FALSE);}

	/// 先頭に表示されている列または行のインデックスを返す。
	size_t getFirstVisible() {return sendMessage(EM_GETFIRSTVISIBLELINE);}
	
	/// テキストメモリのハンドル。
	HLOCAL textHandle() {return cast(HLOCAL) sendMessage(EM_GETHANDLE);}
	
	/// ditto
	void textHandle(HLOCAL m) {sendMessage(EM_SETHANDLE, cast(WPARAM) m);}
	
	/// 最大文字数。
	size_t limitText() {return sendMessage(EM_GETLIMITTEXT);}
	
	/// ditto
	void limitText(size_t limit) {sendMessage(EM_SETLIMITTEXT, limit);}
	
	/// 行を得る。
	size_t getLine(size_t line, TCHAR[] buf) {return sendMessage(EM_GETLINE, line, cast(LPARAM) buf.ptr);}
	
	/// 行数を得る。
	size_t lineCount() {return sendMessage(EM_GETLINECOUNT);}
	
	/// マージン。
	Size margins() {
		auto result = sendMessage(EM_GETMARGINS);
		return Size(LOWORD(result), HIWORD(result));
	}
	
	/// ditto
	void margins(Size s) {
		sendMessage(EM_SETMARGINS, EC_LEFTMARGIN | EC_RIGHTMARGIN, MAKELONG(s.cx, s.cy));
	}
	
	/// フォントに合わせたマージンに変更する。
	void setFontMargin() {sendMessage(EM_SETMARGINS, EC_LEFTMARGIN | EC_RIGHTMARGIN | EC_USEFONTINFO);}
	
	/// マージンを変更する。
	void setMargin(DWORD flag, Size s) {sendMessage(EM_SETMARGINS, flag, MAKELONG(s.cx, s.cy));}
	
	/// 変更フラグ。
	bool modified() {return sendMessage(EM_GETMODIFY) != FALSE;}
	
	/// ditto
	void modified(bool b) {sendMessage(EM_SETMODIFY, b);}
	
	/// パスワード用マスク文字。
	UINT passwordCharacter() {return sendMessage(EM_GETPASSWORDCHAR);}
	
	/// ditto
	void passwordCharacter(UINT ch) {sendMessage(EM_SETPASSWORDCHAR, ch);}
	
	/// 表示領域。
	Rect textRect() {
		Rect r;
		sendMessage(EM_GETRECT, 0, cast(LPARAM) r.ptr);
		return r;
	}
	
	/// ditto
	void textRect(Rect r) {sendMessage(EM_SETRECT, 0, cast(LPARAM) r.ptr);}
	
	/// 再描画無しでテキスト領域を設定する。
	void setTextRectNonPaint(Rect r) {sendMessage(EM_SETRECTNP, 0, cast(LPARAM) r.ptr);}
	
	/// 選択範囲。
	void getSelection(out DWORD start, out DWORD end) {
		sendMessage(EM_GETSEL, cast(WPARAM) &start, cast(LPARAM) &end);
	}
	
	/// ditto
	void setSelection(DWORD start, DWORD end) {sendMessage(EM_SETSEL, start, end);}
	
	/// スクロールバーの位置を返す。
	size_t scrollThumb() {return sendMessage(EM_GETTHUMB);}
	
	/// 単語境界を検知するための関数。
	EDITWORDBREAKPROC wordBreakProcedure() {return cast(EDITWORDBREAKPROC) sendMessage(EM_GETWORDBREAKPROC);}
	
	/// ditto
	void wordBreakProcedure(EDITWORDBREAKPROC proc) {sendMessage(EM_SETWORDBREAKPROC, 0, cast(LPARAM) proc);}
	
	/// ユーザ入力可能文字数を設定する。
	void setInputLimit(size_t limit) {sendMessage(EM_LIMITTEXT, limit);}
	
	/// 指定文字インデックスの行を返す。-1を指定した場合は現在行を返す。
	size_t lineFromCharacter(size_t index) {return sendMessage(EM_LINEFROMCHAR, index);}

	/// 指定行の開始文字インデックスを返す。-1を指定した場合は現在行の開始インデックスを返す。
	size_t characterFromLine(size_t line) {return sendMessage(EM_LINEINDEX, line);}
	
	/// 指定行の長さを返す。
	size_t lineLength(size_t line) {return sendMessage(EM_LINELENGTH, characterFromLine(line));}
	
	/// スクロールを行う。
	void lineScroll(size_t chars, size_t lines) {sendMessage(EM_LINESCROLL, chars, lines);}

	/// 文字の座標を返す。
	Point positionFromCharacter(size_t index) {
		auto result = sendMessage(EM_POSFROMCHAR, index);
		return Point(LOWORD(result), HIWORD(result));
	}
	
	/// 選択範囲の文字列を置換する。
	void replaceSelection(LPCTSTR str, bool undo = true) {
		sendMessage(EM_REPLACESEL, undo ? TRUE : FALSE, cast(LPARAM) str);
	}
	
	/// カレットが画面に表示できる位置までスクロールする。
	void scrollCaret() {sendMessage(EM_SCROLLCARET);}
	
	/// 表示専用にする。
	void readOnly(bool b) {sendMessage(EM_SETREADONLY, b ? TRUE : FALSE);}
	
	/// タブストップを設定する。
	void setTabStops(DWORD[] tabs) {sendMessage(EM_SETTABSTOPS, tabs.length, cast(LPARAM) tabs.ptr);}
	
	/// Undoを行う。
	void undo() {sendMessage(EM_UNDO);}
	
	// const EM_SCROLL=181;	不要？
	
	/// 内容変更時の処理。画面更新後に通知される。
	void onChange(CommandDelegate d) {onReflectCommand(EN_CHANGE, d);}
	
	/// メモリ不足時の処理。
	void onErrorSpace(CommandDelegate d) {onReflectCommand(EN_ERRSPACE, d);}
	
	/// 水平スクロール時の処理。
	void onScrollEditH(CommandDelegate d) {onReflectCommand(EN_HSCROLL, d);}
	
	/// 最大文字数入力時の処理。
	void onMaxText(CommandDelegate d) {onReflectCommand(EN_MAXTEXT, d);}
	
	/// 内容更新時の処理。画面更新前に通知される。
	void onUpdate(CommandDelegate d) {onReflectCommand(EN_UPDATE, d);}
	
	/// 垂直スクロール時の処理。
	void onScrollEditV(CommandDelegate d) {onReflectCommand(EN_VSCROLL, d);}
	
	// 以下は不要?
	// EN_KILLFOCUS
	// EN_SETFOCUS 
	
protected:
	
	/// ウィンドウ生成。
	void createWindow(char[] text, Rect r, Window parent, uint id, DWORD style, DWORD styleEx) {
		auto inst = (parent) ? parent.getInstanceHandle() : Application.instance.handle;
		super.createWindow(
			WINDOW_CLASS.ptr,
			text,
			style | WS_CHILD,
			styleEx,
			r,
			parent ? parent.handle : cast(HWND) null,
			cast(HMENU) id,
			inst,
			null);
	}
}
