﻿/**
 *	APIエラー関連のモジュール。
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module outland.dmajor.exception;

import std.utf;

import outland.dmajor.tstring;

import win32.windows;

/** APIエラー時の例外。
 *
 *	エラーコードを指定して生成する。エラーコードに応じたシステムメッセージを取り出せる。
 */
class ApiException : Exception {
	
	/// ラーコードを指定して生成する。
	this(DWORD code) {
		super("win32 api error!");
		code_ = code;
	}
	
	/// エラーコードを返す。
	DWORD code() {return code_;}
	
	/// 文字列に変換する。
	char[] toString() {return fromTString(toTString());}
	
	/// Win文字列に変換する。
	LPCTSTR toTString() {
		// システムメッセージの取得。
		LPTSTR buf;
		FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
			null, code_, LANG_USER_DEFAULT, cast(LPTSTR) &buf, 0, null);
		scope(exit) LocalFree(cast(HANDLE) buf);
		return buf[0 .. lstrlen(buf)].dup.ptr;
	}
	
private:
	
	/// エラーコード。
	DWORD code_;
}

/** APIエラーで例外を生成する。
 *
 *	Params:
 *		code	= エラーコード。
 *	Throws:
 *		ApiException	必ず投げられる。
 */
void throwError(DWORD code) {throw new ApiException(code);}

/** 最後のAPIエラーで例外を生成する。
 *
 *	Throws:
 *		ApiException	必ず投げられる。
 */
void throwLastError() {throwError(GetLastError());}

/** APIの戻り値をチェックする。
 *
 *	Params:
 *		result = APIの戻り値。
 *	Throws:
 *		ApiException	resultがFALSEだった場合に投げられる。
 */
void checkApi(BOOL result) {
	if(!result) {
		throwLastError();
	}
}

/** APIの戻り値をチェックする。
 *
 *	Params:
 *		result = APIの戻り値。
 *	Throws:
 *		ApiException	resultがnullだった場合に投げられる。
 */
void checkApi(void* result) {
	if(!result) {
		throwLastError();
	}
}

/** 最後のエラーコードをチェックする。
 *
 *	Throws:
 *		ApiException	エラーが発生していた場合に投げられる。
 */
void checkLastError() {
	auto code = GetLastError();
	if(code != ERROR_SUCCESS) {
		throwError(code);
	}
}
