﻿/**
 *	ハンドル管理関連のモジュール。
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module outland.dmajor.handle;

import win32.windows;

import outland.dmajor.trace;

/// NULLハンドル。
const HANDLE NULL = cast(HANDLE) null;

/**	ハンドルラッパ。
 *
 *	Windowsのハンドルをラップするクラスの基本クラス。
 *
 *	Params:
 *		T = ハンドルの型。
 */
abstract class HandleWrapper(T) {
	
	/// ハンドルの型。
	alias T Handle;
	
	/// ハンドルデストラクタの型。
	alias void delegate(Handle) Destructor;
	
	/// ハンドルを指定して生成する。
	this(T h, Destructor dest) {
		handle_ = h;
		destructor_ = dest;
		HandleDatabase.instance.register(h, this);
	}
	
	/// オブジェクト破棄時にハンドルを削除する。
	~this() {
		if(handle_) {
			close();
		}
	}
	
	/// ハンドルを返す。
	Handle handle() {return handle_;}
	
	/// デストラクタを返す。
	Destructor destructor() {return destructor_;}
	
	/// ハンドルを閉じる。
	void close() {
		if(destructor_) {
			destructor_(handle_);
		}
		
		HandleDatabase.instance.unregister(handle_);
		handle_ = cast(Handle) null;
		destructor_ = null;
	}
	
	/// オブジェクトとハンドルを切り離す。
	Handle detach() {
		HandleDatabase.instance.unregister(handle_);
		Handle h = handle;
		handle_ = cast(Handle) null;
		destructor_ = null;
		return h;
	}
	
private:
	
	/// ハンドル。
	Handle handle_;
	
	/// ハンドル破棄関数。
	Destructor destructor_;
}

/**	ハンドルデータベース。
 *
 *	Windowsのオブジェクトに対応するDのオブジェクトを登録しておき、
 *	ハンドルが破棄されるまでGCされないようにする。
 */
class HandleDatabase {
	
	/// 唯一のインスタンスを得る。
	static HandleDatabase instance() {return instance_;}
	
	/// ハンドルとオブジェクトを登録する。
	void register(HANDLE h, Object o)
	in {
		assert(h !is null);
		assert(o !is null);
	} body {
		synchronized(this) {
			database_[h] = o;
		}
	}
	
	/// ハンドルを登録解除する。
	void unregister(HANDLE h)
	in {
		assert(h !is null);
	} body {
		synchronized(this) {
			database_.remove(h);
		}
	}
	
	/** ハンドルからオブジェクトを得る。
	 *
	 *	Params:
	 *		h = 割り当てられているオブジェクトを検索するハンドル。
	 *
	 *	Returns:
	 *		ハンドルを保持しているオブジェクト。存在しなければnull。
	 */
	Object opIndex(HANDLE h) {
		synchronized(this) {
			if(Object* po = h in database_) {
				return *po;
			} else {
				return null;
			}
		}
	}
	
	/// rehashで最適化を行う。
	void rehash() {
		synchronized(this) {
			database_.rehash;
		}
	}
	
private:
	
	/// デフォルトコンストラクタ。
	this() {}
	
	/// デストラクタ。
	~this() {}
	
	/// 静的コンストラクタ。
	static this() {instance_ = new HandleDatabase;}
	
	/// 唯一のインスタンス。
	static HandleDatabase instance_;
	
	/// ハンドルとオブジェクトの連想配列。
	Object[HANDLE] database_;
}
