﻿/**
 *	レイアウト関連クラス。
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module outland.dmajor.layout;

import win32.windows;

import outland.dmajor.exception;
import outland.dmajor.shape;
import outland.dmajor.window;
 
/// ウィンドウ遅延移動クラス。
class DeferWindowPosition {
	
	/// 移動予定のウィンドウ数を指定して生成する。
	this(size_t n = 0) {
		hdwp_ = BeginDeferWindowPos(n);
		checkApi(hdwp_);
	}
	
	/// 移動終了。
	~this() {
		auto result = EndDeferWindowPos(hdwp_);
		assert(result);
	}
	
	/// ウィンドウを移動する。
	void setPosition(Window w, Window after, Point pos, Size size, UINT flag)
	in {
		assert(w !is null);
	} body {
		auto h = DeferWindowPos(hdwp_, w.handle, after ? after.handle : cast(HWND) null, pos.x, pos.y, size.cx, size.cy, flag);
		checkApi(h);
		hdwp_ = h;
	}
	
	/// ditto
	void move(Window w, Point pos) {
		setPosition(w, null, pos, Size(0, 0), SWP_NOSIZE | SWP_NOZORDER);
	}
	
	/// ditto
	void resize(Window w, Size size) {
		setPosition(w, null, Point(0, 0), size, SWP_NOMOVE | SWP_NOZORDER);
	}
	
	/// ditto
	void move(Window w, Rect rect) {
		setPosition(w, null, rect.position, rect.size, SWP_NOMOVE | SWP_NOZORDER);
	}
	
	/// ditto
	void insertAfter(Window w, Window after) {
		setPosition(w, after, Point(0, 0), Size(0, 0), SWP_NOMOVE | SWP_NOSIZE);
	}
	
	/// ditto
	void top(Window w)
	in {
		assert(w !is null);
	} body {
		auto h = DeferWindowPos(hdwp_, w.handle, HWND_TOP, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE);
		checkApi(h);
		hdwp_ = h;
	}
	
	/// ditto
	void bottom(Window w)
	in {
		assert(w !is null);
	} body {
		auto h = DeferWindowPos(hdwp_, w.handle, HWND_BOTTOM, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE);
		checkApi(h);
		hdwp_ = h;
	}
	
	/// ditto
	void topMost(Window w)
	in {
		assert(w !is null);
	} body {
		auto h = DeferWindowPos(hdwp_, w.handle, HWND_TOPMOST, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE);
		checkApi(h);
		hdwp_ = h;
	}
	
	/// ditto
	void noTopMost(Window w)
	in {
		assert(w !is null);
	} body {
		auto h = DeferWindowPos(hdwp_, w.handle, HWND_NOTOPMOST, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE);
		checkApi(h);
		hdwp_ = h;
	}
	
	/// 表示する。
	void show(Window w)
	in {
		assert(w !is null);
	} body {
		setPosition(w, null, Point(0, 0), Size(0, 0), SWP_NOMOVE | SWP_NOZORDER | SWP_SHOWWINDOW);
	}
	
	/// 非表示にする。
	void hide(Window w)
	in {
		assert(w !is null);
	} body {
		setPosition(w, null, Point(0, 0), Size(0, 0), SWP_NOMOVE | SWP_NOZORDER | SWP_HIDEWINDOW);
	}
	
	/// ハンドル。
	private HDWP hdwp_;
}

/// レイアウト項目の基本インターフェイス。
interface LayoutItem {
	
	/// レイアウトする。必ず引数の矩形内にレイアウトしなければならない。
	void doLayout(DeferWindowPosition def, Rect rect);
	
	/// 現在のサイズを得る。
	Size size();
	
	/// 最小サイズを得る。
	Size minSize();
}

/// サイズを一致させるレイアウト。
class FitLayout : LayoutItem {
	
	/// フラグ型。
	typedef int Flag;
	
	enum : Flag {
		BOTH,
		WIDTH,
		HEIGHT,
	}
	
	/// フラグと項目を指定して生成する。
	this(Flag f, LayoutItem item)
	in {
		assert(item !is null);
	} body {
		flag_ = f;
		item_ = item;
	}
	
	/// レイアウトする。
	void doLayout(DeferWindowPosition def, Rect rect)
	in {
		assert(def !is null);
	} body {
		if(flag_ == BOTH) {
			item_.doLayout(def, rect); 
		} else if(flag_ & WIDTH) {
			item_.doLayout(def, Rect(rect.position, Size(rect.cx, item_.size.cy)));
		} else if(flag_ & HEIGHT) {
			item_.doLayout(def, Rect(rect.position, Size(item_.size.cx, rect.cy)));
		}
	}
	
	/// 現在のサイズを得る。
	Size size() {
		return item_.size();
	}
	
	/// 最小サイズを得る。
	Size minSize() {
		return item_.minSize();
	}

private:
	
	/// フラグ。
	Flag flag_;
	
	/// 項目。
	LayoutItem item_;
}

/// 整列させるレイアウト。
class AlignLayout : LayoutItem {
	
	/// フラグ型。
	typedef int Align;
	
	enum : Align {
		LEFT	= 0b00_01,
		CENTER	= 0b00_10,
		RIGHT	= 0b00_11,
		
		TOP		= 0b01_00,
		MIDDLE	= 0b10_00,
		BOTTOM	= 0b11_00,
		
		H_MASK	= 0b00_11,
		V_MASK	= 0b11_00,
	}
	
	/// フラグと項目を指定して生成する。
	this(Align a, LayoutItem item)
	in {
		assert(item !is null);
	} body {
		align_ = a;
		item_ = item;
		size_ = item.size;
	}
	
	/// レイアウトする。
	void doLayout(DeferWindowPosition def, Rect rect)
	in {
		assert(def !is null);
	} body {
		size_ = rect.size;
		auto s = item_.size;
		auto ms = item_.minSize;
		
		// 最小サイズより現在のサイズが小さかった場合、
		// 最小サイズでの移動を試みる。
		if(s.cx < ms.cx) {
			s.cx = ms.cx;
		}
		if(s.cy < ms.cy) {
			s.cy = ms.cy;
		}
		
		// 推奨サイズより指定サイズが小さい場合、指定サイズに合わせる。
		if(size_.cx < s.cx) {
			s.cx = size_.cx;
		}
		if(size_.cx < s.cx) {
			s.cx = size_.cx;
		}
		
		switch(H_MASK & align_) {
		case CENTER:
			rect.move((size_.cx - s.cx) / 2, 0);
			rect.inflate(-(size_.cx - s.cx) / 2, 0);
			break;
		case RIGHT:
			rect.move(size_.cx - s.cx, 0);
			rect.inflate(-(size_.cx - s.cx), 0);
			break;
		default:
		case LEFT:
			break;
		}
		
		switch(V_MASK & align_) {
		case MIDDLE:
			rect.move(0, (size_.cy - s.cy) / 2);
			rect.inflate(0, -(size_.cy - s.cy) / 2);
			break;
		case BOTTOM:
			rect.move(0, size_.cy - s.cy);
			rect.inflate(0, -(size_.cy - s.cy));
			break;
		default:
		case TOP:
			break;
		}
		
		item_.doLayout(def, rect);
	}
	
	/// 現在のサイズを得る。
	Size size() {
		return size_;
	}
	
	/// 最小サイズを得る。
	Size minSize() {
		return item_.minSize();
	}

private:
	
	/// サイズ。
	Size size_;
	
	/// 整列位置。
	Align align_;
	
	/// 項目。
	LayoutItem item_;
}

/// レイアウトグループ基本クラス。
abstract class LayoutGroup : LayoutItem {
	
	/// 項目を追加する。
	LayoutGroup add(LayoutItem item) {
		items_ ~= item;
		return this;
	}
	
	/// 巡回する。
	int opApply(int delegate(inout LayoutItem) dg) {
		foreach(i; items_) {
			if(int result = dg(i)) {
				return result;
			}
		}
		return 0;
	}
	
	/// 水平配置。
	LayoutGroup horizontal() {
		LayoutGroup g = new HorizontalBoxLayout();
		add(g);
		return g;
	}
	
	/// 垂直配置。
	LayoutGroup vertical() {
		LayoutGroup g = new VerticallBoxLayout();
		add(g);
		return g;
	}
	
	/// サイズ一致項目の追加。
	LayoutGroup fitItem(FitLayout.Flag f, LayoutItem i) {
		add(new FitLayout(f, i));
		return this;
	}
	
	/// 整列項目の追加。
	LayoutGroup alignItem(AlignLayout.Align a, LayoutItem i) {
		add(new AlignLayout(a, i));
		return this;
	}
	
	/// 子アイテム数。
	size_t length() {return items_.length;}
	
	/// 子アイテム。
	private LayoutItem[] items_;
}

/// 水平配置レイアウト。
class HorizontalBoxLayout : LayoutGroup {
	
	/// レイアウトを行う。
	void doLayout(DeferWindowPosition def, Rect rect)
	in {
		assert(def !is null);
	} body {
		int dx = rect.cx / items_.length;
		auto r = Rect(rect.position, Size(dx, rect.cy));
		foreach(i; this) {
			i.doLayout(def, r);
			r.move(dx, 0);
		}
	}
	
	/// 現在のサイズを得る。
	Size size() {
		Size result;
		foreach(i; this) {
			auto s = i.size;
			result.cx = result.cx + s.cx;
			if(result.cy < s.cy) {
				result.cy = s.cy;
			}
		}
		return result;
	}
	
	/// 最小サイズを得る。
	Size minSize() {
		Size result;
		foreach(i; this) {
			auto s = i.minSize;
			result.cx = result.cx + s.cx;
			if(result.cy < s.cy) {
				result.cy = s.cy;
			}
		}
		return result;
	}
}

/// 垂直配置レイアウト。
class VerticallBoxLayout : LayoutGroup {
	
	/// レイアウトを行う。
	void doLayout(DeferWindowPosition def, Rect rect)
	in {
		assert(def !is null);
	} body {
		int dy = rect.cy / items_.length;
		auto r = Rect(rect.position, Size(rect.cx, dy));
		foreach(i; items_) {
			i.doLayout(def, r);
			r.move(0, dy);
		}
	}
	
	/// 現在のサイズを得る。
	Size size() {
		Size result;
		foreach(i; this) {
			auto s = i.size;
			result.cy = result.cy + s.cy;
			if(result.cx < s.cx) {
				result.cx = s.cx;
			}
		}
		return result;
	}
	
	/// 最小サイズを得る。
	Size minSize() {
		Size result;
		foreach(i; this) {
			auto s = i.minSize;
			result.cy = result.cy + s.cy;
			if(result.cx < s.cx) {
				result.cx = s.cx;
			}
		}
		return result;
	}
}
