﻿/**
 *	リストボックス関連モジュール。
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module outland.dmajor.listbox;

import win32.windows;

import outland.dmajor.application;
import outland.dmajor.exception;
import outland.dmajor.shape;
import outland.dmajor.tstring;
import outland.dmajor.window;

/// リストボックスコントロール。
class ListBox : Window {
	
	/// ウィンドウクラス名。
	const TCHAR[] WINDOW_CLASS = "LISTBOX";
	
	/// デフォルトのスタイル。
	const DWORD STYLE = WS_VISIBLE;
	
	/// デフォルトの拡張スタイル。
	const DWORD STYLE_EX = 0;
	
	/// コンボボックスを生成する。
	static ListBox create(Rect r, Window parent, uint id, DWORD style = STYLE, DWORD styleEx = STYLE_EX) {
		auto l = new ListBox();
		l.createWindow(r, parent, id, style, styleEx);
		return l;
	}
	
	/// サブクラス化する。
	static ListBox subclassWindow(HWND wnd) {
		auto l = new ListBox();
		l.subclass(wnd);
		return l;
	}
	
public:
	
	/// ファイル名の追加。
	size_t addFile(char[] path) {
		return checkResult(sendMessage(LB_ADDFILE, 0, cast(LPARAM) toTString(path)));
	}
	
	/// 項目の追加。追加項目のインデックスが返る。
	size_t addString(char[] str) {
		return checkResult(sendMessage(LB_ADDSTRING, 0, cast(LPARAM) toTString(str)));
	}
	
	/// 項目の削除。
	void deleteString(size_t i) {checkResult(sendMessage(LB_DELETESTRING, i));}
	
	/// ディレクトリ項目の表示。
	size_t dir(uint attr, char[] path) {
		return checkResult(sendMessage(LB_DIR, attr, cast(LPARAM) toTString(path)));
	}
	
	/// 項目の検索。指定文字列の一部にでもマッチするものを検索する。
	int findString(char[] str, int start = -1) {
		return sendMessage(LB_FINDSTRING, start, cast(LPARAM) toTString(str));
	}
	
	/// 項目の検索。指定文字列全体にマッチするものを検索する。
	int findStringExact(char[] str, int start = -1) {
		return sendMessage(LB_FINDSTRINGEXACT, start, cast(LPARAM) toTString(str));
	}
	
	/// アンカーインデックス。
	int anchorIndex() {return sendMessage(LB_GETANCHORINDEX);}
	
	/// ditto
	void anchorIndex(int i) {checkResult(sendMessage(LB_SETANCHORINDEX, i));}
	
	/// カレットのあるインデックス。
	int caretIndex() {return sendMessage(LB_GETCARETINDEX);}
	
	/// ditto
	void caretIndex(int i, bool scroll = true) {
		checkResult(sendMessage(LB_SETCARETINDEX, i, MAKELPARAM(scroll, 0)));
	}
	
	/// 項目数。  
	size_t length() {return checkResult(sendMessage(LB_GETCOUNT));}
	
	/// ditto
	void length(size_t n) {checkResult(sendMessage(LB_SETCOUNT, n));}
	
	/// 現在の選択項目。LB_ERRならば選択なし。
	int currentSelect() {return sendMessage(LB_GETCURSEL);}
	
	/// ditto
	int currentSelect(int i) {return sendMessage(LB_SETCURSEL, i);}
	
	/// リストボックスの幅。
	size_t horizontalExtent() {return sendMessage(LB_GETHORIZONTALEXTENT);}
	
	/// ditto
	void horizontalExtent(size_t w) {return sendMessage(LB_SETHORIZONTALEXTENT, w);}
	
	/// 追加データ。
	DWORD getItemData(size_t i) {return sendMessage(LB_GETITEMDATA, i);}
	
	/// ditto
	void setItemData(size_t i, DWORD data) {
		checkResult(sendMessage(LB_SETITEMDATA, i, data));
	}
	
	/// 項目の高さ。-1を指定した場合はエディット部分の高さ。オーナードローコントロールの場合は指定項目の高さ。
	size_t getItemHeight(size_t i = 0) {return sendMessage(LB_GETITEMHEIGHT, i);}
	
	/// ditto
	void setItemHeight(size_t i, size_t h) {checkResult(sendMessage(LB_SETITEMHEIGHT, i, MAKELPARAM(h, 0)));}
	
	/// 項目部分の高さ。
	void setItemHeight(size_t h) {setItemHeight(0, h);}
	
	/// 項目の矩形を得る。
	Rect getItemRect(size_t i) {
		Rect r;
		checkResult(sendMessage(LB_GETITEMRECT, i, cast(LPARAM) r.ptr));
		return r;
	}
	
	/// ロケール。
	DWORD locale() {return sendMessage(CB_GETLOCALE);}
	
	/// ditto
	void locale(DWORD loc) {checkResult(sendMessage(CB_SETLOCALE, loc));}
	
	/// 選択項目。
	bool isSelect(size_t i) {return sendMessage(LB_GETSEL, i) > 0;}
	
	/// ditto
	void setSelect(int i, bool b) {sendMessage(LB_SETSEL, b ? TRUE : FALSE, i);}
	
	/// 選択項目数。
	size_t selectCount() {return checkResult(sendMessage(LB_GETSELCOUNT));}
	
	/// 選択項目を得る。
	size_t getSelectItems(INT[] buf) {
		return checkResult(sendMessage(LB_GETSELITEMS, buf.length, cast(LPARAM) buf.ptr));
	}
	
	/// 指定項目の文字列を得る。バッファは自動的に拡張される。
	void getText(size_t i, inout TCHAR[] buf) {
		buf.length = getTextLength(i) + 1;
		checkResult(sendMessage(LB_GETTEXT, i, cast(LPARAM) buf.ptr));
	}
	
	/// 指定項目の文字数を得る。
	size_t getTextLength(size_t i) {return checkResult(sendMessage(LB_GETTEXTLEN, i));}
	
	/// 最上位に表示されている項目。
	size_t topIndex() {return checkResult(sendMessage(LB_GETTOPINDEX));}
	
	/// ditto
	void topIndex(size_t i) {checkResult(sendMessage(LB_SETTOPINDEX, i));}
	
	/// 記憶領域を確保する。
	void initializeStorage(size_t items, size_t len) {
		checkResult(sendMessage(LB_INITSTORAGE, items, len));
	}
	
	/// 文字列を挿入する。
	void insertString(size_t i, char[] str) {
		checkResult(sendMessage(LB_INSERTSTRING, i, cast(LPARAM) toTString(str)));
	}
	
	/// 座標から項目を得る。
	int itemFromPoint(Point pos) {
		return sendMessage(LB_ITEMFROMPOINT, 0, MAKELPARAM(pos.x, pos.y));
	}
	
	/// 内容を破棄する。
	void resetContent() {sendMessage(LB_RESETCONTENT);}
	
	/// 指定文字列の項目を選択する。検索失敗の場合はCB_ERRが返る。
	int selectString(char[] str, int start = -1) {
		return sendMessage(LB_SELECTSTRING, start, cast(LPARAM) toTString(str));
	}
	
	/// 指定範囲の項目を選択する。
	void selectRange(size_t begin ,size_t end, bool b) {
		checkResult(sendMessage(LB_SELITEMRANGE, b ? TRUE : FALSE, MAKELPARAM(begin, end)));
	}
	
	/// 列幅。
	void columnWidth(size_t w) {checkResult(sendMessage(LB_SETCOLUMNWIDTH, w));}
	
	/// タブストップの設定。
	void setTabStops(INT[] tabs) {
		checkResult(sendMessage(LB_SETTABSTOPS, tabs.length, cast(LPARAM) tabs.ptr));
	}
	
	/// ダブルクリック時の処理。
	void onDoubleClick(CommandDelegate dg) {onReflectCommand(LBN_DBLCLK, dg);}
	
	/// メモリ不足時の処理。
	void onErrorSpace(CommandDelegate dg) {onReflectCommand(LBN_ERRSPACE, dg);}
	
	/// 選択項目が変更された時の処理。
	void onSelectChange(CommandDelegate dg) {onReflectCommand(LBN_SELCHANGE, dg);}
	
	/// 項目選択がキャンセルされた時の処理。
	void onSelectCancel(CommandDelegate dg) {onReflectCommand(LBN_SELCANCEL, dg);}
	
	// 不要？
	//LBN_KILLFOCUS  
	//CBN_SETFOCUS 

protected:
	
	/// ウィンドウ生成。
	void createWindow(Rect r, Window parent, uint id, DWORD style, DWORD styleEx) {
		auto inst = (parent) ? parent.getInstanceHandle() : Application.instance.handle;
		super.createWindow(
			WINDOW_CLASS.ptr,
			"",
			style | WS_CHILD,
			styleEx,
			r,
			parent ? parent.handle : cast(HWND) null,
			cast(HMENU) id,
			inst,
			null);
	}
	
private:
	
	/// 戻り値のチェック。
	int checkResult(int result) {
		if(result == LB_ERR || result == LB_ERRSPACE) {
			throwLastError();
		}
		return result;
	}
}
