﻿/**
 *	図形クラス。
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module outland.dmajor.shape;

import win32.windows;

import std.string;

import outland.tl.algorithm;

/// 座標。
struct Point {
	
	/// コンストラクタ。
	static Point opCall(int x, int y) {
		Point p;
		p.x = x;
		p.y = y;
		return p;
	}
	
	/// ditto
	static Point opCall(LPPOINT pos) {
		Point p;
		p.point_ = *pos;
		return p;
	}
	
	/// ネイティブなポインタを返す。
	LPPOINT ptr() {return &point_;}
	
	/// X座標。
	int x() {return point_.x;}
	
	/// ditto
	void x(int val) {point_.x = val;}
	
	/// Y座標。
	int y() {return point_.y;}
	
	/// ditto
	void y(int val) {point_.y = val;}
	
	/// 移動。
	void move(int x, int y) {
		point_.x += x;
		point_.y += y;
	}
	
	/// 加算。
	Point* opAddAssign(Point pos) {
		move(pos.x, pos.y);
		return this;
	}
	
	/// ditto
	Point opAdd(Point pos) {
		Point p = *this;
		p += pos;
		return p;
	}
	
	/// 減算。
	Point* opSubAssign(Point pos) {
		move(-pos.x, -pos.y);
		return this;
	}
	
	/// ditto
	Point opSub(Point pos) {
		Point p = *this;
		p -= pos;
		return p;
	}
	
	/// 乗算。
	Point* opMulAssign(int val) {
		point_.x *= val;
		point_.y *= val;
		return this;
	}
	
	/// ditto
	Point opMul(int val) {
		Point p = *this;
		p *= val;
		return p;
	}
	
	/// 除算。
	Point* opDivAssign(int val) {
		point_.x /= val;
		point_.y /= val;
		return this;
	}
	
	/// ditto
	Point opDiv(int val) {
		Point p = *this;
		p /= val;
		return p;
	}
	
	/// 文字列に変換する。
	char[] toString() {return format("[%d,%d]", point_.x, point_.y);}
	
	/// ネイティブ構造体。
	private POINT point_;
}

/// サイズ。
struct Size {
	
	/// コンストラクタ。
	static Size opCall(int cx, int cy) {
		Size s;
		s.cx = cx;
		s.cy = cy;
		return s;
	}
	
	/// ditto
	static Size opCall(LPSIZE size) {
		Size s;
		s.size_ = *size;
		return s;
	}
	
	/// ネイティブなポインタを返す。
	LPSIZE ptr() {return &size_;}
	
	/// 幅。
	int cx() {return size_.cx;}
	
	/// ditto
	void cx(int val) {size_.cx = val;}
	
	/// 高さ。
	int cy() {return size_.cy;}
	
	/// ditto
	void cy(int val) {size_.cy = val;}
	
	/// 加算。
	Size* opAddAssign(Size size) {
		size_.cx += size.cx;
		size_.cy += size.cy;
		return this;
	}
	
	/// ditto
	Size opAdd(Size size) {
		Size s = *this;
		s += size;
		return s;
	}
	
	/// 減算。
	Size* opSubAssign(Size size) {
		size_.cx -= size.cx;
		size_.cy -= size.cy;
		return this;
	}
	
	/// ditto
	Size opSub(Size size) {
		Size s = *this;
		s -= size;
		return s;
	}
	
	/// 乗算。
	Size* opMulAssign(int val) {
		size_.cx *= val;
		size_.cy *= val;
		return this;
	}
	
	/// ditto
	Size opMul(int val) {
		Size p = *this;
		p *= val;
		return p;
	}
	
	/// 除算。
	Size* opDivAssign(int val) {
		size_.cx /= val;
		size_.cy /= val;
		return this;
	}
	
	/// ditto
	Size opDiv(int val) {
		Size p = *this;
		p /= val;
		return p;
	}
	
	/// 文字列に変換する。
	char[] toString() {return format("[%d-%d]", size_.cx, size_.cy);}
	
	/// ネイティブ構造体。
	private SIZE size_;
}

/// 矩形。
struct Rect {
	
	/// コンストラクタ。
	static Rect opCall(Point pos, Size s) {
		Rect r;
		r.position = pos;
		r.size = s;
		return r;
	}
	
	/// ditto
	static Rect opCall(LPRECT rect)
	in {
		assert(rect !is null);
	} body {
		Rect r;
		r.rect_ = *rect;
		return r;
	}
	
	/// ditto
	static Rect opCall(Point tl, Point br) {
		Rect r;
		r.rect_.top = tl.y;
		r.rect_.left = tl.x;
		r.rect_.bottom = br.y;
		r.rect_.right = br.x;
		r.normalize();
		return r;
	}
	
	/// ネイティブなポインタを返す。
	LPRECT ptr() {return &rect_;}
	
	/// 移動。
	void move(int x, int y) {
		rect_.left += x;
		rect_.top += y;
		rect_.right += x;
		rect_.bottom += y;
	}
	
	/// 拡張。
	void inflate(int cx, int cy) {
		rect_.right += cx;
		rect_.bottom += cy;
	}
	
	/// 位置。
	Point position() {return Point(rect_.left, rect_.top);}
	
	/// ditto
	void position(Point pos) {
		Size s = size;
		rect_.left = pos.x;
		rect_.top = pos.y;
		rect_.right = pos.x + s.cx;
		rect_.bottom = pos.y + s.cy;
	}
	
	/// サイズ。
	Size size() {return Size(rect_.right - rect_.left, rect_.bottom - rect_.top);}
	
	/// ditto
	void size(Size size) {
		rect_.right = rect_.left + size.cx;
		rect_.bottom = rect_.top + size.cy;
	}
	
	/// X座標。
	int left() {return rect_.left;}
	
	/// ditto
	alias left x;
	
	/// Y座標。
	int top() {return rect_.top;}
	
	/// ditto
	alias top y;
	
	/// 右端。
	int right() {return rect_.right;}
	
	/// 下端。
	int bottom() {return rect_.bottom;}
	
	/// 幅。
	int cx() {return size.cx;}
	
	/// ditto
	void cx(int val) {size = Size(val, cy);}
	
	/// 高さ。
	int cy() {return size.cy;}
	
	/// ditto
	void cy(int val) {size = Size(cx, val);}
	
	/// 左上。
	Point topLeft() {return Point(left, top);}
	
	/// 左下。
	Point bottomLeft() {return Point(left, bottom);}
	
	/// 右下。
	Point bottomRight() {return Point(right, bottom);}
	
	/// 右上。
	Point topRight() {return Point(right, top);}
	
	/// 空かどうか返す。
	bool isEmpty() {return IsRectEmpty(&rect_) != FALSE;}
	
	/// 空にする。
	void empty() {SetRectEmpty(&rect_);}
	
	/// 座標が矩形内部にあるか返す。
	bool opIn(Point pos) {return PtInRect(&rect_, *pos.ptr) != FALSE;}
	
	/// 交差部分だけを取り出す。
	Rect* opAndAssign(Rect rect) {
		RECT old = rect_;
		IntersectRect(&rect_, &old, rect.ptr);
		return this;
	}
	
	/// ditto
	Rect opAnd(Rect rect) {
		Rect result;
		IntersectRect(result.ptr, ptr, rect.ptr);
		return result;
	}
	
	/// 包容部分を生成する。
	Rect* opOrAssign(Rect rect) {
		RECT old = rect_;
		UnionRect(&rect_, &old, rect.ptr);
		return this;
	}
	
	/// ditto
	Rect opOr(Rect rect) {
		Rect result;
		UnionRect(result.ptr, ptr, rect.ptr);
		return result;
	}
	
	/// 矩形を切り取る。
	Rect* opSubAssign(Rect rect) {
		RECT old = rect_;
		SubtractRect(&rect_, &old, rect.ptr);
		return this;
	}
	
	/// ditto
	Rect opSub(Rect rect) {
		Rect result;
		SubtractRect(result.ptr, ptr, rect.ptr);
		return result;
	}
	
	/// 正規化を行う。
	Rect* normalize() {
		if(rect_.left > rect_.right) {
			swap(rect_.right, rect_.left);
		}
		if(rect_.top > rect_.bottom) {
			swap(rect_.top, rect_.bottom);
		}
		return this;
	}
	
	/// 文字列に変換する。
	char[] toString() {return format("(%s%s)", position, size);}
	
	/// ネイティブ構造体。
	private RECT rect_;
}
