﻿/**
 *	サンプルアプリケーション。
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module sample;

import std.file;
import std.string;

import win32.windows;

import outland.dmajor.all;
import outland.tl.all;

class MainFrame : Frame {
	
	/// タイトルに表示するアプリケーション名。
	const char[] TITLE = "test";
	
	/// ウィンドウタイトルのフォーマット。
	const char[] WINDOW_TITLE_FORMAT = "%s - %s";
	
	/// デフォルトの拡張子。
	const char[] DEFAULT_EXTENT = "txt";
	
	/// テキストファイルのフィルタ名。
	const char[] TEXT_FILTER_TITLE = "テキスト(*.txt)";
	
	/// テキストファイルのフィルタパターン。
	const char[] TEXT_FILTER_PATTERN = "*.txt";
	
	/// 全てのファイルのフィルタ名。
	const char[] ALL_FILTER_TITLE = "全てのファイル(*.*)";
	
	/// 全てのファイルのフィルタパターン。
	const char[] ALL_FILTER_PATTERN = "*.*";
	
	/// 保存確認ダイアログのメッセージ。
	const char[] CONFIRM_SAVE_MSG = "内容が変更されています。保存しますか？";
	
	/// コントロールID。
	enum : UINT {
		IDC_EDIT = Window.ID_LAST,
		
		ID_FILE_NEW,		/// 新規作成。
		ID_FILE_OPEN,		/// 開く。
		ID_FILE_SAVE,		/// 保存。
		ID_FILE_SAVE_AS,	/// 名前を付けて保存。
		ID_EXIT,			/// 終了。
		
		ID_UNDO,			/// 元に戻す。
		ID_CUT,				/// 切り取り。
		ID_COPY,			/// コピー。
		ID_PASTE,			/// 貼り付け。
		ID_DELETE,			/// 削除。
		ID_ALL_SELECT,		/// 全て選択。
		
		ID_HELP_INDEX,		/// ヘルプ。
		ID_VERSION,			/// バージョン情報。
		
		ID_LAST,			/// 最後のID。
	}
	
	/// エディットボックスのスタイル。
	const UINT EDIT_STYLE = ES_MULTILINE | ES_WANTRETURN | ES_AUTOVSCROLL | WS_VISIBLE | WS_VSCROLL | WS_HSCROLL;
	
	/// メインフレーム生成。
	static MainFrame create() {
		MainFrame frame = new MainFrame();
		Rect rect = Rect(Point(CW_USEDEFAULT, CW_USEDEFAULT), Size(CW_USEDEFAULT, CW_USEDEFAULT));
		frame.createWindow(TITLE, rect, null, frame.menu_);
		return frame;
	}
	
private:
	
	this() {
		// 破棄時に終了。
		onDestroy = (Window w) {Application.instance.quit(0);};
		
		// 生成時の処理。
		onCreate = &handleOnCreate;
		
		// システムコマンド処理。
		onSystemCommand = &handleOnSystemCommand;
		
		// メニュー生成。
		with(menu_ = Menu.create) {
			// ファイルメニュー。
			addSubMenu("ファイル(&F)", (Menu m) {
				m.add(ID_FILE_NEW, "新規作成(&N)");
				m.add(ID_FILE_OPEN, "開く(&O)");
				m.add(ID_FILE_SAVE, "上書き保存(&S)");
				m.add(ID_FILE_SAVE_AS, "名前を付けて保存(&A)");
				m.addSeparator;
				m.add(ID_EXIT, "終了(&X)");
			});
			
			// 編集メニュー。
			addSubMenu("編集(&E)", (Menu m) {
				m.add(ID_UNDO, "元に戻す(&U)");
				m.addSeparator;
				m.add(ID_CUT, "切り取り(&T)");
				m.add(ID_COPY, "コピー(&C)");
				m.add(ID_PASTE, "貼り付け(&P)");
				m.add(ID_DELETE, "削除(&D)");
				m.add(ID_ALL_SELECT, "全て選択(&A)");
			});
			
			// ヘルプメニュー。
			addSubMenu("ヘルプ(&H)", (Menu m) {
				m.add(ID_HELP_INDEX, "目次(&O)");
				m.addSeparator;
				m.add(ID_VERSION, "バージョン情報(&A)");
			});
		}
		
		onMenu(ID_EXIT, &handleOnMenuExit);
		
		onMenu(ID_FILE_NEW, &handleOnMenuNew);
		onMenu(ID_FILE_OPEN, &handleOnMenuOpen);
		onMenu(ID_FILE_SAVE, &handleOnMenuSave);
		onMenu(ID_FILE_SAVE_AS, &handleOnMenuSaveAs);
		onMenu(ID_UNDO, &handleOnMenuUndo);
		onMenu(ID_CUT, &handleOnMenuCut);
		onMenu(ID_COPY, &handleOnMenuCopy);
		onMenu(ID_PASTE, &handleOnMenuPaste);
		onMenu(ID_DELETE, &handleOnMenuDelete);
		onMenu(ID_ALL_SELECT, &handleOnMenuAllSelect);
		
		// ファイルダイアログ生成。
		with(fileDlg_ = new FileDialog(this)) {
			defaultExtent(DEFAULT_EXTENT);
			addFilter(TEXT_FILTER_TITLE, TEXT_FILTER_PATTERN);
			addFilter(ALL_FILTER_TITLE, ALL_FILTER_PATTERN);
		}
	}
	
	/// ウィンドウ生成時の処理。
	bool handleOnCreate(Window w, LPCREATESTRUCT cs) {
		// エディットボックス生成。
		edit_ = Edit.create("", clientRect, this, IDC_EDIT, EDIT_STYLE);
		
		this.layout = new FitLayout(FitLayout.HEIGHT, edit_);
		
		// 新規作成処理。
		handleOnMenuNew();
		
		return true;
	}
	
	/// 新規作成。
	void handleOnMenuNew() {
		if(!confirmModified()) {
			return;
		}
		
		// エディットボックスの内容のクリア。
		resetContent();
		
		filePath_ = "";
		
		// ウィンドウタイトルの変更。
		text = format(WINDOW_TITLE_FORMAT, filePath_, TITLE);
	}
	
	/// ファイルを開く。
	void handleOnMenuOpen() {		
		if(!fileDlg_.selectOpenFileName()) {
			return;
		}
		
		// ファイル読み込み。
		LPCTSTR txt = cast(LPCTSTR) std.file.read(fileDlg_.filePath).ptr;
		scope(exit) delete txt;
		resetContent(txt);
		setFilePath(fileDlg_.filePath);
	}
	
	/// 名前を付けてファイルを保存する。
	void handleOnMenuSaveAs() {saveAs();}
	
	/// ファイルを保存する。
	void handleOnMenuSave() {
		// ファイル未指定の場合は名前をつけて保存。
		if(filePath_.length == 0) {
			saveAs();
		} else {
			saveFile(filePath_);
		}
	}
	
	/// 終了。
	void handleOnMenuExit() {destroy();}
	
	/// 切り取り。
	void handleOnMenuCut() {edit_.cut();}
	
	/// コピー。
	void handleOnMenuCopy() {edit_.copy();}
	
	/// 貼り付け。
	void handleOnMenuPaste() {edit_.paste();}
	
	/// 削除。
	void handleOnMenuDelete() {edit_.clear();}
	
	/// 元に戻す。
	void handleOnMenuUndo() {edit_.undo();}
	
	/// 全選択。
	void handleOnMenuAllSelect() {edit_.setSelection(0, -1);}
	
	/// 終了時の保存確認。
	void handleOnSystemCommand(Window w, UINT id, LPARAM lp) {
		if(id == SC_CLOSE) {
			if(!confirmModified()) {
				return;
			}
		}
		
		// デフォルト処理。
		callDefaultProcedure();
	}
	
	/// ファイル名をつけて保存する。
	bool saveAs() {
		if(!fileDlg_.selectSaveFileName()) {
			return false;
		}
		filePath_ = fileDlg_.filePath;
		saveFile(filePath_);
		setFilePath(fileDlg_.filePath);
		return true;
	}
	
	/// 指定パスにテキストを保存する。
	void saveFile(char[] path) {
		TCHAR[] buf;
		scope(exit) delete buf;
		
		buf.length = edit_.textLength + 1;
		edit_.getText(buf);
		
		std.file.write(path, buf);
		edit_.modified = false;
	}
	
	/// 内容のリセット。
	void resetContent(LPCTSTR txt = null) {
		with(edit_) {
			setSelection(0, -1);
			replaceSelection(txt, false);
			setSelection(0, 0);
			scrollCaret();
			modified = false;
			emptyUndoBuffer();
		}
	}
	
	/// ファイルパスの変更。
	void setFilePath(char[] path) {
		filePath_ = path;
		text = format(WINDOW_TITLE_FORMAT, filePath_, TITLE);
	}
		
	/// 保存確認。キャンセル時はfalse。
	bool confirmModified() {
		if(edit_.modified) {
			switch(messageBox(CONFIRM_SAVE_MSG, TITLE, MB_YESNOCANCEL | MB_ICONQUESTION)) {
			case IDYES:
				return saveAs();
			case IDCANCEL:
				return false;
			case IDNO:
			default:
				break;
			}
		}
		
		return true;
	}
	
	/// エディットボックス。
	Edit edit_;
	
	/// メニュー。
	Menu menu_;
	
	/// ファイルダイアログ。
	FileDialog fileDlg_;
	
	/// ファイルパス。
	char[] filePath_;
}

/// サンプルアプリケーション。
class SampleApplication : ApplicationBase!(SampleApplication) {
	/// 初期化。
	override void initialize() {MainFrame.create();}
}
