﻿/**
 *	アプリケーションモジュール。
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module outland.poet.application;

import std.stdio;

import derelict.sdl.sdl;
import derelict.sdl.image;
import derelict.sdl.mixer;
import derelict.sdl.ttf;

import outland.poet.error;
import outland.poet.font;
import outland.poet.quadtree;
import outland.poet.shape;
import outland.poet.sprite;
import outland.poet.surface;
import outland.tl.list;

/// アプリケーションクラス。簡易Singleton。
class Application {
	
	/// 1フレーム辺りの時間。
	const Uint32 MILLSECONDS_PER_FRAME = 30;
	
	/// FPS更新間隔。
	const Uint32 FPS_UPDATE_ELAPSE = 1000;
	
	/// 初期化フラグ。
	typedef Uint32 InitFlag;
	
	/// ditto
	enum : InitFlag {
		/// タイマの使用。
		USE_TIMER = cast(InitFlag) SDL_INIT_TIMER,
		
		/// オーディオの使用。
		USE_AUDIO = cast(InitFlag) SDL_INIT_AUDIO,
		
		/// ビデオの使用。
		USE_VIDEO = cast(InitFlag) SDL_INIT_VIDEO,
		
		/// CDの使用。
		USE_CDROM = cast(InitFlag) SDL_INIT_CDROM,
		
		/// ジョイスティックの使用。
		USE_JOYSTICK = cast(InitFlag) SDL_INIT_JOYSTICK,
		
		/// シグナルを捕まえない。
		NOPARACHUTE = cast(InitFlag) SDL_INIT_NOPARACHUTE,
		
		/// イベントを別スレッドで取得。
		EVENTTHREAD = cast(InitFlag) SDL_INIT_EVENTTHREAD,
		
		/// 全て。
		EVERYTHING = cast(InitFlag) SDL_INIT_EVERYTHING,
	}
	
	/// イベントコード。
	enum Event {
		NONE,	/// 定義されていないイベント。
		FRAME,	/// フレーム更新イベント。
		FPS,	/// FPS計測イベント。
	}
	
	/// 唯一のインスタンスを返す。
	static Application instance()
	in {
		assert(app_ !is null);
	} body {
		return app_;
	}
	
	/// アプリケーション終了。
	~this() {
		app_ = null;
		// 初期化されていたものを終了。
		if(TTF_WasInit()) TTF_Quit();
		if(wasInit(EVERYTHING)) SDL_Quit();
	}
	
	/// 指定サブシステムが初期化されているか返す。
	static bool wasInit(InitFlag f) {return SDL_WasInit(f) != 0;}
	
	/// アプリケーション終了処理中かどうか返す。
	static bool isFinalizing() {return app_ is null;}
	
	/// 現在時刻を返す。
	static Uint32 tick() {return SDL_GetTicks();}
	
	/// FPSを返す。
	double fps() {
		uint now = tick();
		double result = cast(double)(frameCount_) * 1000.0 / cast(double)(now - beginTick_);
		frameCount_ = 0;
		beginTick_ = now;
		return result;
	}
	
private:
	
	/**	メインループの実行。
	 *	Returns:
	 *		終了コード。
	 */
	int run() {
		// フレーム進行用タイマの起動。
		SDL_TimerID frameTimer = SDL_AddTimer(MILLSECONDS_PER_FRAME, &onTimer!(Event.FRAME, MILLSECONDS_PER_FRAME), null);
		if(!frameTimer) throw new SdlException;
		scope(exit) SDL_RemoveTimer(frameTimer);
		
		// FPS更新用タイマの起動。
		SDL_TimerID fpsTimer = SDL_AddTimer(FPS_UPDATE_ELAPSE, &onTimer!(Event.FPS, FPS_UPDATE_ELAPSE), null);
		if(!fpsTimer) throw new SdlException;
		scope(exit) SDL_RemoveTimer(fpsTimer);
		
		// FPS初期化。
		fps();
		
		for(SDL_Event event; SDL_WaitEvent(&event) >= 0;) {
			if(event.type == SDL_QUIT) break;
			
			// フレームイベントだった場合はフレーム描画。
			if(event.type == SDL_USEREVENT && event.user.code == Event.FRAME) {
				if(sprite_ !is null) sprite_.draw(Point(0, 0), screen_);
				screen_.flip();
				++frameCount_;
			}
		}
		return 0;
	}
	
	/** イベント処理。
	 *	Params:
	 *		e	= イベント。
	 *	Returns:
	 *		アプリケーションを続行するかどうか。
	 */
	bool onEvent(SDL_Event e) {
		switch(e.type) {
		case SDL_QUIT:
			return false;
		default:
			break;
		}
		return true;
	}
	
	/// モジュール外からは呼び出せない。
	this(InitFlag flags = EVERYTHING)
	in {
		assert(app_ is null);
	} body {
		app_ = this;
		
		// Derelictのロード。
		DerelictSDL.load();
		DerelictSDLImage.load();
		DerelictSDLMixer.load();
		DerelictSDLttf.load();

		// 初期化。
		if(SDL_Init(flags) != 0) throw new SdlException();
		if(TTF_Init() != 0) throw new SdlException();
		
		screen_ = new Screen(Size(640, 480), 0, Screen.HARDWARE);
	}
	
	/// タイマ関数。
	static extern(C) Uint32 onTimer(Event EV, Uint32 ELAPSE)(Uint32 interval, void* param) {
		SDL_Event event;
		event.type = SDL_USEREVENT;
		event.user.code = EV;
		SDL_PushEvent(&event);
		return ELAPSE;
	}
	
	/// 唯一のインスタンス。
	static Application app_;
	
	/// 画面。
	Screen screen_;
	
	/// スプライト。
	Sprite sprite_;
	
	/// FPS計測開始時間。
	uint beginTick_;
	
	/// 描画したフレーム数。
	uint frameCount_;
}

/// メイン関数。
int main(char[][] args) {
	// アプリケーション生成。
	scope app = new Application();
	
	auto root = new SpriteGroup();
	auto suf1 = Surface.load("test.bmp");
	auto suf2 = Surface.load("sup.jpg");
	auto suf3 = Surface.load("test.png");
	
	auto sp1 = new SurfaceSprite(suf1);
	auto sp2 = new SurfaceSprite(suf2);
	auto sp3 = new SurfaceSprite(suf3);
	
	root.position = Point(50, 20);
	sp1.position = Point(10, 10);
	
	root.add(sp1);
	root.add(sp2);
	root.add(sp3);
	
	auto font = new Font("C:\\WINDOWS\\Fonts\\msmincho.ttc", 128);
	auto suf4 = font.renderBlended("ためし", color(255, 0, 0));
	auto sp4 = new SurfaceSprite(suf4);
	root.add(sp4);
	
	app.sprite_ = root;
	
	return app.run();
}
