﻿/**
 *	4分木関連モジュール。
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module outland.poet.quadtree;

import outland.tl.algorithm;
import outland.tl.pool;

import outland.poet.shape;

/// 4分木。
class QuadTree {
	
	/// デフォルトコンストラクタ。
	this() {this(Rect.init);}
	
	/// コンストラクタ。
	this(Rect r) {area_ = r;}
	
	/// デストラクタ。
	~this() {begin_.clear();}
	
	/// 領域。
	Rect area() {return area_;}
	
	/// ditto
	void area(Rect r) {area_ = r;}
	
	/// 領域の追加。
	void add(Rect target) {begin_.add(area_, target);}
	
	/// 領域の巡回。
	void walk(void delegate(Rect) dg) {begin_.walk(area_, dg);}
	
	/// 全葉の解放。
	void clear() {begin_.clear();}
	
private:

	/// ノード。
	struct Node {
		
		/// 領域の追加。
		void add(Rect area, Rect target) {
			eachSubRect(area, (size_t i, Rect r) {addLeaf(i, r, r & target);});
		}
		
		/// 領域の巡回。
		void walk(Rect area, void delegate(Rect) dg) {
			// 全部埋まっている場合。
			if(findIf(leafs_, (Node* p) {return p !is &FULL_LEAF;}) == leafs_.length) {
				dg(area);
				return;
			}
			
			// 各領域を巡回。
			eachSubRect(area, (size_t i, Rect r) {
				if(leafs_[i] is &FULL_LEAF) {
					dg(r);
				} else if(leafs_[i] !is null) {
					leafs_[i].walk(r, dg);
				}
			});
		}
		
		/// 全葉の解放。
		void clear() {
			for(size_t i = 0; i < leafs_.length; ++i) {
				deallocateLeaf(i);
			}
		}
		
	private:
		
		/// 領域を分割して関数に渡す。
		void eachSubRect(Rect area, void delegate(size_t, Rect) dg) {
			Rect r = area;
			auto lw = r.w / 2;
			auto th = r.h / 2;
			
			// 左上。
			r.w = lw;
			r.h = th;
			dg(0, r);
			
			// 右上。
			r = area;
			r.x = r.x + lw;
			r.w = r.w - lw;
			r.h = th;
			dg(1, r);
			
			// 左下。
			r = area;
			r.y = r.y + th;
			r.w = lw;
			r.h = r.h - th;
			dg(2, r);
			
			// 右下。
			r = area;
			r.x = r.x + lw;
			r.y = r.y + th;
			r.w = r.w - lw;
			r.h = r.h - th;
			dg(3, r);
		}
		
		/// 範囲追加。
		void addLeaf(size_t i, Rect area, Rect target)
		in {
			assert(i < leafs_.length);
		} body {
			// 既に埋まっているかtargetが空ならば終了。
			if(leafs_[i] is &FULL_LEAF || target.isEmpty) {
				return;
			}
			
			if(area == target) {
				deallocateLeaf(i);
				leafs_[i] = &FULL_LEAF;
			} else {
				if(leafs_[i] is null) {
					leafs_[i] = pool_.allocate();
				}
				leafs_[i].add(area, target);
			}
		}
		
		/// 葉の解放。
		void deallocateLeaf(size_t i)
		in {
			assert(i < leafs_.length);
		} body {
			if(leafs_[i] !is &FULL_LEAF && leafs_[i] !is null) {
				leafs_[i].clear();
				pool_.deallocate(leafs_[i]);
			}
			leafs_[i] = null;
		}
		
		/// 葉。
		Node* leafs_[4];
	}

	/// 全て埋まっている場合の葉。
	static Node FULL_LEAF;

	/// オブジェクトプール。
	static StructPool!(Node) pool_;
	
	/// オブジェクトプールの初期化。
	static this() {pool_ = new StructPool!(Node);}
	
	/// 開始ノード。
	Node begin_;
	
	/// 領域。
	Rect area_;
}

unittest {
	auto tree = new QuadTree(Rect(Point(0, 0), Size(640, 480)));
	
	Rect[] result;
	Rect[] expect;
	
	void testClear() {
		tree.clear();
		result.length = 0;
		expect.length = 0;
	}
	
	Rect target = Rect(Point(0, 0), Size(640, 480));
	tree.add(target);
	tree.walk((Rect r) {result ~= r;});
	
	expect ~= Rect(Point(0, 0), Size(640, 480));
	assert(result == expect);
	
	testClear();
	
	target = Rect(Point(0, 0), Size(1, 1));
	tree.add(target);
	tree.walk((Rect r) {result ~= r;});
	
	expect ~= Rect(Point(0, 0), Size(1, 1));
	assert(result == expect);
	
	testClear();
	
	target = Rect(Point(319, 239), Size(2, 2));
	tree.add(target);
	tree.walk((Rect r) {result ~= r;});
	
	expect ~= Rect(Point(319, 239), Size(1, 1));
	expect ~= Rect(Point(320, 239), Size(1, 1));
	expect ~= Rect(Point(319, 240), Size(1, 1));
	expect ~= Rect(Point(320, 240), Size(1, 1));
	assert(result == expect);
	
	testClear();
	
	target = Rect(Point(0, 0), Size(0, 0));
	tree.add(target);
	tree.walk((Rect r) {result ~= r;});
	
	assert(result == expect);
}
