﻿/**
 *	サーフェース関連モジュール。
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module outland.poet.surface;

import std.string;
import std.stdio;

import derelict.sdl.video;
import derelict.sdl.image;
import derelict.sdl.types;

import outland.poet.application;
import outland.poet.error;
import outland.poet.shape;

/// 色構造体。
alias SDL_Color Color;

/// 色の生成。
Color color(Uint8 r, Uint8 g, Uint8 b) {
	Color col;
	col.r = r;
	col.g = g;
	col.b = b;
	return col;
}

/// パレット。
struct Palette {
	
	static assert(Palette.sizeof == SDL_Palette.sizeof);
	
	/// 色を指定して生成する。
	static Palette opCall(Color[] colors) {
		Palette p;
		p.palette_.ncolors = colors.length;
		p.palette_.colors = colors.dup.ptr;
		return p;
	}
	
	/// 色を得る。
	Color opIndex(size_t i) {return palette_.colors[i];}
	
	/// 色を設定する。
	Color opIndexAssign(Color c, size_t i) {
		palette_.colors[i] = c;
		return c;
	}
	
	/// 長さを返す。
	size_t length() {return palette_.ncolors;}
	
	/// 長さを設定する。
	void length(size_t i) {
		Color[] cls = palette_.colors[0 .. palette_.ncolors];
		cls.length = i;
		palette_.ncolors = i;
		palette_.colors = cls.ptr;
	}
	
	/// ポインタを返す。
	SDL_Palette* ptr() {return &palette_;}
	
	/// パレット。
	private SDL_Palette palette_;
}

/** サーフェイス。
 *	Attention:
 *		アプリケーション終了時までに破棄が行われなかった場合、終了時のファイナライザでアクセス違反が発生する。
 */
class Surface {
	
	alias uint Flag;
	enum : Flag {
		SOFTWARE = SDL_SWSURFACE,
		HARDWARE = SDL_HWSURFACE,
		COLOR_KEY = SDL_SRCCOLORKEY,
		ALPHA = SDL_SRCALPHA,
	}
	
	/// ピクセルフォーマット。
	struct Format {
		int depth;
		uint rmask;
		uint gmask;
		uint bmask;
		uint amask;
	}
	
	/// 定義済みピクセルフォーマット。
	static const {
		
		Format PALETTE1 = {1};
		Format PALETTE2 = {2};
		Format PALETTE4 = {4};
		Format PALETTE8 = {8};
		
		Format RGB555 = {16};
		Format RBG555 = {16};
		Format GRB555 = {16};
		Format GBR555 = {16};
		Format BGR555 = {16};
		Format BRG555 = {16};
		
		Format RGB565 = {16};
		Format RBG565 = {16};
		Format GRB565 = {16};
		Format GBR565 = {16};
		Format BGR565 = {16};
		Format BRG565 = {16};
		
		Format RGB888 = {24};
		Format RBG888 = {24};
		Format GRB888 = {24};
		Format GBR888 = {24};
		Format BGR888 = {24};
		Format BRG888 = {24};
		
		Format RGBA888 = {32};
		Format RBGA888 = {32};
		Format GRBA888 = {32};
		Format GBRA888 = {32};
		Format BGRA888 = {32};
		Format BRGA888 = {32};
	}
	
	/** 画像ファイルのロード。
	 *	Params:
	 *		fn	= ファイル名。
	 *	Throws:
	 *		SdlException	エラー発生時に投げられる。
	 */
	static Surface load(char[] fn) {
		auto sur = IMG_Load(toStringz(fn));
		if(!sur) throw new SdlException();
		return new Surface(sur);
	}
	
	/** サーフェイスを指定して生成する。
	 *	Params:
	 *		sur		= サーフェイス。
	 */
	this(SDL_Surface* sur) {
		this();
		surface_ = sur;
	}
	
	/** 空のサーフェイスの生成。
	 *	Params:
	 *		s		= サイズ。
	 *		fmt		= ピクセルフォーマット。
	 *		flags	= 制御フラグ。
	 *	Throws:
	 *		PoetException	エラー発生時に投げられる。
	 */
	this(Size s, Format fmt, Flag flags) {
		this();
		surface_ = SDL_CreateRGBSurface(
			flags, s.w, s.h, fmt.depth, fmt.rmask, fmt.gmask, fmt.bmask, fmt.amask);
		if(!surface_) throw new SdlException();
	}
	
	/** サーフェイスの生成。
	 *	Params:
	 *		src		= 画像情報。サーフェイスが存在する間は破棄してはいけない。
	 *		s		= サイズ。
	 *		pitch	= 1ライン当たりのバイト数。
	 *		fmt		= ピクセルフォーマット。
	 *		flags	= 制御フラグ。
	 *	Throws:
	 *		PoetException	エラー発生時に投げられる。
	 */
	this(void[] src, Size s, uint pitch, Format fmt) {
		this();
		surface_ = SDL_CreateRGBSurfaceFrom(
			src.ptr, s.w, s.h, fmt.depth, pitch, fmt.rmask, fmt.gmask, fmt.bmask, fmt.amask);
		if(!surface_) throw new SdlException();
	}
	
	/// サーフェイスの破棄。
	~this() {free();}
	
	/** ビットブロック転送を行う。このサーフェイスが転送先となる。
	 *	Params:
	 *		pos		= 転送先位置。
	 *		src		= 転送元。
	 *		srcr	= 転送元矩形。
	 *	Throws:
	 *		PoetException	エラー発生時に投げられる。
	 */
	void blit(Point pos, Surface src, Rect srcr) {
		if(SDL_BlitSurface(src.surface_, srcr.ptr, surface_, Rect(pos, Size(0, 0)).ptr) < 0) {
			throw new SdlException();
		}
	}
	
	/** サーフェイスをロックする。
	 *	Returns:
	 *		ロックできたかどうか。
	 */
	bool lock() {return SDL_LockSurface(surface_) == 0;}
	
	/// サーフェイスをロック解除する。
	void unlock() {SDL_UnlockSurface(surface_);}
	
	/** ロック中に処理を行う。
	 *	Returns:
	 *		ロックできたかどうか。
	 */
	bool duringLock(void delegate() dg) {
		if(!lock()) return false;
		scope(exit) unlock();
		dg();
	}
	
	/// ロックが必要かどうか返す。
	bool mustLock() {return SDL_MUSTLOCK(surface_) != 0;}
	
	/// サイズを返す。
	Size size() {return Size(surface_.w, surface_.h);}
	
	/** 画面フォーマットに変換する。
	 *	Params:
	 *		alpha	= アルファチャンネルを使うかどうか。
	 *	Throws:
	 *		SdlException	エラー発生時に投げられる。
	 */
	void toDisplayFormat(bool alpha) {
		auto surf = alpha ? SDL_DisplayFormatAlpha(surface_) : SDL_DisplayFormat(surface_);
		if(!surf) throw new SdlException();
		
		// 古い画像と入れ替える。
		SDL_FreeSurface(surface_);
		surface_ = surf;
	}
	
	/// ポインタを返す。
	SDL_Surface* ptr() {return surface_;}
	
protected:
	
	/**	サーフェイスを入れ替える。
	 *	Params:
	 *		sur	= 新しいサーフェイス。
	 */
	void swapSurface(SDL_Surface* sur) {
		free();
		surface_ = sur;
	}
	
	/// サーフェイスを破棄する。
	void free() {
		// 終了処理中は破棄を行わない。
		if(!Application.isFinalizing) {
			SDL_FreeSurface(surface_);
		}
		surface_ = null;
	}
	
	/// デフォルトコンストラクタ。
	this() {}
	
private:
	
	/// サーフェイス。
	SDL_Surface* surface_;
}

/// 画面サーフェイスのクラス。
class Screen : Surface {
	
	enum : Flag {
		ASYNC_BLIT = SDL_ASYNCBLIT,
		ANY_FORMAT = SDL_ANYFORMAT,
		HARDWARE_PALETTE = SDL_HWPALETTE,
		DOUBLE_BUFFER = SDL_DOUBLEBUF | HARDWARE,
		FULLSCREEN = SDL_FULLSCREEN,
		OPENGL = SDL_OPENGL,
		OPENGLBLIT = SDL_OPENGLBLIT,
		RESIZABLE = SDL_OPENGLBLIT,
		NOFRAME = SDL_NOFRAME,
	}
	
	/**	現在の画面モードのサーフェイスを生成。
	 *	Throws:
	 *		SdlException	エラー発生時に投げられる。
	 */
	this() {
		auto sur = SDL_GetVideoSurface();
		if(sur is null) throw new SdlException();
		swapSurface(sur);
	}
	
	/** 画面モードを指定して生成。
	 *	Params:
	 *		s		= サイズ。
	 *		bpp		= ピクセルあたりのビット数。
	 *		flags	= 制御フラグ。
	 *	Throws:
	 *		SdlException	エラー発生時に投げられる。
	 */
	this(Size s,int bpp, Flag flags) {setVideoMode(s, bpp, flags);}
	
	/** 画面モードの設定。
	 *	Params:
	 *		s		= サイズ。
	 *		bpp		= ピクセルあたりのビット数。
	 *		flags	= 制御フラグ。
	 *	Throws:
	 *		SdlException	エラー発生時に投げられる。
	 */
	void setVideoMode(Size s, int bpp, Flag flags) {
		auto sur = SDL_SetVideoMode(s.w, s.h, bpp, flags);
		if(sur is null) throw new SdlException();
		swapSurface(sur);
	}
	
	/** 更新予約。
	 *	Params:
	 *		rect	= 更新予約する矩形。
	 */
	void update(Rect rect) {SDL_UpdateRect(surface_, rect.x, rect.y, rect.w, rect.h);}
	
	/// 画面全体の更新予約。
	void update() {update(Rect(0, 0, 0, 0));}
	
	/// 複数領域の更新予約。
	void update(Rect[] rects) {SDL_UpdateRects(surface_, rects.length, cast(SDL_Rect*) rects.ptr);}
	
	/** 高速な画面表示を行う。
	 *	Throws:
	 *		SdlException	エラー発生時に投げられる。
	 */
	void flip() {if(SDL_Flip(surface_) < 0) throw new SdlException();}
}
