﻿/**
 *	ページ関連。
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		NYSL Version 0.9982
 *	History:
 *		$Log$
 */

module outlandish.os.page;

import std.stdint;

import outlandish.os.memorymap;

/// 整数フラグ用テンプレート。
template IntegerFlagMixture(T) {
	bool getFlag(T F)() {return (value & (1U<<F)) != 0;}
	void setFlag(T F)(bool b) {
		if(b) {
			value |= (1U<<F);
		} else {
			value &= ~cast(T)(1U<<F);
		}
	}
}

/// ページエントリ。
struct PageEntry {
	
	mixin IntegerFlagMixture!(uint32_t);
	
	enum : uint32_t {
		BIT_PRESENT,
		BIT_WRITEABLE,
		BIT_USER,
		BIT_WRITE_THROUGH,
		BIT_CACHE_DISABLE,
		BIT_ACCESSED,
		BIT_DIRTY,
		BIT_LARGE_SIZE,
		BIT_GLOBAL,		
	}
	
	const BASE_OFFSET = 12U;
	const BASE_MASK = 0xffff_f000;
	
	/// コンストラクタ。
	static PageEntry opCall(void *address, bool present = true, bool writeable = true, bool user = false) {
		PageEntry e;
		e.address = address;
		e.present = present;
		e.user = user;
		e.writeable = writeable;
		return e;
	}
	
	alias getFlag!(BIT_PRESENT) present;	/// 存在フラグ。
	alias setFlag!(BIT_PRESENT) present;	/// ditto
	
	alias getFlag!(BIT_WRITEABLE) writeable;	/// 書き込み可フラグ。
	alias setFlag!(BIT_WRITEABLE) writeable;	/// ditto
	
	alias getFlag!(BIT_USER) user;	/// ユーザ権限フラグ。
	alias setFlag!(BIT_USER) user;	/// ditto
	
	alias getFlag!(BIT_WRITE_THROUGH) writeThrough;	/// ライトスルーキャッシュフラグ。
	alias setFlag!(BIT_WRITE_THROUGH) writeThrough;	/// ditto
	
	alias getFlag!(BIT_CACHE_DISABLE) cacheDisable;	/// キャッシュ無効フラグ。
	alias setFlag!(BIT_CACHE_DISABLE) cacheDisable;	/// ditto
	
	alias getFlag!(BIT_ACCESSED) accessed;	/// アクセス済みフラグ。
	alias setFlag!(BIT_ACCESSED) accessed;	/// ditto
	
	alias getFlag!(BIT_DIRTY) dirty;	/// 書き込み済みフラグ。
	alias setFlag!(BIT_DIRTY) dirty;	/// ditto
	
	alias getFlag!(BIT_LARGE_SIZE) largeSize;	/// サイズフラグ。
	alias setFlag!(BIT_LARGE_SIZE) largeSize;	/// ditto
	
	alias getFlag!(BIT_LARGE_SIZE) attribute;	/// ページ・テーブル属性インデックス。
	alias setFlag!(BIT_LARGE_SIZE) attribute;	/// ditto
	
	alias getFlag!(BIT_GLOBAL) global;	/// グローバルフラグ。
	alias setFlag!(BIT_GLOBAL) global;	/// ditto
	
	/// ページ番号。
	uint32_t pageIndex() {return value >> BASE_OFFSET;}
	
	/// ditto
	void pageIndex(uint32_t i) {value = (~BASE_MASK & value) | (i << BASE_OFFSET);}
	
	/// ページアドレス。
	void* address() {return cast(void*)(value & BASE_MASK);}
	
	/// ditto
	void address(void* p) {value = (~BASE_MASK & value) | (cast(uint32_t) p & BASE_MASK);}
	
	/// 整数値。
	uint32_t value;
}

/// ページ・テーブル。
alias PageEntry[1024] PageTable;

/// ページ・ディレクトリ。
alias PageEntry[1024] PageDirectory;

/// ページング有効化。
void enablePaging() {
	asm {
		mov EAX, CR0;
		or EAX, 0x8000_0000;
		mov CR0, EAX;
	}
}

/// ページング無効化。
void disablePaging() {
	asm {
		mov EAX, CR0;
		and EAX, 0x7fff_ffff;
		mov CR0, EAX;
	}
}

/// ページ・ディレクトリの設定。
void loadPageDirectory(PageDirectory* dir) {
	asm {
		mov EAX, dir;
		mov CR3, EAX;
	}
}

enum : uint32_t {
	/// ページ長。
	PAGE_LENGTH = 4096,
	
	/// ページ・ディレクトリ1エントリ分の長さ。
	PAGE_DIRECTORY_LENGTH = PAGE_LENGTH * 1024,
	
	/// ページ・オフセットまでのビット・オフセット。
	PAGE_OFFSET_OFFSET = 0,
	
	/// ページ・エントリ・インデックスまでのビット・オフセット。
	PAGE_ENTRY_OFFSET = 12,
	
	/// ページ・テーブル・インデックスまでのビット・オフセット。
	PAGE_TABLE_OFFSET = 22,
	
	/// ページ・オフセット部分のマスク。
	PAGE_OFFSET_MASK = 0x0000_0fff,
	
	/// ページ・エントリ・インデックス部分のマスク。
	PAGE_ENTRY_MASK = 0x003f_f000,
	
	/// ページ・テーブル・インデックス部分のマスク。
	PAGE_TABLE_MASK = 0xffc0_0000,
}

/// アドレスからページ・オフセットを得る。
uint32_t getPageOffset(void* p) {return cast(uint32_t) p & PAGE_OFFSET_MASK;}

/// アドレスからページ・エントリ・インデックスを得る。
uint32_t getPageEntryIndex(void* p) {return (cast(uint32_t) p & PAGE_ENTRY_MASK) >> PAGE_ENTRY_OFFSET;}

/// アドレスからページ・テーブル・インデックスを得る。
uint32_t getPageTableIndex(void* p) {return (cast(uint32_t) p & PAGE_TABLE_MASK) >> PAGE_TABLE_OFFSET;}

/// ページングの初期化。
bool initializePaging() {
	// メモリサイズを得る。
	uint64_t end = 0;
	foreach(e; MEMORY_MAP) {
		if(e.type == MemoryMapEntry.Type.MEMORY && end < e.end) {
			end = e.end;
		}
	}
	
	// ページ・テーブル数を求める。
	auto tableCount = cast(size_t)(((end + ~PAGE_TABLE_MASK) & PAGE_TABLE_MASK) >> PAGE_TABLE_OFFSET);
	
	// メモリ量があまりにも少なかったらエラー。(1MB未満?)
	if(tableCount == 0) {
		return false;
	}
	
	// 1メガ以上のメモリが全ページ・テーブルに十分なだけ存在すれば、
	// 1メガ以上の部分にページ・テーブルを設定。
	// そうでなければカーネル末尾に設定。
	if(end > cast(uint32_t)(PAGE_DIRECTORY_HIGH + PAGE_LENGTH + tableCount * PAGE_LENGTH)) {
		directory_ = cast(PageDirectory*) PAGE_DIRECTORY_HIGH;
		tables_ = (cast(PageTable*)(directory_ + 1))[0 .. tableCount];
	} else {
		directory_ = cast(PageDirectory*) kernel_end;
		tables_ = (cast(PageTable*)(directory_ + 1))[0 .. tableCount];
	}
	
	// ページ・ディレクトリ初期化。
	foreach(i, inout e; *directory_) {
		// 物理メモリの範囲内だった場合はページ・テーブルと対応付ける。
		if(i * PAGE_DIRECTORY_LENGTH < end) {
			e = PageEntry(tables_.ptr + i);
		} else {
			// 存在しない領域のエントリは0で初期化。
			e = PageEntry.init;
		}
	}
	
	// ページ・テーブル初期化。
	foreach(i, inout t; tables_) {
		foreach(j, inout e; t) {
			e = PageEntry(cast(void*)(i * PAGE_DIRECTORY_LENGTH + j * PAGE_LENGTH));
		}
	}
	
	// ページ・ディレクトリのロード。
	loadPageDirectory(directory_);
	
	// ページングの開始。
	enablePaging();
	
	return true;
}

/// ページ・ディレクトリを得る。
PageEntry[] getPageDirectory() {return *directory_;}

/// ページ・テーブルを得る。
PageTable[] getPageTables() {return tables_;}

private:

/// カーネル終端。
extern(C) void* kernel_end;

/// メモリが大量にある場合のページ・ディレクトリ位置。
void* PAGE_DIRECTORY_HIGH = cast(void*) 0x100000;

/// ページ・ディレクトリ。
PageDirectory* directory_;

/// ページ・テーブル。
PageTable[] tables_;
