﻿/**
 *	シリアル・コントローラ。
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		NYSL Version 0.9982
 *	History:
 *		$Log$
 */

module outlandish.os.serial;

import std.stdint;
import outlandish.os.io;

/// COMポート。
enum Port {
	COM1 = 0x3f8,
	COM2 = 0x2f8,
}

/// ボーレート。
enum Baudrate : uint16_t {
	BPS_115_2K = 0x0001,
	BPS_57_6K = 0x0002,
	BPS_38_4K = 0x0003,
	BPS_19_2K = 0x0006,
	BPS_9600 = 0x000c,
	BPS_4800 = 0x0018,
	BPS_2400 = 0x0030,
	BPS_1200 = 0x0060,
	BPS_600 = 0x00c0,
	BPS_300 = 0x0180,
}

/// 割り込みマスク。
enum InterruptEnable : uint8_t {
	RX_DATA	= 0b0001,
	TX_DATA	= 0b0010,
	RX_LINE	= 0b0100,
	MODEM	= 0b1000,
	NONE	= 0b0000,
}

/// ライン・ステータス。
enum LineStatus : uint8_t {
	RX_ERROR			= 0b1000_0000,
	COMPLETE_TRANSMIT	= 0b0100_0000,
	EMPTY_REGISTER		= 0b0010_0000,
	RECEIVE_BREAK		= 0b0001_0000,
	FRAMING_ERROR		= 0b0000_1000,
	PARITY_ERROR		= 0b0000_0100,
	RECEIVE_OVERFLOW	= 0b0000_0010,
	RECEIVE_DATA		= 0b0000_0001,
}

/// モデム・ステータス。
enum ModemStatus : uint8_t {
	DCD_ASSERT	= 0b1000_0000,
	RI_ASSERT	= 0b0100_0000,
	DSR_ASSERT	= 0b0010_0000,
	CTS_ASSERT	= 0b0001_0000,
	CHANGE_DCD	= 0b0000_1000,
	RI_NEGATE	= 0b0000_0100,
	CHANGE_DSR	= 0b0000_0010,
	CHANGE_CTS	= 0b0000_0001,
}

/// ライン・コントロール。
enum LineControl : uint8_t {
	ACCESS_BAUDRATE = 0b1000_0000,
	BREAK			= 0b0100_0000,
	FIXED_PARITY	= 0b0010_0000,
	EVEN_PARITY		= 0b0001_0000,
	ENABLE_PARITY	= 0b0000_1000,
	STOP_BITS_2		= 0b0000_0100,
	DATA_BITS_8		= 0b0000_0011,
}

/// モデム・コントロール。
enum ModemControl : uint8_t {
	LOOP_BACK			= 0b0001_0000,
	INTERRUPT_ENABLE	= 0b0000_1000,
	LOOP_BACK2			= 0b0000_0100,
	ASSERT_RTS			= 0b0000_0010,
	ASSERT_DTS			= 0b0000_0001,
}

/// シリアル・ポート操作関数。
class SerialPort {
	
	/// ポート・ボーレート・ラインコントロール・割り込みマスクを指定して初期化する。
	this(Port p,
		 Baudrate bps,
		 LineControl line = LineControl.DATA_BITS_8,
		 uint8_t modem = ModemControl.INTERRUPT_ENABLE | ModemControl.ASSERT_RTS | ModemControl.ASSERT_DTS,
		 InterruptEnable itr = InterruptEnable.NONE) {
		port_ = p;
			  	
		// ボーレート設定。
		outp(ioPort(Register.LINE_CONTROL), LineControl.ACCESS_BAUDRATE);
		outp(ioPort(Register.BAUDRATE_HIGH), cast(uint8_t)(bps >> 8u));
		outp(ioPort(Register.BAUDRATE_LOW), cast(uint8_t)(bps & 0x00ff));
		
		// ライン・コントロール設定。
		outp(ioPort(Register.LINE_CONTROL), line);
		
		// モデム・コントロール設定。
		outp(ioPort(Register.MODEM_CONTROL), modem);
		
		// 割り込みマスク設定。
		outp(ioPort(Register.INTERRUPT_ENABLE), itr);
	}
	
	/// データの出力。
	void put(uint8_t val) {outp(ioPort(Register.TRANSMIT_DATA), val);}
	
	/// データの入力。
	uint8_t get() {return inp(ioPort(Register.RECEIVE_DATA));}
	
	/// ライン・ステータス。
	uint8_t lineStatus() {return inp(ioPort(Register.LINE_STATUS));}
	
	/// モデム・ステータス。
	uint8_t modemStatus() {return inp(ioPort(Register.MODEM_STATUS));}
	
	/// 割り込みID。
	uint8_t interruptId() {return inp(ioPort(Register.INTERRUPT_ID));}
	
	/// 転送が終了したかどうか。
	bool isCompleteTransmit() {return (lineStatus() & LineStatus.COMPLETE_TRANSMIT) != 0;}
	
private:
	
	/// 各レジスタ。
	enum Register : uint16_t {
		RECEIVE_DATA = 0x0,		/// 受信データ。
		TRANSMIT_DATA = 0x0,	/// 送信データ。
		BAUDRATE_LOW = 0x0,		/// ボーレート下位。
		BAUDRATE_HIGH = 0x1,	/// ボーレート上位。
		INTERRUPT_ENABLE = 0x1,	/// 割り込みマスク。
		INTERRUPT_ID = 0x2,		/// 割り込みID。
		FIFO_CONTROL = 0x2,		/// FIFOコントロール。
		LINE_CONTROL = 0x3,		/// ライン・コントロール。
		MODEM_CONTROL = 0x4,	/// モデム・コントロール。
		LINE_STATUS = 0x5,		/// ライン・ステータス。
		MODEM_STATUS = 0x6,		/// モデム・ステータス。
		SCRATCH = 0x7,			/// スクラッチ・パッド。
	}
	
	/// IOポート番号を得る。
	uint16_t ioPort(Register f) {return cast(uint16_t)(port_ + f);}
	
	/// ポート番号。
	Port port_;
}
