﻿/**
 *	stream interfaces.
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module drt.stream;

import std.string;

interface Closeable {
	void close();
	bool isOpen();
}

interface Flushable {
	void flush();
}

interface Endable {
	bool eof();
}

/// data input stream.
interface InputStream {
	
	size_t read(void[] buffer);
	void readExact(void[] buffer);
	
	void read(out byte x);
	void read(out ubyte x);
	void read(out short x);
	void read(out ushort x);
	void read(out int x);
	void read(out uint x);
	void read(out long x);
	void read(out ulong x);
	void read(out float x);
	void read(out double x);
	void read(out real x);
	void read(out ifloat x);
	void read(out idouble x);
	void read(out ireal x);
	void read(out cfloat x);
	void read(out cdouble x);
	void read(out creal x);
	void read(out char x);
	void read(out wchar x);
	void read(out dchar x);
	void read(char[] s);
}

/// data output stream.
interface OutputStream {
	
	size_t write(void[] buffer);
	void writeExact(void[] buffer);
	
	void write(byte x);
	void write(ubyte x);
	void write(short x);
	void write(ushort x);
	void write(int x);
	void write(uint x);
	void write(long x);
	void write(ulong x);
	void write(float x);
	void write(double x);
	void write(real x);
	void write(ifloat x);
	void write(idouble x);
	void write(ireal x);
	void write(cfloat x);
	void write(cdouble x);
	void write(creal x);
	void write(char x);
	void write(wchar x);
	void write(dchar x);
	void write(char[] s);
}

/// data to string stream.
abstract class OutputPrintStream : OutputStream {
	
	/// write a character.
	void write(char x) {writeImpl(x);}
	
	/// write a string.
	void write(char[] s) {foreach(c; s) write(c);}
	
	/// ditto
	size_t write(void[] buffer) {
		foreach(c; cast(char[]) buffer) write(c);
		return buffer.length;
	}
	
	/// ditto
	void writeExact(void[] buffer) {write(buffer);}
	
	/// ditto
	void write(byte x) {outputDecimal(x);}
	/// ditto
	void write(ubyte x) {outputHexadecimal(x);}
	/// ditto
	void write(short x) {outputDecimal(x);}
	/// ditto
	void write(ushort x) {outputHexadecimal(x);}
	/// ditto
	void write(int x) {outputDecimal(x);}
	/// ditto
	void write(uint x) {outputHexadecimal(x);}
	
	/// not implemented.
	void write(long x) {outputDecimal(cast(int)x);}
	/// ditto
	void write(ulong x) {outputHexadecimal(x);}
	/// ditto
	void write(float x) {}
	/// ditto
	void write(double x) {}
	/// ditto
	void write(real x) {}
	/// ditto
	void write(ifloat x) {}
	/// ditto
	void write(idouble x) {}
	/// ditto
	void write(ireal x) {}
	/// ditto
	void write(cfloat x) {}
	/// ditto
	void write(cdouble x) {}
	/// ditto
	void write(creal x) {}
	/// ditto
	void write(wchar x) {}
	/// ditto
	void write(dchar x) {}
	
protected:
	
	/// abstract output function. used all write functions.
	abstract void writeImpl(char x);
	
private:
	
	/// decimal to string.
	void outputDecimal(int val) {
		if(val < 0) {
			write('-');
			val = -val;
		}
		
		int div = 10_0000_0000;
		for(; div > val; div /= 10) {}
		
		for(; div > 0; val %= div, div /= 10) {
			write(std.string.digits[cast(size_t)(val / div)]);	
		}
	}
	
	/// hexadecimal to string.
	void outputHexadecimal(T)(T val) {
		write('0');
		write('x');
		
		for(int i = (T.sizeof * 8) - 4; i >= 0; i -= 4) {
			write(std.string.hexdigits[cast(uint)(val >> i) & 0x0f]);
		}
	}
}
