﻿/**
 *	cache structures.
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module os.i386.cache;

import std.stdint;

import os.i386.page;

import drt.list;
import drt.algorithm;

/**	object cache allocator.
 *
 *	Params:
 *		N	= object size.
 *		A	= object alignment.
 */
struct CacheAllocator {
	
	static size_t calculateObjectSize(size_t size, size_t aln) {
		return (size + aln - 1) / aln * aln;
	}
	
	static CacheAllocator opCall(size_t size, size_t aln) {
		size = calculateObjectSize(size, aln);
		CacheAllocator cache;
		cache.size_ = size;
		cache.align_ = aln;
		cache.len_ = max!(size_t)((PAGE_LENGTH - cache.headSize) / size, 1U);
		auto useLength = cache.len_ * size + cache.headSize;
		cache.slabPages_ = (useLength + PAGE_LENGTH) / PAGE_LENGTH;
		cache.colors_ = (cache.slabPages_ * PAGE_LENGTH - useLength) / aln;
		return cache;
	}
	
	void* allocate() {
		if(used_ is null) {
			Slab* slab;
			if(empty_ !is null) {
				// use from empty
				slab = empty_;
				empty_ = slab.next;
				slab.erase();
			} else {
				// allocate new slab
				auto buf = cast(ubyte*) allocatePages(slabPages_, Page.Purpose.KERNEL);
				if(buf is null) {
					// failure. not enough memory
					return null;
				}
				
				// initialize slab
				slab = cast(Slab*) buf;
				*slab = Slab((buf + headSize + color_ * align_)[0 .. len_ * size_], size_);
				
				// next color
				if(++color_ > colors_) {
					color_ = 0;
				}
			}
			
			// empty slab to use
			used_ = slab;
		}
		
		auto result = used_.allocate();
		if(!used_.hasMore) {
			// add full slab list
			Slab* slab = used_;
			used_ = used_.next;
			
			if(full_ !is null) {
				full_.insertBefore(slab);
			} else {
				slab.erase();
			}
			full_ = slab;
		}
		
		return result;
	}
	
	void deallocate(void* obj) {		
		if(auto slab = deallocateList(obj, full_)) {			
			if(full_ is slab) {
				full_ = slab.next;
			}
			if(used_ !is null) {
				used_.insertBefore(slab);
			} else {
				slab.erase();
			}
			used_ = slab;
		} else if(auto slab = deallocateList(obj, used_)) {
			if(slab.used == 0) {
				if(used_ is slab) {
					used_ = used_.next;
				}
				if(empty_ !is null) {
					empty_.insertBefore(slab);
				} else {
					slab.erase();
				}
				empty_ = slab;
			}
		}
	}
	
	size_t usedCount() {return (used_ !is null) ? used_.length : 0;}
	size_t fullCount() {return (full_ !is null) ? full_.length : 0;}
	size_t emptyCount() {return (empty_ !is null) ? empty_.length : 0;}
	size_t slabCount() {return usedCount() + fullCount() + emptyCount();}
	size_t objectSize() {return size_;}
	size_t objectAlign() {return align_;}
	size_t objectCountParSlab() {return len_;}
	size_t colors() {return colors_;}
	
	bool deallocateEmptySlabs() {
		if(empty_ is null) {
			return false;
		}
		
		for(Slab* p = empty_; p !is null;) {
			auto tmp = p;
			p = p.next;
			deallocatePages(p, slabPages_);
		}
		empty_ = null;
		return true;
	}
	
private:
	
	struct Slab {
		
		static Slab opCall(ubyte[] buf, size_t size) {
			Slab slab;
			slab.buffer_ = buf;
			if(buf.length > 0) {
				// initialize free list.
				buf[] = 0;
				for(size_t i = 0; i < (buf.length - size); i += size) {
					(cast(Chunk*)(&buf[i])).next = cast(Chunk*)(&buf[i + size]);
				}
				
				slab.free_ = cast(Chunk*) buf.ptr;
			}
			return slab;
		}
		
		void* allocate() {
			if(free_ !is null) {
				auto result = free_;
				free_ = free_.next;
				++used_;
				return result;
			}
			return null;
		}
		
		bool deallocate(void* obj) {				
			if((used_ > 0) && (buffer_.ptr <= obj) && (obj < (buffer_.ptr + buffer_.length))) {
				auto chunk = cast(Chunk*) obj;
				chunk.next = free_;
				free_ = chunk;
				--used_;
				return true;
			}
			return false;
		}
		
		bool hasMore() {return free_ !is null;}
		
		size_t used() {return used_;}
		
	private:
		
		struct Chunk {
			Chunk *next;
		}
		
		Chunk* free_;
		ubyte[] buffer_;
		size_t used_;
		
	public:
		
		mixin ListMixture;
	}
	
	Slab* deallocateList(void* obj, Slab* begin) {
		for(Slab* s = begin; s !is null; s = s.next) {
			if(s.deallocate(obj)) {
				return s;
			}
		}
		return null;
	}
	
	size_t headSize() {
		return (Slab.sizeof + align_ - 1) / align_ * align_;
	}
	
	Slab* used_;
	Slab* empty_;
	Slab* full_;
	size_t len_;
	size_t size_;
	size_t align_;
	size_t slabPages_;
	size_t colors_;
	size_t color_;
	
public:
	
	mixin ListMixture;
}

void initializeCache() {
	cacheCache_ = CacheAllocator(CacheAllocator.sizeof, CacheAllocator.alignof);
	cacheTable_ = CacheTable.init;
}

CacheAllocator* createCache(size_t n, size_t aln) {
	auto ptr = cast(CacheAllocator*) cacheCache_.allocate();
	if(ptr is null) {
		return null;
	}
	*ptr = CacheAllocator(n, aln);
	return ptr;
}

void[] allocateFromCache(size_t n) {return cacheTable_.allocate(n);}

void deallocateFromCache(void[] obj) {cacheTable_.deallocate(obj);}

T* allocateStructFromCache(T)() {
	auto result = cast(T*) cacheTable_.allocate(T.sizeof).ptr;
	*result = T.init;
	return result;
}

void deallocateStructFromCache(T)(T* obj) {
	cacheTable_.deallocate((cast(void*)obj)[0 .. T.sizeof]);
}

bool deallocateFreeCache() {
	return cacheTable_.deallocateFreeCache();
}

class KernelObject {
	
	new(uint size) {
		auto buf = allocateFromCache(size_t.sizeof + size);
		*(cast(size_t*) buf.ptr) = size_t.sizeof + size;		
		return buf.ptr + size_t.sizeof;
	}
	
	delete(void* p) {
		p = p - size_t.sizeof;
		deallocateFromCache(p[0 .. *(cast(size_t*) p)]);
	}
}

class KernelList(T) : List!(T) {
protected:
	Entry* allocate() {
		return allocateStructFromCache!(Entry)();
	}
	
	void deallocate(Entry* e) {
		deallocateStructFromCache(e);
	}
}

private:

/// CacheAllocator hash table. key is object size.
struct CacheTable {
	
	const size_t HASH_RANGE = 32;
	const size_t ALIGN = 4;
	
	void[] allocate(size_t n) {
		n = CacheAllocator.calculateObjectSize(n, ALIGN);
		auto cache = getAllocator(n);
		if(cache !is null) {
			auto ptr = cache.allocate();
			if(ptr !is null) {
				return ptr[0 .. n];
			}
			
			// retry
			if(deallocateFreeCache()) {
				ptr = cache.allocate();
				if(ptr !is null) {
					return ptr[0 .. n];
				}
			}
		}
		return (void[]).init;
	}
	
	void deallocate(void[] obj) {
		if(auto cache = getAllocator(obj.length)) {
			cache.deallocate(obj.ptr);
		}
	}
	
	bool deallocateFreeCache() {
		bool found = false;
		foreach(p; index_) {
			if(p !is null) {
				found |= p.deallocateEmptySlabs();
			}
		}
		return found;
	}
	
private:
	
	/// get allocator what match size. if not found allocator, create new one
	CacheAllocator* getAllocator(size_t size) {
		auto i = hashSize(size);
		auto head = index_[i];
		
		// find same size cache
		for(CacheAllocator* p = head; p !is null; p = p.next) {
			if(p.objectSize == size) {
				return p;
			}
		}
		
		// create new cache
		auto cache = createCache(size, ALIGN);
		if(cache is null) {
			// retry
			if(deallocateFreeCache()) {
				cache = createCache(size, ALIGN);
				if(cache is null) {
					return null;
				}
			}
		}
		
		// insert list
		if(index_[i] !is null) {
			head.insertBefore(cache);
		}
		index_[i] = cache;
		
		return cache;
	}
	
	/// object size to hash value.
	size_t hashSize(size_t n) {return n % HASH_RANGE;}
	
	CacheAllocator*[HASH_RANGE] index_;
}

CacheAllocator cacheCache_;
CacheTable cacheTable_;
