﻿/**
 *	paging structures.
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module os.i386.page;

import std.stdint;
import std.c.string : memset;

import os.i386.memorymap;

import drt.algorithm;
import drt.stream;

import os.i386.bitmap;
import os.i386.cache;
import os.i386.memorymap;
import os.i386.sync;

template IntegerFlagMixture(T) {
	bool getFlag(uint32_t F)() {return (value & F) != 0;}
	void setFlag(uint32_t F)(bool b) {
		if(b) {
			value |= F;
		} else {
			value &= ~F;
		}
	}
	
	T value;
}

enum : uint32_t {
	PAGE_LENGTH = 4096,
	PAGE_DIRECTORY_LENGTH = PAGE_LENGTH * 1024,
	PAGE_OFFSET_OFFSET = 0,
	PAGE_ENTRY_OFFSET = 12,
	PAGE_TABLE_OFFSET = 22,
	
	PAGE_OFFSET_MASK = 0x0000_0fff,
	PAGE_ENTRY_MASK = 0x003f_f000,
	PAGE_TABLE_MASK = 0xffc0_0000,
}

void initializePages(MemoryMapEntry[] map) {
	
	initializeMemoryMap(map);
	initializeCache();
	
	auto pagesBegin = cast(Page*) DYNAMIC_MEMORY_BEGIN;
	auto memoryEnd = cast(uintptr_t) getMemoryEnd();
	size_t pageCount = 0;
	for(uintptr_t p = 0; p < memoryEnd; p += PAGE_LENGTH, ++pageCount) {
		pagesBegin[pageCount] = Page(pageCount, Page.Purpose.NONE);
	}
	pages_ = pagesBegin[0 .. pageCount];
	
	// set reserved pages.
	foreach(MemoryMapEntry e; getMemoryMap()) {
		if(e.type != MemoryMapEntry.Type.MEMORY || e.attr != MemoryMapEntry.Attribute.ENABLED) {
			auto end = cast(uintptr_t)((e.end + PAGE_LENGTH - 1) / PAGE_LENGTH);
			for(auto p = cast(uintptr_t)((e.base + PAGE_LENGTH - 1) / PAGE_LENGTH); p < end; ++p) {
				pages_[p].purpose = Page.Purpose.RESERVED;
			}
		}
	}
	
	// initialize page allocator.
	auto pend = cast(ubyte*)(pages_.ptr + pages_.length);
	allocator_.initialize(pages_, pend);
	auto end = cast(uintptr_t) pend;
	end = (end + PAGE_OFFSET_MASK) & ~PAGE_OFFSET_MASK;
	
	// initialize page directory
	directory_ = cast(PageDirectory*) end;
	(*directory_)[] = PageEntry.init;
	end += PageDirectory.sizeof;
	
	// initialize page table
	tables_ = (cast(PageTable*)end)[0 .. (pages_.length + PAGE_LENGTH - 1) / PAGE_LENGTH];
	foreach(inout t; tables_) {
		t[] = PageTableEntry.init;
	}
	end = cast(uintptr_t)(tables_.ptr + tables_.length);
	
	// kernel memory mapping
	
	// page directory
	for(uintptr_t p = 0; p < memoryEnd; p += PAGE_DIRECTORY_LENGTH) {
		auto ptr = cast(void*) p;
		setPageDirectoryEntry(ptr, PageEntry(&tables_[getPageDirectoryIndex(ptr)]));
	}
	
	// page tables
	for(uintptr_t p = 0; p < memoryEnd; p += PAGE_LENGTH) {
		auto ptr = cast(void*) p;
		setPageTableEntry(ptr, PageEntry(ptr));
	}
	
	// reserve kernel code & data
	foreach(i, inout p; pages_) {
		if(cast(uintptr_t) p.baseAddress < end) {
			p.purpose = Page.Purpose.KERNEL;
		}
	}

	// free dynamic memories
	allocator_.deallocate(pages_[end >> PAGE_ENTRY_OFFSET .. $]);
	
	// start paging
	loadPageDirectory(directory_);
	enablePaging();
}

struct Page {
	
	enum Purpose : uint8_t {
		NONE,
		RESERVED,
		KERNEL,
		USER,
		SHARE,
		DMA_BUFFER,
	}
	
	static Page opCall(size_t i, Purpose p) {
		Page page;
		page.index_ = i;
		page.purpose = p;
		return page;
	}
	
	Purpose purpose() {return purpose_;}
	void purpose(Purpose p) {purpose_ = p;}
	
	size_t index() {return index_;}
	
	void* baseAddress() {return cast(void*)(index_ * PAGE_LENGTH);}
	
	bool use() {return use_;}
	void use(bool b) {use_ = b;}
	
private:
	
	size_t index_;
	Purpose purpose_;
	bool use_ = true;
}

struct PageAllocator {
	
	/// make page bitmaps. pbuf is bitmaps begin. pbuf moved to bitmap end.
	void initialize(Page[] pages, inout ubyte* pbuf) {
		auto n = (pages.length + 1) / 2;
		foreach(inout m; maps_) {
			auto bsize = max(1U, (n + 7) / 8);
			m = Bitmap(pbuf, bsize);
			pbuf += bsize;
			n /= 2;
		}
		pages_ = pages;
	}
	
	/// allocate sequential pages.
	Page[] allocate(size_t n, Page.Purpose purpose) {
		Page[] result;
		foreach(i, inout m; maps_) {
			if(n <= (1U << i)) {
				size_t pi = m.find(true);
				if(pi < m.length) {
					m[pi] = false;
					size_t base = 1U << (i + 1);
					pi *= base;
					if(pages_[pi].use) {
						pi += (base / 2);
					}
					
					deallocate(pages_[pi + n .. min(pi + base, pages_.length)]);
					
					result = pages_[pi .. pi + n];
					foreach(inout p; result) {
						p.use = true;
						p.purpose = purpose;
					}
					return result;
				}
			}
		}
		return result;
	}
	
	/// deallocate *sequential* pages.
	void deallocate(Page[] pages) {
		foreach(inout p; pages) {
			p.use = false;
		}
		for(size_t pi = 0; pi < pages.length;) {
			foreach(i, inout m; maps_) {
				size_t n = 1U << i;
				if((pages[pi].index & n) || ((pages.length - pi) < (n << 1)) || (i == maps_.length - 1)) {
					for(; i < maps_.length; ++i) {
						size_t bi = pages[pi].index >> (i + 1);
						if(maps_[i][bi]) {
							maps_[i][bi] = false;
						} else {
							maps_[i][bi] = true;
							break;
						}
					}
					pi += n;
					break;
				}
			}
		}
	}
	
	/// print bitmap states.
	void print(OutputPrintStream dest) {
		foreach(m; maps_) {
			m.print(dest);
			dest.write("\r\n");
		}
	}
	
private:
	
	Bitmap maps_[10];
	Page[] pages_;
}

/// page table entry structure. it used by MMU.
struct PageEntry {
	
	mixin IntegerFlagMixture!(uint32_t);
	
	enum : uint32_t {
		PRESENT_FLAG = 0b0000_0001,
		READ_WRITE_FLAG = 0b0000_0010,
		USER_SUPERVISOR_FLAG = 0b0000_0100,
		WRITE_THROW_FLAG = 0b0000_1000,
		CACHE_DISABLE_FLAG = 0b0001_0000,
		ACCESS_FLAG = 0b0010_0000,
		DIRTY_FLAG = 0b0100_0000,
		PAGE_SIZE_FLAG = 0b1000_0000,
		
		BASE_ADDRESS_MASK = 0xfffff000,
	}
	
	const size_t BASE_ADDRESS_OFFSET = 12;
	
	static PageEntry opCall(void* base, bool present = true, bool user = false, bool write = true) {
		PageEntry e;
		e.base = (cast(uint32_t) base) >> BASE_ADDRESS_OFFSET;
		e.present = present;
		e.userSupervisor = user;
		e.readWrite = write;
		return e;
	}
	
	alias getFlag!(PRESENT_FLAG) present;
	alias getFlag!(DIRTY_FLAG) dirty;
	alias getFlag!(ACCESS_FLAG) access;
	alias getFlag!(CACHE_DISABLE_FLAG) cacheDisable;
	alias getFlag!(WRITE_THROW_FLAG) writeThrow;
	alias getFlag!(USER_SUPERVISOR_FLAG) userSupervisor;
	alias getFlag!(READ_WRITE_FLAG) readWrite;
	alias getFlag!(PAGE_SIZE_FLAG) pageSize;
	
	alias setFlag!(PRESENT_FLAG) present;
	alias setFlag!(DIRTY_FLAG) dirty;
	alias setFlag!(ACCESS_FLAG) access;
	alias setFlag!(CACHE_DISABLE_FLAG) cacheDisable;
	alias setFlag!(WRITE_THROW_FLAG) writeThrow;
	alias setFlag!(USER_SUPERVISOR_FLAG) userSupervisor;
	alias setFlag!(READ_WRITE_FLAG) readWrite;
	alias setFlag!(PAGE_SIZE_FLAG) pageSize;
	
	uint32_t base() {return value >> BASE_ADDRESS_OFFSET;}
	void base(uint32_t val) {value = (value & ~BASE_ADDRESS_MASK) | (val << BASE_ADDRESS_OFFSET);}
	
	void baseAddress(void* p) {value = (value & ~BASE_ADDRESS_MASK) | (cast(uint32_t)p & BASE_ADDRESS_MASK);}
	void* baseAddress() {return cast(void*)(value & BASE_ADDRESS_MASK);}
}

alias PageEntry PageTableEntry;
alias PageEntry PageDirectoryEntry;

alias PageTableEntry[1024] PageTable;
alias PageDirectoryEntry[1024] PageDirectory;

Page[] getPages() {return pages_;}

size_t getPageIndex(void* p) {
	return (cast(uintptr_t) p) >> PAGE_ENTRY_OFFSET;
}

size_t getPageDirectoryIndex(void* p) {
	return (cast(uintptr_t) p & PAGE_TABLE_MASK) >> PAGE_TABLE_OFFSET;
}

size_t getPageTableIndex(void* p) {
	return (cast(uintptr_t) p & PAGE_ENTRY_MASK) >> PAGE_ENTRY_OFFSET;
}

PageDirectoryEntry getPageDirectoryEntry(void* p) {
	auto index = getPageDirectoryIndex(cast(void*) p);
	return (*directory_)[index];
}

void setPageDirectoryEntry(void* p, PageDirectoryEntry e) {
	auto index = getPageDirectoryIndex(cast(void*) p);
	(*directory_)[index] = e;
}

PageTableEntry getPageTableEntry(void* p) {
	auto ptr = cast(void*) p;
	return tables_[getPageDirectoryIndex(ptr)][getPageTableIndex(ptr)];
}

void setPageTableEntry(void* p, PageTableEntry e) {
	auto ptr = cast(void*) p;
	tables_[getPageDirectoryIndex(ptr)][getPageTableIndex(ptr)] = e;
}

void enablePaging() {
	asm {
		mov EAX, CR0;
		or EAX, 0x8000_0000;
		mov CR0, EAX;
	}
}

void disablePaging() {
	asm {
		mov EAX, CR0;
		and EAX, 0x7fff_ffff;
		mov CR0, EAX;
	}
}

void flushTlb() {
	asm {
		mov EAX, CR3;
		mov CR3, EAX;
	}
}

void loadPageDirectory(PageDirectory* dir) {
	asm {
		mov EAX, dir;
		mov CR3, EAX;
	}
}

PageDirectory* getPageDirectory() {
	PageDirectory* dir;
	asm {
		mov EAX, CR3;
		mov dir, EAX;
	}
	return dir;
}

PageDirectory* getKernelPageDirectory() {return directory_;}

void* allocatePages(size_t n, Page.Purpose purpose, bool zeroed = true) {
	auto pages = allocator_.allocate(n, purpose);
	void* result = (pages.length == 0) ? null : pages[0].baseAddress;
	if(zeroed) {
		memset(result, 0, n * PAGE_LENGTH);
	}
	return result;
}

void deallocatePages(void* begin, size_t n) {
	auto i = getPageIndex(begin);
	allocator_.deallocate(pages_[i .. i + n]);
}

private:

Page[] pages_;
PageAllocator allocator_;
PageDirectory* directory_;
PageTable[] tables_;
