﻿/**
 *	kernel startup code.
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module os.i386.startup;

import os.i386.cache;
import os.i386.errorcode;
import os.i386.interrupt;
import os.i386.io;
import os.i386.memorymap;
import os.i386.page;
import os.i386.pic8259a;
import os.i386.process;
import os.i386.segment;
import os.i386.serial;
import os.i386.sync;
import os.i386.systemcall;
import os.i386.tss;

import drt.stream;

import std.stdint;

extern(C) void _moduleCtor();
extern(C) void _moduleDtor();

/**
 *	C interface startup function
 *
 *	call main function.
 */
extern(C) void startup(MemoryMapEntry* map, size_t len) {
	
	// call module constructor
	_moduleCtor();
	
	// cancel 1MB memory limit
	enableA20Pin();
	
	Serial!(Port.COM1).setup(Baudrate.BPS_9600, LineControl.DATA_BITS_8, InterruptEnable.NONE);
	Serial!(Port.COM2).setup(Baudrate.BPS_9600, LineControl.DATA_BITS_8, InterruptEnable.NONE);
	
	// initialize pages information.
	initializePages(map[0 .. len / MemoryMapEntry.sizeof]);
	
	// print memory map.
	scope stream = new ComStream!(Port.COM1);
	foreach(MemoryMapEntry e; getMemoryMap()) {
		e.print(stream);
	}

	with(stream) {
		write("kernel end:");
		write(cast(uintptr_t) getKernelEnd());
		write("\r\n");
	
		write("memory end:");
		write(getMemoryEnd());
		write("\r\n");
	
		write("page count:");
		write(getPages().length);
		write("\r\n");
	}
	
	if(!initializeTaskStateRegister()) {
		stream.write("TSS allocate error!");
		stream.write("\r\n");
		stop();
	}
	
	loadInterruptDescriptorTable(new Pic8259A, &defaultInterruptHandler);
	
	// set system call handler.
	initializeSystemCall();
	
	initializeProcessSubsystem();
	
	// setup IRQ handlers.
	for(auto i = Interrupt.IRQ_BEGIN + 0; i < Interrupt.IRQ_END; ++i) {
		setInterruptHandler(i, HandlerType.INTERRUPT, HandlerPrivilege.KERNEL, &onIrq);
	}
	setInterruptHandler(Interrupt.IRQ_SYSTEM_TIMER, HandlerType.INTERRUPT, HandlerPrivilege.KERNEL, &onSystemTimer);
	
	auto process = new Process();
	auto thread = new Thread(process);
	
	if(!thread.firstBegin(&firstThread)) {
		stream.write("first thread start error!");
		stream.write("\r\n");
		stop();
	}
		
	// call module destructor
	_moduleDtor();
}

/**
 *	Enable A20 pin.
 *
 *	cancel 1MB memory wraparound.
 */
void enableA20Pin() {
	
	// reset A20 pin
	while(inp(0x64) & 0x02) {}
	
	outp(0x64, 0xd1);
	while(inp(0x64) & 0x02) {}
	
	outp(0x60, 0xdf);
	while(inp(0x64) & 0x02) {}
	
	outp(0x64, 0xff);
	while(inp(0x64) & 0x02) {}
}

void firstThread(void* arg) {
	scope stream = new ComStream!(Port.COM1);
	if(!forkThread(&secondThread, cast(void*) 1)) {
		stream.write("failure fork!\r\n");
		stop();
	}
	
	// wait interrupt
	stream.write("enableInterrupt\r\n");
	stream.write("first  thread:");
	stream.write(cast(uintptr_t) arg);
	stream.write("\r\n");
	
	enableInterrupt();
	for(;;) {
		stop();
	}
}

void secondThread(void* arg) {
	scope stream = new ComStream!(Port.COM1);
	stream.write("second thread:");
	stream.write(cast(uintptr_t) arg);
	stream.write("\r\n");
	
	for(;;) {
		stop();
	}
}

/// stop kernel
void die() {
	asm {
		cli;
		hlt;
	}
}

/// stop kernel
void stop() {
	asm {
		hlt;
	}
}

void initializeSystemCall() {
	setInterruptHandler(Interrupt.SYSTEM_CALL, HandlerType.TRAP, HandlerPrivilege.KERNEL, &onSystemCall);
}

SysRes testSystemCall(SysArg n, SysArg a1, SysArg a2, SysArg a3, SysArg a4, SysArg a5) {
	return Errno.SUCCESS;
}

void onIrq(inout InterruptContext ctx) {
	scope stream = new ComStream!(Port.COM1);
	stream.write("IRQ receive : ");
	stream.write(ctx.n - Interrupt.IRQ_BEGIN);
	stream.write("\r\n");
	
	getInterruptController().ackInterrupt(ctx.n);
}

void onSystemTimer(inout InterruptContext ctx) {
	getInterruptController().ackInterrupt(ctx.n);
	if(!schedule()) {
		scope stream = new ComStream!(Port.COM1);
		stream.write("failure switch thread!\r\n");
	}
}

void onIrqSilent(inout InterruptContext ctx) {
	getInterruptController().ackInterrupt(ctx.n);
}

void defaultInterruptHandler(inout InterruptContext ctx) {
	scope stream = new ComStream!(Port.COM1);
	with(stream) {
		write("unknown interrupt!\r\n");
		write("edi = ");
		write(ctx.edi);
		write("\r\n");
		write("esi = ");
		write(ctx.esi);
		write("\r\n");
		write("ebp = ");
		write(ctx.ebp);
		write("\r\n");
		write("esp = ");
		write(ctx.esp);
		write("\r\n");
		write("ebx = ");
		write(ctx.ebx);
		write("\r\n");
		write("edx = ");
		write(ctx.edx);
		write("\r\n");
		write("ecx = ");
		write(ctx.ecx);
		write("\r\n");
		write("eax = ");
		write(ctx.eax);
		write("\r\n");
		write("es = ");
		write(ctx.es);
		write("\r\n");
		write("ds = ");
		write(ctx.ds);
		write("\r\n");
		write("n = ");
		write(ctx.n);
		write("\r\n");
		write("err = ");
		write(ctx.err);
		write("\r\n");
		write("eip = ");
		write(ctx.eip);
		write("\r\n");
		write("cs = ");
		write(ctx.cs);
		write("\r\n");
		write("eflags = ");
		write(ctx.eflags);
		write("\r\n");
	}
	
	stop();
}
