﻿/**
 *	TSS structure.
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module os.i386.tss;

import std.stdint;

import os.i386.io;
import os.i386.page;
import os.i386.segment;

/// Task State Segment structure
struct TaskStateSegment {
	uint16_t backLink;

	uint32_t esp0;
	uint16_t ss0;
	uint16_t padding0;
	
	uint32_t esp1;
	uint16_t ss1;
	uint16_t padding1;
	
	uint32_t esp2;
	uint16_t ss2;
	uint16_t padding2;
	
	uint32_t cr3;
	uint32_t eip;
	uint32_t eflags;
	uint32_t eax;
	uint32_t ecx;
	uint32_t edx;
	uint32_t ebx;
	uint32_t esp;
	uint32_t ebp;
	uint32_t esi;
	uint32_t edi;
	
	uint16_t es;
	uint16_t padding3;
	
	uint16_t cs;
	uint16_t padding4;
	
	uint16_t ss;
	uint16_t padding5;
	
	uint16_t ds;
	uint16_t padding6;
	
	uint16_t fs;
	uint16_t padding7;
	
	uint16_t gs;
	uint16_t padding8;
	
	uint16_t ldt;
	uint16_t padding9;
	
	uint16_t debugTrap;	
	uint16_t ioBitmapBase;
	
	ubyte[IO_PORT_COUNT / 8 + 1] ioBits;
	
	void initialize(uint32_t esp, uint32_t eip) {
		cr3 = cast(uint32_t) getPageDirectory();
		esp0 = 0x1000;
		ss0 = cast(uint16_t) GdtSelector.KERNEL_DATA;
		
		esp2 = 0x1000;
		ss2 = cast(uint16_t) GdtSelector.USER_DATA;
		
		uint32_t flags;
		asm {
			pushfd;
			pop flags;
		}
		eflags = flags;
		
		ioBitmapBase = cast(uint16_t) ioBits.offsetof;
		ioBits[] = 0xff;
	}
	
	void enableIoPort(uint port) {ioBits[port >> 3] |= 0x1 << (port & 0x07);}
	void disableIoPort(uint port) {ioBits[port >> 3] &= ~(0x1 << (port & 0x07));}
	bool isEnableIoPort(uint port) {return (ioBits[port >> 3] & (0x1 << (port & 0x07))) == 0;}
}

bool initializeTaskStateRegister() {
	tss_ = cast(TaskStateSegment*) allocatePages(
		(TaskStateSegment.sizeof + PAGE_LENGTH - 1) / PAGE_LENGTH, Page.Purpose.KERNEL);
	if(tss_ is null) {
		return false;
	}
	tss_.initialize(0x1000, 0);
	
	GDT[GdtIndex.TSS].base = cast(uint32_t) tss_;
	GDT[GdtIndex.TSS].limit = TaskStateSegment.sizeof - 1;
	
	loadTaskStateRegister(GdtSelector.TSS);
	return true;
}

TaskStateSegment* getTaskStateSegment() {return tss_;}

private:

void loadTaskStateRegister(uint16_t i) {
	asm {
		ltr i;
		jmp reset_pipeline;
	reset_pipeline:;
	}
}

TaskStateSegment* tss_;
