﻿/**
 *  整数パーサのモジュール。
 *
 *  Version:
 *      $Revision$
 *  Date:
 *      $Date$
 *  License:
 *      MIT/X Consortium License
 *  History:
 *      $Log$
 */

module outland.parser.intparser;

import std.ctype;
import std.stdio;
import std.string;

import outland.parser.parser;
import outland.parser.spaceparser;

/// 数字パーサ戻り値の型。
struct DigitParserResultType(I) {
    I begin;        /// 開始位置。
    size_t length;  /// マッチした長さ。
    bool match;     /// マッチしたかどうか。
    ubyte value;    /// 数値。
}

/** 数字パーサ。
 *  Params:
 *      digits  = 数字の配列。文字とインデックス値が対応していること。アルファベットは大文字で指定する。
 */
struct DigitParser(char[] digits) {
    
    static assert(digits.length < ubyte.max);
    
    /// 戻り値の型。
    template ResultType(I) {
        alias DigitParserResultType!(I) ResultType;
    }
    
    /// 数字。
    const char[] DIGITS = digits;
    
    /// 解析する。
    ResultType!(I) parse(I)(inout I i) {
        ResultType!(I) result;
        result.begin = i.clone;
        if(!i.hasMore) {
            return result;
        }
        
        int val = find(digits, std.ctype.toupper(i.next));
        if(val < 0) {
            i = result.begin.clone;
            return result;
        }
        
        result.length = 1;
        result.match = true;
        result.value = cast(ubyte) val;
        
        return result;
    }
}

/// 10進数字。
const DigitParser!(std.string.digits) digit;

/// 2進数字。
const DigitParser!("01") binDigit;

/// 8進数字。
const DigitParser!(std.string.octdigits) octDigit;

/// 16進数字。
const DigitParser!(std.string.hexdigits) hexDigit;

unittest {
    
    // 数字の読み取り。
    auto r1 = digit.parse(iterator("5"));
    assert(r1.match);
    assert(r1.length == 1);
    assert(r1.value == 5);
    
    // 不正な数字
    r1 = digit.parse(iterator("a"));
    assert(!r1.match);
    assert(r1.length == 0);
    assert(r1.value == 0);
    
    // 2進数字の読み取り。
    auto r2 = binDigit.parse(iterator("1"));
    assert(r2.match);
    assert(r2.length == 1);
    assert(r2.value == 1);
    
    r2 = binDigit.parse(iterator("0"));
    assert(r2.match);
    assert(r2.length == 1);
    assert(r2.value == 0);
    
    // 不正な2進数字
    r2 = binDigit.parse(iterator("2"));
    assert(!r2.match);
    assert(r2.length == 0);
    assert(r2.value == 0);
    
    // 8進数字の読み取り。
    auto r3 = octDigit.parse(iterator("7"));
    assert(r3.match);
    assert(r3.length == 1);
    assert(r3.value == 7);
    
    // 不正な8進数字
    r3 = octDigit.parse(iterator("8"));
    assert(!r3.match);
    assert(r3.length == 0);
    assert(r3.value == 0);
    
    // 16進数字の読み取り。
    auto r4 = hexDigit.parse(iterator("f"));
    assert(r4.match);
    assert(r4.length == 1);
    assert(r4.value == 0xf);
    
    r4 = hexDigit.parse(iterator("F"));
    assert(r4.match);
    assert(r4.length == 1);
    assert(r4.value == 0xf);
    
    // 不正な16進数字
    r4 = hexDigit.parse(iterator("g"));
    assert(!r4.match);
    assert(r4.length == 0);
    assert(r4.value == 0);
}

/// 数値パーサ戻り値の型。
struct NumberParserResultType(I) {
    I begin;        /// 開始位置。
    size_t length;  /// マッチした長さ。
    bool match;     /// マッチしたかどうか。
    ulong value;    /// 数値。
}

/** 数値パーサ。
 *  Params:
 *      D   = 数字解析用のパーサ。
 */
struct NumberParser(D) {
    /// 戻り値の型。
    template ResultType(I) {
        alias NumberParserResultType!(I) ResultType;
    }
    
    /// 解析する。
    ResultType!(I) parse(I)(inout I i) {
        ResultType!(I) result;
        D digit;
        
        result.begin = i.clone;
        
        for(D.ResultType!(I) r; (r = digit.parse(i)).match;) {
            result.match = r.match;
            result.length += r.length;
            result.value *= D.DIGITS.length;
            result.value += r.value;
        }
        return result;
    }
}

/// 10進数値。
const NumberParser!(DigitParser!(std.string.digits)) number;

/// 2進数値。
const NumberParser!(DigitParser!("01")) binNumber;

/// 8進数値。
const NumberParser!(DigitParser!(std.string.octdigits)) octNumber;

/// 16進数値。
const NumberParser!(DigitParser!(std.string.hexdigits)) hexNumber;

unittest {
    
    // 数値の読み取り。
    auto r1 = number.parse(iterator("12345"));
    assert(r1.match);
    assert(r1.length == 5);
    assert(r1.value == 12345);
    
    // 不正な数値
    r1 = number.parse(iterator("a123"));
    assert(!r1.match);
    assert(r1.length == 0);
    assert(r1.value == 0);
    
    // 2進数値の読み取り。    
    auto r2 = binNumber.parse(iterator("1010"));
    assert(r2.match);
    assert(r2.length == 4);
    assert(r2.value == 0b1010);
    
    // 不正な2進数値
    r2 = binNumber.parse(iterator("2110"));
    assert(!r2.match);
    assert(r2.length == 0);
    assert(r2.value == 0);
    
    // 8進数値の読み取り。
    auto r3 = octNumber.parse(iterator("7700"));
    assert(r3.match);
    assert(r3.length == 4);
    assert(r3.value == 07700);
    
    // 不正な8進数値
    r3 = octNumber.parse(iterator("8777"));
    assert(!r3.match);
    assert(r3.length == 0);
    assert(r3.value == 0);
    
    // 16進数値の読み取り。
    auto r4 = hexNumber.parse(iterator("ffff"));
    assert(r4.match);
    assert(r4.length == 4);
    assert(r4.value == 0xffff);
    
    r4 = hexNumber.parse(iterator("FFff"));
    assert(r4.match);
    assert(r4.length == 4);
    assert(r4.value == 0xffff);
    
    // 不正な16進数値
    r4 = hexNumber.parse(iterator("gffF"));
    assert(!r4.match);
    assert(r4.length == 0);
    assert(r4.value == 0);
}

/// 整数パーサ。
struct IntegerParser {
    
    private const char PLUS = '+';
    private const char MINUS = '-';
    private const char ZERO = '0';
    private const char BIN_PRIFIX = 'B';
    private const char HEX_PRIFIX = 'X';
    private const char UNSIGNED_POSTFIX = 'U';
    
    /// 戻り値の型。
    struct ResultType(I) {
        I begin;        /// 開始位置。
        size_t length;  /// マッチした長さ。
        bool match;     /// マッチしたかどうか。
        ulong value;    /// 数値。
        bool signed;    /// 符号の有無。
        bool negative;  /// 負の値かどうか。
    }
    
    /// 解析する。
    ResultType!(I) parse(I)(inout I i) {
        ResultType!(I) result;
        result.begin = i.clone;
        if(!i.hasMore) {
            return result;
        }
        
        I tmp;
        char c;
        
        // 現在位置の保持。
        void saveCurrent() {tmp = i.clone;}
        
        // 次の文字の取得。
        void getNext() {c = i.next;}
        
        // 現在位置を保持して次へ。
        void saveAndNext() {saveCurrent(); getNext();}
        
        // 以前の位置に戻る。
        void restore() {i = tmp.clone;}
        
        // 符号の読み取り。
        saveCurrent();
        getNext();
        if(c == PLUS || c == MINUS) {
            // 空白の読み飛ばし。
            size_t splen = space.parse(i).length;
            
            // 符号だけならエラー。
            if(!i.hasMore) {
                return result;
            }
            result.signed = true;
            result.negative = (c == MINUS);
            result.length += 1 + splen;
            
            // 次の文字へ。
            saveAndNext();
        }
        
        // 数値部分の解析結果。
        NumberParserResultType!(I) numResult;
        
        // プリフィックスを確かめる。
        if(c == ZERO) {
            // とりあえず0一文字マッチ。
            result.match = true;
            ++result.length;
        
            if(!i.hasMore) {
                // 0一文字。
                return result;
            }
            
            // 基数のチェック。
            saveAndNext();
            c = std.ctype.toupper(c);
            if(c == BIN_PRIFIX) {
                // 2進数。
                ++result.length;
                numResult = binNumber.parse(i);
            } else if(c == HEX_PRIFIX) {
                // 16進数。
                ++result.length;
                numResult = hexNumber.parse(i);
            } else {
                // 8進数。
                restore();
                numResult = octNumber.parse(i);
                if(!numResult.match) {
                    // 0一文字。
                    return result;
                }
            }
        } else {
            // 10進文字。
            restore();
            numResult = number.parse(i);
            
            // 10進数はデフォルトで符号付。
            if(numResult.match) {
                result.signed = true;
            }
        }
        
        // 数値部分の解析結果を反映。
        if(numResult.match) {
            result.match = true;
            result.value = numResult.value;
            result.length += numResult.length;
            
            // 符号なしポストフィックスの検査。
            if(i.hasMore) {
                saveAndNext();
                if(std.ctype.toupper(c) == UNSIGNED_POSTFIX) {
                    result.signed = false;
                    ++result.length;
                } else {
                    // 符号なしポストフィックスなし。
                    restore();
                }
            }
        } else {
            // マッチしなかった。
            result.match = false;
            result.length = 0;
            i = result.begin.clone;
        }
                
        return result;
    }
}

/// 整数パーサの生成。
const IntegerParser integer;

unittest {
    
    // 数値の読み取り。
    auto r = integer.parse(iterator("12345"));
    assert(r.match);
    assert(r.length == 5);
    assert(r.value == 12345);
    assert(r.signed);
    assert(!r.negative);
    
    r = integer.parse(iterator("12345u"));
    assert(r.match);
    assert(r.length == 6);
    assert(r.value == 12345U);
    assert(!r.signed);
    assert(!r.negative);
    
    r = integer.parse(iterator("12345U"));
    assert(r.match);
    assert(r.length == 6);
    assert(r.value == 12345U);
    assert(!r.signed);
    assert(!r.negative);
    
    r = integer.parse(iterator("-12345"));
    assert(r.match);
    assert(r.length == 6);
    assert(r.value == 12345);
    assert(r.signed);
    assert(r.negative);
    
    r = integer.parse(iterator("-12345U"));
    assert(r.match);
    assert(r.length == 7);
    assert(r.value == 12345U);
    assert(!r.signed);
    assert(r.negative);
    
    // 不正な数値
    r = integer.parse(iterator("a123"));
    assert(!r.match);
    assert(r.length == 0);
    assert(r.value == 0);
    assert(!r.signed);
    assert(!r.negative);
    
    // 2進数値の読み取り。    
    r = integer.parse(iterator("0b1010"));
    assert(r.match);
    assert(r.length == 6, format(r.length));
    assert(r.value == 0b1010);
    assert(!r.signed);
    assert(!r.negative);
    
    r = integer.parse(iterator("+0b1010"));
    assert(r.match);
    assert(r.length == 7, format(r.length));
    assert(r.value == 0b1010);
    assert(r.signed);
    assert(!r.negative);
    
    r = integer.parse(iterator("-0b1010"));
    assert(r.match);
    assert(r.length == 7);
    assert(r.value == 0b1010);
    assert(r.signed);
    assert(r.negative);
    
    // 不正な2進数値
    r = integer.parse(iterator("0b2110"));
    assert(!r.match);
    assert(r.length == 0);
    assert(r.value == 0);
    assert(!r.signed);
    assert(!r.negative);
    
    // 8進数値の読み取り。
    r = integer.parse(iterator("07700"));
    assert(r.match);
    assert(r.length == 5);
    assert(r.value == 07700);
    assert(!r.signed);
    assert(!r.negative);
    
    r = integer.parse(iterator("+07700"));
    assert(r.match);
    assert(r.length == 6);
    assert(r.value == 07700);
    assert(r.signed);
    assert(!r.negative);
    
    r = integer.parse(iterator("-07700"));
    assert(r.match);
    assert(r.length == 6);
    assert(r.value == 07700);
    assert(r.signed);
    assert(r.negative);
    
    // 不正な8進数値
    r = integer.parse(iterator("08777"));
    assert(r.match);
    assert(r.length == 1);
    assert(r.value == 0);
    assert(!r.signed);
    assert(!r.negative);
    
    // 16進数値の読み取り。
    r = integer.parse(iterator("0xffff"));
    assert(r.match);
    assert(r.length == 6);
    assert(r.value == 0xffff);
    assert(!r.signed);
    assert(!r.negative);
    
    r = integer.parse(iterator("+0xffff"));
    assert(r.match);
    assert(r.length == 7);
    assert(r.value == 0xffff);
    assert(r.signed);
    assert(!r.negative);
    
    r = integer.parse(iterator("-0xffff"));
    assert(r.match);
    assert(r.length == 7);
    assert(r.value == 0xffff);
    assert(r.signed);
    assert(r.negative);
    
    r = integer.parse(iterator("0xFFff"));
    assert(r.match);
    assert(r.length == 6);
    assert(r.value == 0xffff);
    assert(!r.signed);
    assert(!r.negative);
    
    // 不正な16進数値
    r = integer.parse(iterator("0xgffF"));
    assert(!r.match);
    assert(r.length == 0);
    assert(r.value == 0);
    assert(!r.signed);
    assert(!r.negative);
}
