﻿/**
 *	連結リストのモジュール。
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module outland.tl.list;

/// 両方向連結リストのクラス。
class List(T) {
	
	/// 反復子。
	struct Iterator {
		
		/// コンストラクタ。
		static Iterator opCall(List list)
		in {
			assert(list);
		} body {
			return opCall(list.begin_);
		}
		
		/// コンストラクタ。
		static Iterator opCall(Entry* entry) {
			Iterator iter;
			iter.entry_ = entry;
			return iter;
		}
		
		/// 反復子のクローンを新たに生成する。
		Iterator clone() {return *this;}
		
		/// 次の要素を得る。
		T next() {
			Entry* next = entry_.next;
			T val = entry_.value;
			entry_.next = next;
			return val;
		}
		
		/// 取り出せる要素があるか返す。
		bool hasMore() {return entry_ !is null;}
		
	private:
		
		/// エントリ。
		Entry* entry_;
	}
	
	/// デフォルトコンストラクタ。
	this() {}
	
	/// コピーコンストラクタ。
	this(List src)
	in {
		assert(src is null);
	} body {
		this();
		foreach(e; src) pushBack(e);
	}
	
	/// デストラクタ。
	~this() {clear();}
	
	/// 全要素の破棄。
	void clear() {
		if(begin_) {
			// 末尾と開始を切断。
			begin_.prev.next = null;
			
			// 全要素を破棄。
			for(Entry* e = begin_; e;) {
				Entry* next = e.next;
				deallocate(e);
				e = next;
			}
			begin_ = null;
			length_ = 0;
		}
	}
	
	/// サイズを得る。
	size_t length() {return length_;}
	
	/// 先頭に要素を追加する。
	void pushFront(T value) {
		// 末尾に追加。
		pushBack(value);
		// 環状リストを回転させて追加された末尾を先頭にする。
		begin_ = begin_.prev;
	}
	
	/// 先頭の要素を削除する。
	void popFront() {
		if(begin_) {
			Entry* front = begin_;
			
			// 最後の要素だった場合はbeginも破棄。
			// 要素が残っている場合はポインタを付け替える。
			if(front.next == begin_) {
				begin_ = null;
			} else {
				front.next.prev = front.prev;
				front.prev.next = front.next;
				begin_ = front.next;
			}
			deallocate(front);
			--length_;
		}
	}
	
	/// 末尾に要素を追加する。
	void pushBack(T value) {
		Entry* entry = allocate();
		entry.value = value;
		if(begin_) {
			Entry* before = begin_.prev;
			begin_.prev = entry;
			before.next = entry;
			entry.next = begin_;
			entry.prev = before;
		} else {
			// 最初の要素だった場合。
			entry.next = entry;
			entry.prev = entry;
			begin_ = entry;
		}
		++length_;
	}
	
	/// 末尾の要素を削除する。
	void popBack() {
		if(begin_) {
			Entry* back = begin_.prev;
			
			// 最後の要素だった場合はbeginも破棄。
			// 要素が残っている場合はポインタを付け替える。
			if(back == begin_) {
				begin_ = null;
			} else {
				back.next.prev = back.prev;
				back.prev.next = back.next;
			}
			deallocate(back);
			--length_;
		}
	}
	
	/// 先頭の要素を得る。
	T front()
	in {
		assert(begin_ !is null);
	} body {
		return begin_.value;
	}
	
	/// 末尾の要素を得る。
	T back()
	in {
		assert(begin_ !is null && begin_.prev !is null);
	} body {
		return begin_.prev.value;
	}
	
	/// 巡回する。
	int opApply(int delegate(inout T) dg) {
		if(Entry* e = begin_) {
			for(;; e = e.next) {
				if(int result = dg(e.value)) return result;
				if(e.next is begin_) break;
			}
		}
		return 0;
	}
	
	/// 巡回する。
	int opApply(int delegate(inout size_t, inout T) dg) {
		if(Entry* e = begin_) {
			for(size_t i = 0; ; ++i, e = e.next) {
				if(int result = dg(i, e.value)) return result;
				if(e.next is begin_) break;
			}
		}
		return 0;
	}

	/// 逆順に巡回する。
	int opApplyReverse(int delegate(inout T) dg) {
		if(begin_) {
			for(Entry* e = begin_.prev; ; e = e.prev) {
				if(int result = dg(e.value)) return result;
				if(e is begin_) break;
			}
		}
		return 0;
	}

	
private:
	
	/// リストのエントリ。
	struct Entry {
		Entry* next;
		Entry* prev;
		T value;
	}
	
	/// エントリの新規作成。
	/// いずれオブジェクトプールで実装するかもしれない。
	static Entry* allocate() {return new Entry;}
	
	/// エントリの破棄。
	/// いずれオブジェクトプールで実装するかもしれない。
	static void deallocate(Entry* e) {delete e;}
	
	/// 始点。
	Entry* begin_;
	
	/// 要素数。
	size_t length_;
}

unittest {
	auto list = new List!(int);
	
	list.pushBack(1);
	list.pushBack(2);
	list.pushBack(3);
	
	assert(list.front == 1);
	list.popFront();
	assert(list.front == 2);
	list.popFront();
	assert(list.front == 3);
	list.popFront();
	assert(list.length == 0);
	
	list.pushFront(1);
	list.pushFront(2);
	list.pushFront(3);
	
	assert(list.back == 1);
	list.popBack();
	assert(list.back == 2);
	list.popBack();
	assert(list.back == 3);
	list.popBack();
	assert(list.length == 0);
	
	list.pushFront(1);	// 1
	list.pushBack(2);	// 12
	list.pushFront(3);	// 312
	list.pushBack(4);	// 3124
	list.pushFront(5);	// 53124
	list.pushBack(6);	// 531246
	
	// 531246
	assert(list.front == 5);
	assert(list.back == 6);
	assert(list.length == 6);
	
	// 53124
	list.popBack();
	assert(list.front == 5);
	assert(list.back == 4);
	assert(list.length == 5);
	
	// 3124
	list.popFront();
	assert(list.front == 3);
	assert(list.back == 4);
	assert(list.length == 4);
	
	// 312
	list.popBack();
	assert(list.front == 3);
	assert(list.back == 2);
	assert(list.length == 3);
	
	// 12
	list.popFront();
	assert(list.front == 1);
	assert(list.back == 2);
	assert(list.length == 2);
	
	// 1
	list.popBack();
	assert(list.front == 1);
	assert(list.back == 1);
	assert(list.length == 1);
	
	// 空
	list.popFront();
	assert(list.length == 0);
	
	list.pushBack(list.length);
	list.pushBack(list.length);
	list.pushBack(list.length);
	list.pushBack(list.length);
	list.pushBack(list.length);
	list.pushBack(list.length);
	foreach(i, e; list) assert(i == e);

	list.clear();
	assert(list.length == 0);
	foreach(e; list) {assert(false);}
	list.pushFront(999);
	foreach(e; list) {assert(e == 999);}
	foreach_reverse(e; list) {assert(e == 999);}
}
