﻿module outland.widget.win32.frame;

import std.stdio;

import win32.windows;

import outland.widget.application;
import outland.widget.error;
import outland.widget.event;
import outland.widget.graphics;
import outland.widget.keyboard;
import outland.widget.mouse;
import outland.widget.shape;
import outland.widget.widget;

import outland.widget.win32.error;
import outland.widget.win32.graphics;
import outland.widget.win32.string;

/// Win32フレームクラス。
class WinFrame : IFrame {
	
	// イベントハンドラマップの取り込み。
	mixin MEventHandlerMap;
	
	// イベントIDプロパティの取り込み。
	mixin MEventIdProperty;
	
	// 子ウィジェット管理機能の取り込み。
	mixin MWidgetContainer;
	
	/// ウィンドウクラス名。
	private const TCHAR[] WINDOW_CLASS_NAME = cast(TCHAR[]) "class_outland_widget_win32_frame";
	
	/// ウィンドウクラス。
	const WNDCLASSEX WINDOW_CLASS = {
		cbSize			: WNDCLASSEX.sizeof,
		style			: CS_DBLCLKS | CS_HREDRAW | CS_VREDRAW,
		lpfnWndProc		: &callbackProcedure,
		lpszClassName	: WINDOW_CLASS_NAME.ptr,
		hbrBackground	: cast(HBRUSH)(COLOR_WINDOW + 1),
	};
	
	/// ウィンドウクラス登録。
	static this() {
		WNDCLASSEX cls = WINDOW_CLASS;
		cls.hInstance = GetModuleHandle(null);
		if(!cls.hInstance || !RegisterClassEx(&cls)) throw new Win32Exception;
	}
	
	/// 矩形を指定して生成する。
	this(Rect rect) {
		auto t = delegateToEventTarget(&dispatchMouseEvent);
		foreach(code; [EVENT_CODE_MOUSE_MOVE,
					   EVENT_CODE_MOUSE_DOWN,
					   EVENT_CODE_MOUSE_UP,
					   EVENT_CODE_MOUSE_DOUBLE_CLICK,
					   EVENT_CODE_MOUSE_WHEEL]) {
			addTargetAny(code, t);
		}
		
		// ウィンドウ生成。
		if(!CreateWindowEx(WS_EX_APPWINDOW | WS_EX_ACCEPTFILES,
						   WINDOW_CLASS_NAME.ptr,
						   toTString(title_),
						   WS_POPUP,
						   rect.x,
						   rect.y,
						   rect.width,
						   rect.height,
						   cast(HANDLE) null,
						   cast(HANDLE) null,
						   cast(HINSTANCE) GetModuleHandle(null),
						   cast(void*) this)) {
			throw new Win32Exception;
		}
	}
	
	/// 位置。
	Point position() {
		RECT rect;
		GetWindowRect(wnd_, &rect);
		return Point(rect.left, rect.top);
	}
	
	/// サイズ。
	Size size() {
		RECT rect;
		GetWindowRect(wnd_, &rect);
		return Size(rect.right - rect.left, rect.bottom - rect.top);
	}
	
	/// 矩形。
	Rect rect() {
		RECT rect;
		GetWindowRect(wnd_, &rect);
		return Rect(Point(rect.left, rect.top), Size(rect.right - rect.left, rect.bottom - rect.top));
	}
	
	// レイアウト用のサイズプロパティ取り込み。
	mixin MLayoutSizeProperty;
	
	/// 位置。(ローカル座標)
	void position(Point p) {
		if(!SetWindowPos(wnd_, null, p.x, p.y, 0, 0, SWP_NOZORDER | SWP_NOACTIVATE | SWP_NOSIZE)) {
			throw new Win32Exception;
		}
	}
	
	/// サイズ。
	void size(Size s) {
		if(!SetWindowPos(wnd_, null, 0, 0, s.width, s.height, SWP_NOZORDER | SWP_NOACTIVATE | SWP_NOMOVE)) {
			throw new Win32Exception;
		}
	}
	
	/// 矩形。
	void rect(Rect r) {
		if(!SetWindowPos(wnd_, null, r.x, r.y, r.width, r.height, SWP_NOZORDER | SWP_NOACTIVATE)) {
			throw new Win32Exception;
		}
	}
	
	/// クライアント座標をスクリーン座標にする。
	Point clientToScreen(Point p) {
		POINT pos;
		pos.x = p.x;
		pos.y = p.y;
		if(!ClientToScreen(wnd_, &pos)) throw new Win32Exception;
		return Point(pos.x, pos.y);
	}
	
	/// スクリーン座標をクライアント座標にする。
	Point screenToClient(Point p) {
		POINT pos;
		pos.x = p.x;
		pos.y = p.y;
		if(!ScreenToClient(wnd_, &pos)) throw new Win32Exception;
		return Point(pos.x, pos.y);
	}
	
	/// 表示されているかどうか。
	bool visible() {return IsWindowVisible(wnd_) != FALSE;}
	
	/// ditto
	void visible(bool b) {ShowWindow(wnd_, (b) ? SW_SHOW : SW_HIDE);}
	
	/// 有効かどうか。
	bool enable() {return IsWindowEnabled(wnd_) != FALSE;}
	
	/// ditto
	void enable(bool e) {
		if(!EnableWindow(wnd_, e ? TRUE : FALSE)) throw new Win32Exception;
	}
	
	/// 親ウィジェット。
	IWidget parent() {return null;}
	
	/// ditto
	void parent(IWidget p) {throw new NotSupportedException;}
	
	/// フレーム。
	IFrame frame() {return this;}
	
	/// フォーカスされているウィジェット。
	IWidget keyboardFocus() {return focus_;}
	
	/// ditto
	void keyboardFocus(IWidget w) {
		IWidget old = focus_;
		focus_ = null;
		
		scope unfocusEvent = new KeyUnfocusEvent(old, w);
		handle(unfocusEvent);
		
		focus_ = w;
		scope focusEvent = new KeyFocusEvent(old, w);
		handle(focusEvent);
	}
	
	/// マウス入力を受け取るウィジェット。
	IWidget mouseCapture() {
	}
	
	/// ditto
	void mouseCapture(IWidget w) {
	}
	
	/// タイトル表示有無。
	bool showTitle() {return showTitle_;}
	
	/// ditto
	void showTitle(bool b) {showTitle_ = b;}
	
	/// タイトル文字列。
	char[] title() {return title_;}
	
	/// ditto
	void title(char[] str) {
		if(!SetWindowText(wnd_, toTString(str))) throw new Win32Exception;
		title_ = str.dup;
	}
	
	/// 最大化有無。
	void maximizeable(bool b) {maximizeable_ = b;}
	
	/// ditto
	bool maximizeable() {return maximizeable_;}
	
	/// 最大化。
	void maximize(bool b) {
		if(!ShowWindow(wnd_, b ? SW_MAXIMIZE : SW_RESTORE)) throw new Win32Exception;
	}
	
	/// ditto
	bool maximize() {return IsZoomed(wnd_) != FALSE;}
	
	/// 最小化。
	void minimize(bool b) {
		if(!ShowWindow(wnd_, b ? SW_MINIMIZE : SW_RESTORE)) throw new Win32Exception;
	}
	
	/// ditto
	bool minimize() {return IsIconic(wnd_) != FALSE;}
	
	/// 最小化有無。
	void minimizeable(bool b) {minimizeable_ = b;}
	
	/// ditto
	bool minimizeable() {return minimizeable_;}
	
	/// サイズ変更有無。
	void resizeable(bool b) {resizeable_ = b;}
	
	/// ditto
	bool resizeable() {return resizeable_;}
	
	/// 閉じるボタン有無。
	void closeable(bool b) {closeable_ = b;}
	
	/// ditto
	bool closeable() {return closeable_;}
	
	/// 閉じられた時に終了するかどうか。
	void quitOnClose(bool b) {quitOnClose_ = b;}
	
	/// ditto
	bool quitOnClose() {return quitOnClose_;}
	
	/// 再描画領域の指定。
	void update(Rect r) {
		RECT rect;
		rect.left = r.left;
		rect.top = r.top;
		rect.right = r.right;
		rect.bottom = r.bottom;
		if(!InvalidateRect(wnd_, &rect, TRUE)) throw new Win32Exception;
	}
	
	/// 再描画。
	void update() {
		if(!InvalidateRect(wnd_, null, TRUE)) throw new Win32Exception;
	}
	
	/// グラフィックスに対する処理。
	void duringGraphics(void delegate(IGraphics g) dg) {
		auto dc = GetDC(wnd_);
		if(!dc) throw new Win32Exception;
		auto g = new WinGraphics(dc, (HDC dc) {ReleaseDC(wnd_, dc);});
		g.duringBackup({dg(g);});
	}
	
	/// 破棄を行う。
	void close() {DestroyWindow(wnd_);}
	
private:
	
	/// 登録用ウィンドウプロシージャ。
	extern(Windows) static LRESULT callbackProcedure(HWND wnd, UINT msg, WPARAM wp, LPARAM lp) {
		// 生成時。
		if(msg == WM_CREATE) {
			if(LPCREATESTRUCT cs = cast(LPCREATESTRUCT) lp) {
				if(WinFrame frame = cast(WinFrame) cs.lpCreateParams) {
					SetLastError(ERROR_SUCCESS);
					SetWindowLongPtr(wnd, GWLP_USERDATA, cast(ULONG_PTR) cast(void*) frame);
					if(GetLastError() != ERROR_SUCCESS) throw new Win32Exception;
					frame.wnd_ = wnd;
				}
			}
		}
		
		if(void* p = cast(void*) GetWindowLongPtr(wnd, GWLP_USERDATA)) {
			WinFrame frame = cast(WinFrame) p;
			assert(frame);
			return frame.procedure(wnd, msg, wp, lp);
		}
		return DefWindowProc(wnd, msg, wp, lp);
	}
	
	/// WPARAMをマウス状態にする。
	MouseState wParamToMouseState(WPARAM wp) {
		MouseState result;
		if(wp & MK_CONTROL)	result |= MOUSE_CTRL;
		if(wp & MK_SHIFT)	result |= MOUSE_SHIFT;
		if(wp & MK_LBUTTON)	result |= MOUSE_BUTTON1;
		if(wp & MK_MBUTTON)	result |= MOUSE_BUTTON2;
		if(wp & MK_RBUTTON)	result |= MOUSE_BUTTON3;
		return result;
	}
	
	/// LPARAMを座標にする。
	Point lParamToPoint(LPARAM lp) {return Point(LOWORD(lp), HIWORD(lp));}
	
	/// ウィンドウプロシージャ。
	LRESULT procedure(HWND wnd, UINT msg, WPARAM wp, LPARAM lp) {
		switch(msg) {
		case WM_PAINT: {
				RECT rect;
				if(GetUpdateRect(wnd_, &rect, TRUE)) {
					PAINTSTRUCT ps;
					HDC dc = BeginPaint(wnd_, &ps);
					scope(exit) EndPaint(wnd_, &ps);
					scope g = new WinGraphics(dc);
					draw(g);
					return 0;
				}
			}
			break;
		case WM_MOUSEMOVE: {
				scope e = new MouseMoveEvent(lParamToPoint(lp), wParamToMouseState(wp));
				handle(e);
			}
			return 0;
		case WM_LBUTTONDOWN: {
				scope e = new MouseDownEvent(lParamToPoint(lp), MOUSE_BUTTON1, wParamToMouseState(wp));
				handle(e);
			}
			return 0;
		case WM_LBUTTONUP: {
				scope e = new MouseUpEvent(lParamToPoint(lp), MOUSE_BUTTON1, wParamToMouseState(wp));
				handle(e);
			}
			return 0;
		case WM_LBUTTONDBLCLK: {
				scope e = new MouseDoubleClickEvent(lParamToPoint(lp), MOUSE_BUTTON1, wParamToMouseState(wp));
				handle(e);
			}
			return 0;
		case WM_RBUTTONDOWN: {
				scope e = new MouseDownEvent(lParamToPoint(lp), MOUSE_BUTTON2, wParamToMouseState(wp));
				handle(e);
			}
			return 0;
		case WM_RBUTTONUP: {
				scope e = new MouseUpEvent(lParamToPoint(lp), MOUSE_BUTTON2, wParamToMouseState(wp));
				handle(e);
			}
			return 0;
		case WM_RBUTTONDBLCLK: {
				scope e = new MouseDoubleClickEvent(lParamToPoint(lp), MOUSE_BUTTON2, wParamToMouseState(wp));
				handle(e);
			}
			return 0;
		case WM_MBUTTONDOWN: {
				scope e = new MouseDownEvent(lParamToPoint(lp), MOUSE_BUTTON3, wParamToMouseState(wp));
				handle(e);
			}
			return 0;
		case WM_MBUTTONUP: {
				scope e = new MouseUpEvent(lParamToPoint(lp), MOUSE_BUTTON3, wParamToMouseState(wp));
				handle(e);
			}
			return 0;
		case WM_MBUTTONDBLCLK: {
				scope e = new MouseDoubleClickEvent(lParamToPoint(lp), MOUSE_BUTTON3, wParamToMouseState(wp));
				handle(e);
			}
			return 0;
		case WM_MOUSEWHEEL: {
				scope e = new MouseWheelEvent(lParamToPoint(lp), wParamToMouseState(LOWORD(wp)), HIWORD(wp) * WHEEL_DELTA);
				handle(e);
			}
			return 0;
		case WM_CLOSE:
			DestroyWindow(wnd_);
			break;
		case WM_DESTROY:
			if(quitOnClose_) getApplication.quit(0);
			break;
		default:
			break;
		}
		return DefWindowProc(wnd, msg, wp, lp);
	}
	
	/// マウス移動イベント。
	bool dispatchMouseEvent(MouseEvent e) {
		foreach_reverse(c; this) if(e.position in c.rect) return c.handle(e);
		return false;
	}
	
	/// ネイティブなウィンドウハンドル。
	HWND wnd_;
	
	/// フォーカスウィジェット。
	IWidget focus_;
	
	/// タイトル表示有無。
	bool showTitle_;
	
	/// タイトル。
	char[] title_;
	
	/// 最大化有無。
	bool maximizeable_;
	
	/// 最小化有無。
	bool minimizeable_;
	
	/// サイズ変更可能かどうか。
	bool resizeable_;
	
	/// 閉じるボタン有無。
	bool closeable_;
	
	/// 終了時にアプリケーションを終了するか。
	bool quitOnClose_;
}
