﻿/**
 *   PEGパーサモジュール。
 *
 *   Version:
 *       $Revision$
 *   Date:
 *       $Date$
 *   License:
 *       MIT/X Consortium License
 *   History:
 *       $Log$
 */

module outland.ctfe.peg.parser;

/// ノード構造体。
struct Node {
    string type;        /// ノードの型。
    string value;       /// ノードの値。
    size_t line;        /// 行番号。(0開始)
    Node[] children;    /// 子ノード。
}

/// エラー構造体。
struct ParseError {
    string description; /// 説明。
    size_t line;        /// 行番号。(0開始)
}

/// 解析結果構造体。
struct ParseResult {
    bool match;             /// マッチしたかどうか。
    string rest;            /// 解析できなかった残り。
    Node[] nodes;           /// 生成されたノード。
    ParseError[] errors;    /// 発生したエラー。
    size_t nextLine;        /// 次の行番号。
}

/// ノードを文字列に出力する。
string dumpNode(Node node, size_t nest = 0) {
    string result;
    for(size_t i = 0; i < nest; ++i) {
        result ~= "  ";
    }
    
    result ~= node.type ~ ":" ~ node.value ~ "\n";
    
    for(size_t i = 0; i < node.children.length; ++i) {
        result ~= dumpNode(node.children[i], nest + 1);
    }
    
    return result;
}


/// PEGパーサの生成。
string compilePegParser(string src) {
    auto r = parsePeg(src, 0);
    assert(r.match, "Invalid PEG source!");
    assert(r.nodes.length == 1, "Invalid PEG source!");
    assert(r.nodes[0].type == PEG_ROOT, "Invalid PEG source!");
    
    /// パーサ情報。
    struct Parser {
        bool noSkip = false;
        bool makeNode = false;
        bool text = false;
        bool newLine = false;
        bool always = false;
        bool skipper = false;
        string id;
        Node content;
    }
    
    Parser[] parsers;
    string skipParser;
    auto defs = r.nodes[0].children;
    
    foreach(d; defs) {
        assert(d.type == PEG_DEFINE, "Invalid PEG source!");
        assert(d.children.length == 2, "Invalid PEG source!");
        assert(d.children[0].type == PEG_DECLARE, "Invalid PEG source!");
        
        Parser p;

        auto decl = d.children[0].children;
        foreach(n; decl) {
            switch(n.type) {
            case PEG_DECLARE_SKIPPER:
                p.skipper = true;
                break;
            case PEG_DECLARE_NO_SKIP:
                p.noSkip = true;
                break;
            case PEG_DECLARE_MAKE_NODE:
                p.makeNode = true;
                break;
            case PEG_DECLARE_TEXT_NODE:
                p.text = true;
                p.makeNode = true;
                break;
            case PEG_DECLARE_NEW_LINE:
                p.newLine = true;
                break;
            case PEG_DECLARE_ALWAYS:
                p.always = true;
                break;
            case PEG_IDENTIFIER:
                p.id = n.value;
                break;
            default:
                break;
            }
        }

        assert(p.id.length > 0, "Invalid parser definition!");
        if(p.skipper) {
            skipParser = p.id;
        }
        p.content = d.children[1];
        
        parsers ~= p;
    }
    
    string result;
    foreach(p; parsers) {
        result ~= makeParserDefineCode(
            p.id, p.noSkip, p.makeNode, p.text, p.newLine, p.always, p.content, (p.noSkip ? "" : skipParser));
    }
    
    return result;
}

/// 1文字にマッチする。
ParseResult parseAnyChar(string src, size_t line) {
    ParseResult result;
    result.rest = src;
    
    if(src.length > 0) {
        result.match = true;
        result.rest = src[1 .. $];
        result.nextLine = line;
    }
    
    return result;
}

/// 文字列にマッチする。
ParseResult parseString(string val)(string src, size_t line) {
    ParseResult result;
    result.rest = src;
    
    if(src.length >= val.length && src[0 .. val.length] == val) {
        result.match = true;
        result.rest = src[val.length .. $];
        result.nextLine = line;
    }
    
    return result;
}

/// 文字範囲にマッチする。
ParseResult parseCharRange(char low, char high)(string src, size_t line) {
    ParseResult result;
    result.rest = src;
    
    if(src.length > 0 && low <= src[0] && src[0] <= high) {
        result.match = true;
        result.rest = src[1 .. $];
        result.nextLine = line;
    }
    
    return result;
}

/// エラーを生成する。
ParseResult parseError(string desc)(string src, size_t line) {
    ParseResult result;
    result.rest = src;
    result.nextLine = line;
    result.match = true;
    
    ParseError err;
    err.description = desc;
    err.line = line;
    
    result.errors ~= err;
    
    return result;
}

/// 行番号を加算する。
ParseResult parseAddLine(string src, size_t line) {
    ParseResult result;
    result.rest = src;
    result.nextLine = line + 1;
    result.match = true;
    
    return result;
}

/// ソース終端。
ParseResult parseEof(string src, size_t line) {
    ParseResult result;
    result.rest = src;
    result.nextLine = line;
    result.match = (src.length == 0);
    
    return result;
}

/// オプション。
ParseResult parseOption(alias P)(string src, size_t line) {
    auto result = P(src, line);
    result.match = true;
    return result;
}

/// 0以上。
ParseResult parseZeroOrMore(alias P)(string src, size_t line) {
    ParseResult result;
    result.rest = src;
    result.match = true;
    result.nextLine = line;
    
    while(true) {
        auto r = P(result.rest, line);
        result.rest = r.rest;
        if(r.match) {
            for(size_t i = 0; i < r.nodes.length; ++i) {
                result.nodes ~= r.nodes[i];
            }
            for(size_t i = 0; i < r.errors.length; ++i) {
                result.errors ~= r.errors[i];
            }
            result.nextLine = r.nextLine;
        } else {
            break;
        }
    }
    return result;
}

/// 1以上。
ParseResult parseOneOrMore(alias P)(string src, size_t line) {
    ParseResult result;
    result.rest = src;
    
    while(true) {
        auto r = P(result.rest, line);
        result.rest = r.rest;
        if(r.match) {
            for(size_t i = 0; i < r.nodes.length; ++i) {
                result.nodes ~= r.nodes[i];
            }
            for(size_t i = 0; i < r.errors.length; ++i) {
                result.errors ~= r.errors[i];
            }
            result.nextLine = r.nextLine;
            result.match = true;
        } else {
            break;
        }
    }
    return result;
}

/// andテスト。
ParseResult parseAndTest(alias P)(string src, size_t line) {
    ParseResult result;
    result.rest = src;
    result.match = P(src, line).match;
    result.nextLine = line;
    return result;
}

/// notテスト。
ParseResult parseNotTest(alias P)(string src, size_t line) {
    ParseResult result;
    result.rest = src;
    result.match = !P(src, line).match;
    result.nextLine = line;
    return result;
}

/// 連接。
ParseResult parseSequence(P...)(string src, size_t line) {
    ParseResult result;
    result.rest = src;
    result.match = true;
    result.nextLine = line;
    
    foreach(p; P) {
        auto r = p(result.rest, result.nextLine);
        if(!r.match) {
            ParseResult noMatch;
            noMatch.rest = src;
            return noMatch;
        } else {
            for(size_t i = 0; i < r.nodes.length; ++i) {
                result.nodes ~= r.nodes[i];
            }
            for(size_t i = 0; i < r.errors.length; ++i) {
                result.errors ~= r.errors[i];
            }
            result.nextLine = r.nextLine;
            result.rest = r.rest;
        }
    }
    
    return result;
}

/// 選択。
ParseResult parseChoice(P...)(string src, size_t line) {
    foreach(p; P) {
        auto r = p(src, line);
        if(r.match) {
            return r;
        }
    }
    
    ParseResult result;
    result.rest = src;
    return result;
}

/// ノード生成。
ParseResult parseNode(string type, bool text, bool makeAlways, alias P)(string src, size_t line) {
    ParseResult result;
    result.rest = src;
    
    auto r = P(src, line);
    if(r.match) {
        // 常時生成ではない場合、子ノードが1つしかない時はノードを生成せずに子ノードを直接返す。
        static if(!makeAlways) {
            if(r.nodes.length == 1) {
                return r;
            }
        }
        
        result = r;
        
        Node node;
        node.type = type;
        if(text) {
            node.value = src[0 .. ($ - r.rest.length)];
        } else {
            node.children = r.nodes;
        }
        
        result.nodes = (Node[]).init;
        result.nodes ~= node;
    }
    
    return result; 
}

/// スキップ用パーサ。
ParseResult parseSkip(alias S, alias P)(string src, size_t line) {
    auto r = S(src, line);
    if(r.match) {
        return P(r.rest, r.nextLine);
    } else {
        return P(src, line);
    }
}

/// 任意文字演算子の型。
const string PEG_ANY_CHAR = "ANY_CHAR";

/// 文字列演算子の型。
const string PEG_STRING = "STRING";

/// 文字範囲演算子の型。
const string PEG_RANGE = "RANGE";

/// 文字範囲演算子の文字の型。
const string PEG_RANGE_CHAR = "RANGE_CHAR";

/// エラー演算子の型。
const string PEG_ERROR = "ERROR";

/// 識別子の型。
const string PEG_IDENTIFIER = "IDENTIFIER";

/// オプション演算子。
const string PEG_OPTION_OP = "OPTION_OP";

/// 0以上演算子。
const string PEG_ZERO_OR_MORE_OP = "ZERO_OR_MORE_OP";

/// 1以上演算子。
const string PEG_ONE_OR_MORE_OP = "ONE_OR_MORE_OP";

/// 繰り返し演算子。
const string PEG_REPEAT = "REPEAT";

/// ANDテスト演算子。
const string PEG_AND_TEST_OP = "AND_TEST_OP";

/// NOTテスト演算子。
const string PEG_NOT_TEST_OP = "NOT_TEST_OP";

/// テスト演算子。
const string PEG_TEST = "TEST";

/// 連接式。
const string PEG_SEQUENCE = "SEQUENCE";

/// 選択式。
const string PEG_CHOICE = "CHOICE";

/// 構文規則宣言。
const string PEG_DECLARE = "DECLARE";

/// スキッパー宣言。
const string PEG_DECLARE_SKIPPER = "SKIPPER";

/// 非スキップ宣言。
const string PEG_DECLARE_NO_SKIP = "NO_SKIP";

/// ノード生成宣言。
const string PEG_DECLARE_MAKE_NODE = "MAKE_NODE";

/// テキストノード構文規則宣言。
const string PEG_DECLARE_TEXT_NODE = "TEXT_NODE";

/// 改行構文規則宣言。
const string PEG_DECLARE_NEW_LINE = "NEW_LINE";

/// 常時ノード生成宣言。
const string PEG_DECLARE_ALWAYS = "ALWAYS";

/// 構文規則の定義。
const string PEG_DEFINE = "DEFINE";

/// PEGのルートノード。
const string PEG_ROOT = "PEG_ROOT";

private:

/// UTF8BOM。
ParseResult parsePegUtf8Bom(string src, size_t line) {
    return parseString!("\xEF\xBB\xBF")(src, line);
}

/// 空白。
ParseResult parsePegSpace(string src, size_t line) {
    return parseZeroOrMore!(
        parseChoice!(
            parseOneOrMore!(parseString!("\t")),
            parseOneOrMore!(parseString!("\v")),
            parseOneOrMore!(parseString!(" ")),
            parseOneOrMore!(parsePegNewline),
            parseSequence!(
                parseAndTest!(parseString!("/")),
                parseChoice!(
                    parsePegLineComment,
                    parsePegBlockComment
                )
            )
        )
    )(src, line);
}

/// PEGの式。
ParseResult parsePegExp(string src, size_t line) {
    return parsePegChoice(src, line);
}

/// PEGの任意文字演算子。
ParseResult parsePegAnyChar(string src, size_t line) {
    return parseNode!(PEG_ANY_CHAR, false, true, parseString!("."))(src, line);
}

/// PEGの識別子。
ParseResult parsePegIdentifier(string src, size_t line) {
    return parseNode!(PEG_IDENTIFIER, true, true,
        parseSequence!(
            parseChoice!(parseCharRange!('a', 'z'), parseCharRange!('A', 'Z'), parseString!("_")),
            parseZeroOrMore!(
                parseChoice!(parseCharRange!('a', 'z'), parseCharRange!('A', 'Z'), parseCharRange!('0', '9'), parseString!("_"))
            )
        )
    )(src, line);
}

/// PEGのリテラル文字列。
ParseResult parsePegString(string src, size_t line) {
    return parseNode!(PEG_STRING, true, true,
        parseSequence!(
            parseString!("\""),
            parseZeroOrMore!(
                parseChoice!(
                    parseSequence!(parseString!("\\"), parseAnyChar),
                    parseSequence!(parseNotTest!(parseString!("\"")), parseAnyChar)
                )
            ),
            parseString!("\"")
        )
    )(src, line);
}

/// PEGのリテラル文字。
ParseResult parsePegChar(string src, size_t line) {
    return parseNode!(PEG_RANGE_CHAR, true, true,
        parseSequence!(
            parseString!("'"),
            parseOneOrMore!(
                parseChoice!(
                    parseSequence!(parseString!("\\"), parseAnyChar),
                    parseSequence!(parseNotTest!(parseString!("'")), parseAnyChar)
                )
            ),
            parseString!("'")
        )
    )(src, line);
}

/// 文字範囲演算子。
ParseResult parsePegCharRange(string src, size_t line) {
    return parseNode!(PEG_RANGE, false, true,
        parseSequence!(
            parseString!("["),
            parsePegSpace,
            parsePegChar,
            parsePegSpace,
            parseString!(".."),
            parsePegSpace,
            parsePegChar,
            parsePegSpace,
            parseString!("]")
        )
    )(src, line);
}

/// PEGのエラー演算子。
ParseResult parsePegError(string src, size_t line) {
    return parseNode!(PEG_ERROR, false, true,
        parseSequence!(
            parseString!("E{"),
            parsePegIdentifier,
            parsePegSpace,
            parseString!("}")
        )
    )(src, line);
}

/// 改行。
ParseResult parsePegNewline(string src, size_t line) {
    return parseSequence!(
        parseChoice!(
            parseString!("\r\n"),
            parseString!("\r"),
            parseString!("\n")
        ),
        parseAddLine
    )(src, line);
}

/// 行端までコメント。
ParseResult parsePegLineComment(string src, size_t line) {
    return parseSequence!(
        parseString!("//"),
        parseZeroOrMore!(
            parseSequence!(
                parseNotTest!(parsePegNewline),
                parseAnyChar
            )
        ),
        parseChoice!(parsePegNewline, parseEof)
    )(src, line);
}

/// ブロックコメント。
ParseResult parsePegBlockComment(string src, size_t line) {
        return parseSequence!(
        parseString!("/*"),
        parseZeroOrMore!(
            parseSequence!(
                parseNotTest!(parseString!("*/")),
                parseAnyChar
            )
        ),
        parseString!("*/")
    )(src, line);
}

/// パーサの原始式。
ParseResult parsePegPrimary(string src, size_t line) {
    return parseSequence!(
        parsePegSpace,
        parseChoice!(
            parsePegError,
            parsePegIdentifier,
            parsePegString,
            parseSequence!(
                parseString!("("),
                parsePegExp,
                parsePegSpace,
                parseString!(")")
            ),
            parsePegCharRange,
            parsePegAnyChar
        )
    )(src, line);
}

/// パーサの繰り返し式。
ParseResult parsePegRepeat(string src, size_t line) {
    return parseNode!(PEG_REPEAT, false, false,
        parseSequence!(
            parsePegPrimary,
            parsePegSpace,
            parseOption!(
                parseChoice!(
                    parseNode!(PEG_OPTION_OP, false, true, parseString!("?")),
                    parseNode!(PEG_ZERO_OR_MORE_OP, false, true, parseString!("*")),
                    parseNode!(PEG_ONE_OR_MORE_OP, false, true, parseString!("+"))
                )
            )
        )
    )(src, line);
}

/// パーサのANDテスト演算子。
ParseResult parsePegAndTestOp(string src, size_t line) {
    return parseNode!(PEG_AND_TEST_OP, false, true, parseString!("&"))(src, line);
}

/// パーサのNOTテスト演算子。
ParseResult parsePegNotTestOp(string src, size_t line) {
    return parseNode!(PEG_NOT_TEST_OP, false, true, parseString!("!"))(src, line);
}

/// パーサのテスト式。
ParseResult parsePegTest(string src, size_t line) {
    return parseNode!(PEG_TEST, false, false,
        parseSequence!(
            parsePegSpace,
            parseOption!(
                parseChoice!(
                    parsePegAndTestOp,
                    parsePegNotTestOp
                )
            ),
            parsePegRepeat
        )
    )(src, line);
}

/// パーサの連接式。
ParseResult parsePegSequence(string src, size_t line) {
    return parseNode!(PEG_SEQUENCE, false, false,
        parseOneOrMore!(parsePegTest)
    )(src, line);
}

/// パーサの選択式。
ParseResult parsePegChoice(string src, size_t line) {
    return parseNode!(PEG_CHOICE, false, false,
        parseSequence!(
            parsePegSequence,
            parseZeroOrMore!(
                parseSequence!(
                    parsePegSpace,
                    parseString!("/"),
                    parsePegSequence
                )
            )
        )
    )(src, line);
}

/// 構文規則の宣言型。
ParseResult parsePegDeclareType(string src, size_t line) {
    return parseChoice!(
        parseNode!(PEG_DECLARE_SKIPPER, false, true, parseString!(PEG_DECLARE_SKIPPER)),
        parseNode!(PEG_DECLARE_NO_SKIP, false, true, parseString!(PEG_DECLARE_NO_SKIP)),
        parseNode!(PEG_DECLARE_MAKE_NODE, false, true, parseString!(PEG_DECLARE_MAKE_NODE)),
        parseNode!(PEG_DECLARE_TEXT_NODE, false, true, parseString!(PEG_DECLARE_TEXT_NODE)),
        parseNode!(PEG_DECLARE_NEW_LINE, false, true, parseString!(PEG_DECLARE_NEW_LINE)),
        parseNode!(PEG_DECLARE_ALWAYS, false, true, parseString!(PEG_DECLARE_ALWAYS))
    )(src, line);
}

/// 構文規則の宣言部分。
ParseResult parsePegDeclare(string src, size_t line) {
    return parseNode!(PEG_DECLARE, false, true,
        parseSequence!(
            parseZeroOrMore!(
                parseSequence!(
                    parsePegSpace,
                    parsePegDeclareType
                )
            ),
            parsePegSpace,
            parsePegIdentifier
        )
    )(src, line);
}

/// 構文規則の定義。
ParseResult parsePegDefine(string src, size_t line) {
    return parseNode!(PEG_DEFINE, false, true,
        parseSequence!(
            parsePegDeclare,
            parsePegSpace,
            parseString!("="),
            parsePegExp,
            parsePegSpace,
            parseString!(";")
        )
    )(src, line);
}

/// PEG全体。
ParseResult parsePeg(string src, size_t line) {
    return parseNode!(PEG_ROOT, false, true,
        parseSequence!(
            parseOption!(parsePegUtf8Bom),
            parseOneOrMore!(parsePegDefine),
            parsePegSpace,
            parseEof
        )
    )(src, line);
}

/// パーサ定義コード生成。
string makeParserDefineCode(
        string id, bool noSkip, bool makeNode, bool text, bool newLine, bool always, Node content, string skipper) {
    string result = "outland.ctfe.peg.parser.ParseResult " ~ id ~ "(string src, size_t line) {\n";
    
    auto inner = makeParserCode(content, skipper);
    if(makeNode) {
        auto textFlag = (text ? "true" : "false");
        auto alwaysFlag = (always ? "true" : "false");
        
        result ~= "auto r = outland.ctfe.peg.parser.parseNode!(\"" ~ id ~ "\"," ~ textFlag ~ "," ~ alwaysFlag ~ ",\n"
                ~ "                   " ~ inner ~ ")(src,line);\n";
    } else {
        result ~= "auto r = " ~ inner ~ "(src,line);\n";
    }
    
    if(newLine) {
        result ~= "if(r.match) {++r.nextLine;}";
    }
    
    result ~= "return r;\n}\n\n";
    
    return result;
}

/// ノードからのコード生成。
string makeParserCode(Node node, string skipper) {
    switch(node.type) {
    case PEG_ANY_CHAR:
        return makeAnyCharCode(skipper);
    case PEG_STRING:
        return makeStringCode(node.value, skipper);
    case PEG_RANGE:
        assert(node.children.length == 2, "Invalid range expression!");
        assert(node.children[0].type == PEG_RANGE_CHAR, "Invalid range expression!");
        assert(node.children[1].type == PEG_RANGE_CHAR, "Invalid range expression!");
        return makeRangeCode(node.children[0].value, node.children[1].value, skipper);
    case PEG_ERROR:
        assert(node.children.length == 1, "Invalid error expression!");
        assert(node.children[0].type == PEG_IDENTIFIER, "Invalid error expression!");
        return makeErrorCode(node.children[0].value, skipper);
    case PEG_IDENTIFIER:
        return makeIdentifierCode(node.value, skipper);
    case PEG_REPEAT:
        assert(node.children.length == 2, "Invalid repeat expression!");
        switch(node.children[1].type) {
        case PEG_OPTION_OP:
            return makeOptionCode(node.children[0], skipper);
        case PEG_ZERO_OR_MORE_OP:
            return makeZeroOrMoreCode(node.children[0], skipper);
        case PEG_ONE_OR_MORE_OP:
            return makeOneOrMoreCode(node.children[0], skipper);
        default:
            assert(false, "Invalid repeat expression! " ~ node.children[0].type);
            break;
        }
    case PEG_TEST:
        assert(node.children.length == 2, "Invalid test expression!");
        switch(node.children[0].type) {
        case PEG_AND_TEST_OP:
            return makeAndTestCode(node.children[1], skipper);
        case PEG_NOT_TEST_OP:
            return makeNotTestCode(node.children[1], skipper);
        default:
            assert(false, "Invalid test expression! " ~ node.children[0].type);
            break;
        }
    case PEG_SEQUENCE:
        assert(node.children.length > 0, "Invalid sequence expression!");
        return makeSequenceCode(node.children, skipper);
    case PEG_CHOICE:
        assert(node.children.length > 0, "Invalid choice expression!");
        return makeChoiceCode(node.children, skipper);
    // 通常は来ない演算子。
    case PEG_OPTION_OP:
    case PEG_ZERO_OR_MORE_OP:
    case PEG_ONE_OR_MORE_OP:
    case PEG_AND_TEST_OP:
    case PEG_NOT_TEST_OP:
    default:
        assert(false, "Invalid Node type! " ~ node.type);
        break;
    }
    return "";
}

/// 任意文字パーサの生成。
string makeAnyCharCode(string skipper) {
    string p = "outland.ctfe.peg.parser.parseAnyChar";
    if(skipper.length > 0) {
        return "outland.ctfe.peg.parser.parseSkip!(" ~  skipper ~ "," ~ p ~ ")";
    } else {
        return p;
    }
}

/// 文字列パーサの生成。
string makeStringCode(string val, string skipper) {
    string p = "outland.ctfe.peg.parser.parseString!(" ~ val ~ ")";
    if(skipper.length > 0) {
        return "outland.ctfe.peg.parser.parseSkip!(" ~  skipper ~ "," ~ p ~ ")";
    } else {
        return p;
    }
}

/// 文字範囲パーサの生成。
string makeRangeCode(string l, string h, string skipper) {
    string p = "outland.ctfe.peg.parser.parseCharRange!(" ~ l ~ "," ~ h ~ ")";
    if(skipper.length > 0) {
        return "outland.ctfe.peg.parser.parseSkip!(" ~  skipper ~ "," ~ p ~ ")";
    } else {
        return p;
    }
}

/// エラーパーサの生成。
string makeErrorCode(string err, string skipper) {
    string p = "outland.ctfe.peg.parser.parseError!(\"" ~ err ~ "\")";
    if(skipper.length > 0) {
        return "outland.ctfe.peg.parser.parseSkip!(" ~  skipper ~ "," ~ p ~ ")";
    } else {
        return p;
    }
}

/// 識別子パーサの生成。
string makeIdentifierCode(string id, string skipper) {
    if(skipper.length > 0) {
        return "outland.ctfe.peg.parser.parseSkip!(" ~  skipper ~ "," ~ id ~ ")";
    } else {
        return id;
    }
}

/// オプションパーサの生成。
string makeOptionCode(Node child, string skipper) {return "parseOption!(" ~ makeParserCode(child, skipper) ~ ")";}

/// 0以上パーサの生成。
string makeZeroOrMoreCode(Node child, string skipper) {return "parseZeroOrMore!(" ~ makeParserCode(child, skipper) ~ ")";}

/// 1以上パーサの生成。
string makeOneOrMoreCode(Node child, string skipper) {return "parseOneOrMore!(" ~ makeParserCode(child, skipper) ~ ")";}

/// ANDテストパーサの生成。
string makeAndTestCode(Node child, string skipper) {return "parseAndTest!(" ~ makeParserCode(child, skipper) ~ ")";}

/// NOTテストパーサの生成。
string makeNotTestCode(Node child, string skipper) {return "parseNotTest!(" ~ makeParserCode(child, skipper) ~ ")";}

/// 連接パーサの生成。
string makeSequenceCode(Node[] children, string skipper)
in {
    assert(children.length > 0);
} body {
    if(children.length == 1) {
        return makeParserCode(children[0], skipper);
    }
    
    string parser = makeParserCode(children[0], skipper);
    foreach(child; children[1 .. $]) {
        parser ~= ",\n";
        parser ~= makeParserCode(child, skipper);
    }
    
    return "parseSequence!(" ~ parser ~ ")";
}

/// 選択パーサの生成。
string makeChoiceCode(Node[] children, string skipper)
in {
    assert(children.length > 0);
} body {
    if(children.length == 1) {
        return makeParserCode(children[0], skipper);
    }
    
    string parser = makeParserCode(children[0], skipper);
    foreach(child; children[1 .. $]) {
        parser ~= ",";
        parser ~= makeParserCode(child, skipper);
    }
    
    return "parseChoice!(" ~ parser ~ ")";
}

/// パーサのテスト。
bool testParser() {

    assert(parseAnyChar("a", 0).match);
    assert(parseAnyChar("a", 0).rest == "", parseAnyChar("a", 0).rest);
    assert(!parseAnyChar("", 0).match);
    
    assert(!parseEof("a", 0).match);
    assert(parseEof("a", 0).rest == "a", parseEof("a", 0).rest);
    assert(parseEof("", 0).match);
    
    assert(parseString!("abc")("abcd", 0).match);
    assert(parseString!("abc")("abcd", 0).rest == "d", parseString!("abc")("abcd", 0).rest);
    assert(!parseString!("abc")("ab", 0).match);
    assert(parseString!("abc")("ab", 0).rest == "ab", parseString!("abc")("ab", 0).rest);
    
    assert(parseCharRange!('a', 'z')("abcd", 0).match);
    assert(parseCharRange!('a', 'z')("abcd", 0).rest == "bcd", parseCharRange!('a', 'z')("abcd", 0).rest);
    assert(!parseCharRange!('a', 'z')("1b", 0).match);
    assert(parseCharRange!('a', 'z')("1b", 0).rest == "1b", parseCharRange!('a', 'z')("1b", 0).rest);
    
    assert(parseError!("error!")("abcd", 0).match);
    assert(parseError!("error!")("abcd", 0).rest == "abcd", parseError!("error!")("abcd", 0).rest);
    assert(parseError!("error!")("abcd", 0).errors.length == 1);
    assert(parseError!("error!")("abcd", 0).errors[0].description == "error!");
    
    assert(parseOption!(parseString!("abc"))("abcd", 0).match);
    assert(parseOption!(parseString!("abc"))("abcd", 0).rest == "d", parseOption!(parseString!("abc"))("abcd", 0).rest);
    
    assert(parseOption!(parseString!("abc"))("def", 0).match);
    assert(parseOption!(parseString!("abc"))("def", 0).rest == "def", parseOption!(parseString!("abc"))("def", 0).rest);
    
    assert(parseZeroOrMore!(parseString!("abc"))("abcd", 0).match);
    assert(parseZeroOrMore!(parseString!("abc"))("abcd", 0).rest == "d", parseOption!(parseString!("abc"))("abcd", 0).rest);
    
    assert(parseZeroOrMore!(parseString!("abc"))("abcabcab", 0).match);
    assert(parseZeroOrMore!(parseString!("abc"))("abcabcab", 0).rest == "ab", parseZeroOrMore!(parseString!("abc"))("abcabcab", 0).rest);
    
    assert(parseZeroOrMore!(parseString!("abc"))("def", 0).match);
    assert(parseZeroOrMore!(parseString!("abc"))("def", 0).rest == "def", parseOption!(parseString!("abc"))("def", 0).rest);
    
    assert(parseOneOrMore!(parseString!("abc"))("abcd", 0).match);
    assert(parseOneOrMore!(parseString!("abc"))("abcd", 0).rest == "d", parseOption!(parseString!("abc"))("abcd", 0).rest);
    
    assert(parseOneOrMore!(parseString!("abc"))("abcabcab", 0).match);
    assert(parseOneOrMore!(parseString!("abc"))("abcabcab", 0).rest == "ab", parseZeroOrMore!(parseString!("abc"))("abcabcab", 0).rest);
    
    assert(!parseOneOrMore!(parseString!("abc"))("def", 0).match);
    assert(parseOneOrMore!(parseString!("abc"))("def", 0).rest == "def", parseOption!(parseString!("abc"))("def", 0).rest);
    
    assert(parseAndTest!(parseString!("abc"))("abcd", 0).match);
    assert(parseAndTest!(parseString!("abc"))("abcd", 0).rest == "abcd", parseAndTest!(parseString!("abc"))("abcd", 0).rest);
    assert(!parseAndTest!(parseString!("abc"))("ab", 0).match);
    assert(parseAndTest!(parseString!("abc"))("ab", 0).rest == "ab", parseAndTest!(parseString!("abc"))("ab", 0).rest);
    
    assert(!parseNotTest!(parseString!("abc"))("abcd", 0).match);
    assert(parseNotTest!(parseString!("abc"))("abcd", 0).rest == "abcd", parseAndTest!(parseString!("abc"))("abcd", 0).rest);
    assert(parseNotTest!(parseString!("abc"))("ab", 0).match);
    assert(parseNotTest!(parseString!("abc"))("ab", 0).rest == "ab", parseAndTest!(parseString!("abc"))("ab", 0).rest);
    
    assert(parseSequence!(parseString!("ab"), parseString!("cd"))("abcde", 0).match);
    assert(parseSequence!(parseString!("ab"), parseString!("cd"))("abcde", 0).rest == "e",
        parseSequence!(parseString!("ab"), parseString!("cd"))("abcde", 0).rest);
        
    assert(!parseSequence!(parseString!("ab"), parseString!("cd"))("def", 0).match);
    assert(parseSequence!(parseString!("ab"), parseString!("cd"))("def", 0).rest == "def",
        parseSequence!(parseString!("ab"), parseString!("cd"))("abc", 0).rest);
        
    assert(!parseSequence!(parseString!("ab"), parseString!("cd"))("abc", 0).match);
    assert(parseSequence!(parseString!("ab"), parseString!("cd"))("abc", 0).rest == "abc",
        parseSequence!(parseString!("ab"), parseString!("cd"))("abc", 0).rest);
        
    assert(parseChoice!(parseString!("ab"), parseString!("cd"))("abcd", 0).match);
    assert(parseChoice!(parseString!("ab"), parseString!("cd"))("abcd", 0).rest == "cd",
        parseChoice!(parseString!("ab"), parseString!("cd"))("abcd", 0).rest);
        
        assert(parseChoice!(parseString!("ab"), parseString!("cd"))("cdab", 0).match);
    assert(parseChoice!(parseString!("ab"), parseString!("cd"))("cdab", 0).rest == "ab",
        parseChoice!(parseString!("ab"), parseString!("cd"))("cdab", 0).rest);
        
    assert(!parseChoice!(parseString!("ab"), parseString!("cd"))("acf", 0).match);
    assert(parseChoice!(parseString!("ab"), parseString!("cd"))("acf", 0).rest == "acf",
        parseChoice!(parseString!("ab"), parseString!("cd"))("acf", 0).rest);
        
    assert(parseNode!("test", false, true, parseString!("abc"))("abcd", 0).match);
    assert(parseNode!("test", false, true, parseString!("abc"))("abcd", 0).rest == "d",
        parseNode!("test", false, true, parseString!("abc"))("abcd", 0).rest);
    assert(parseNode!("test", false, true, parseString!("abc"))("abcd", 0).nodes.length == 1);
    assert(parseNode!("test", false, true, parseString!("abc"))("abcd", 0).nodes[0].type == "test");
    
    assert(parseNode!("test", true, true, parseString!("abc"))("abcd", 0).match);
    assert(parseNode!("test", true, true, parseString!("abc"))("abcd", 0).rest == "d",
        parseNode!("test", true, true, parseString!("abc"))("abcd", 0).rest);
    assert(parseNode!("test", true, true, parseString!("abc"))("abcd", 0).nodes.length == 1);
    assert(parseNode!("test", true, true, parseString!("abc"))("abcd", 0).nodes[0].value == "abc");
    
    assert(parseSequence!(parseString!("test"), parseAddLine)("test", 0).match);
    assert(parseSequence!(parseString!("test"), parseAddLine)("test", 0).nextLine == 1);
    assert(!parseSequence!(parseString!("test"), parseAddLine)("abc", 0).match);
    assert(parseSequence!(parseString!("test"), parseAddLine)("abc", 0).nextLine == 0);
    
    assert(parsePegIdentifier("test;", 0).match);
    assert(parsePegIdentifier("test;", 0).rest == ";", parsePegIdentifier("test;", 0).rest);
    assert(parsePegIdentifier("test;", 0).nodes.length == 1);
    assert(parsePegIdentifier("test;", 0).nodes[0].type == PEG_IDENTIFIER);
    assert(parsePegIdentifier("test;", 0).nodes[0].value == "test");
    
    assert(!parsePegIdentifier("0est;", 0).match);
    assert(parsePegIdentifier("0est;", 0).rest == "0est;", parsePegIdentifier("0est;", 0).rest);
    
    assert(parsePegString(`"test";`, 0).match);
    assert(parsePegString(`"test";`, 0).rest == ";", parsePegString(`"test";`, 0).rest);
    assert(parsePegString(`"test";`, 0).nodes.length == 1);
    assert(parsePegString(`"test";`, 0).nodes[0].type == PEG_STRING);
    assert(parsePegString(`"test";`, 0).nodes[0].value == `"test"`);
    
    assert(parsePegString(`"te\"st";`, 0).match);
    assert(parsePegString(`"te\"st";`, 0).rest == ";", parsePegString(`"te\"st";`, 0).rest);
    assert(parsePegString(`"te\"st";`, 0).nodes.length == 1);
    assert(parsePegString(`"te\"st";`, 0).nodes[0].type == PEG_STRING);
    assert(parsePegString(`"te\"st";`, 0).nodes[0].value == `"te\"st"`);
    
    assert(!parsePegString("\"test;", 0).match);
    assert(parsePegString("\"test;", 0).rest == "\"test;", parsePegString("\"test;", 0).rest);
    
    assert(parsePegNewline("\r", 0).match);
    assert(parsePegNewline("\r", 0).nextLine == 1);
    assert(parsePegNewline("\n", 0).match);
    assert(parsePegNewline("\n", 0).nextLine == 1);
    assert(parsePegNewline("\r\n", 0).match);
    assert(parsePegNewline("\r\n", 0).nextLine == 1);
    assert(parsePegNewline("\r\n", 0).rest == "", parsePegNewline("\r\n", 0).rest);
    
    assert(parsePegLineComment("//test \r\nnext", 0).match);
    assert(parsePegLineComment("//test \r\nnext", 0).rest == "next", parsePegLineComment("//test \r\nnext", 0).rest);
    assert(parsePegLineComment("//test \r\nnext", 0).nextLine == 1);
    assert(parsePegLineComment("//test", 0).match);
    assert(parsePegLineComment("//test", 0).rest == "", parsePegLineComment("//test", 0).rest);
    assert(parsePegLineComment("//test", 0).nextLine == 0);
    
    assert(parsePegBlockComment("/* \r\n*/next", 0).match);
    assert(parsePegBlockComment("/* \r\n*/next", 0).rest == "next", parsePegBlockComment("/* \r\n*/next", 0).rest);
    assert(parsePegBlockComment("/**/", 0).match);
    assert(parsePegBlockComment("/**/", 0).rest == "", parsePegBlockComment("/**/", 0).rest);
    
    assert(!parsePegBlockComment("/*test", 0).match);
    assert(parsePegBlockComment("/*test", 0).rest == "/*test", parsePegBlockComment("/*test", 0).rest);
    
    assert(parsePegChar("'a'", 0).match);
    assert(parsePegChar("'a'", 0).nodes.length == 1);
    assert(parsePegChar("'a'", 0).nodes[0].value == "'a'", parsePegChar("'a'", 0).nodes[0].value);
    assert(parsePegCharRange(`[  'a'  ..  'z' ] `, 0).match);
    assert(parsePegCharRange(`['a'..'z']`, 0).nodes.length == 1);
    assert(parsePegCharRange(`['a'..'z']`, 0).nodes[0].type == PEG_RANGE);
    assert(parsePegCharRange(`['a'..'z']`, 0).nodes[0].children.length == 2);
    assert(parsePegCharRange(`['a'..'z']`, 0).nodes[0].children[0].type == PEG_RANGE_CHAR);
    assert(parsePegCharRange(`['a'..'z']`, 0).nodes[0].children[0].value == "'a'");
    assert(parsePegCharRange(`['a'..'z']`, 0).nodes[0].children[1].type == PEG_RANGE_CHAR);
    assert(parsePegCharRange(`['a'..'z']`, 0).nodes[0].children[1].value == "'z'");
    
    //assert(parsePeg(import("outland/ctfe/peg/test.peg"), 0).match);
    
    return true;
}

unittest {
    static assert(testParser());
    //pragma(msg, dumpNode(parsePeg(import("outland/ctfe/peg/peg.peg"), 0).nodes[0]));
    //pragma(msg, compilePegParser(import("outland/ctfe/peg/peg.peg")));
    //static assert(defineParser(`NO_SKIP utf8Bom = "\xEF\xBB\xBF";`, 0).match);
}
