﻿/**
 *	メニュー。
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module outland.dmajor.menu;

import win32.windows;

import outland.dmajor.exception;
import outland.dmajor.handle;
import outland.dmajor.shape;
import outland.dmajor.tstring;
import outland.dmajor.window;

/// メニュークラス。
class Menu : HandleWrapper!(HMENU) {
	
	/// 生成する。
	static Menu create() {
		auto menu = CreateMenu();
		checkApi(menu);
		return new Menu(menu);
	}
	
	/// ditto
	static Menu createPopup() {
		auto menu = CreatePopupMenu();
		checkApi(menu);
		return new Menu(menu);
	}
	
	/// ditto
	static Menu fromHandle(HMENU menu) {
		if(auto m = cast(Menu) HandleDatabase.instance[menu]) {
			return m;
		}
		return new Menu(menu, null);
	}
	
	/// ditto
	static void duringObject(HMENU menu, void delegate(Menu) dg) {
		if(auto m = cast(Menu) HandleDatabase.instance[menu]) {
			dg(m);
		} else {
			scope m = new Menu(menu, null);
			dg(m);
		}
	}
	
	/// リソースからの取得。
	static Menu load(HINSTANCE inst, LPCTSTR name) {
		auto m = LoadMenu(inst, name);
		checkApi(m);
		return new Menu(m);
	}
	
	/// 有効なメニューかどうか返す。
	bool isValid() {return IsMenu(handle) != FALSE;}
	
	/// 項目を破棄する。
	void deleteById(UINT id) {checkApi(DeleteMenu(handle, id, MF_BYCOMMAND));}
	
	/// ditto
	void deleteByPosition(size_t i) {checkApi(DeleteMenu(handle, i, MF_BYPOSITION));}
	
	/// 有効・無効。
	void enableById(UINT id, UINT state) {
		checkApi(EnableMenuItem(handle, id, state | MF_BYCOMMAND));
	}
	
	/// ditto
	void enableByPosition(size_t pos, UINT state) {
		checkApi(EnableMenuItem(handle, pos, state | MF_BYPOSITION));
	}
	
	/// デフォルト項目。
	void setDefaultItemById(UINT id) {checkApi(SetMenuDefaultItem(handle, id, FALSE));}
	
	/// ditto
	void setDefaultItemByPosition(size_t pos) {checkApi(SetMenuDefaultItem(handle, pos, TRUE));}
	
	/// デフォルト項目。
	UINT getDefaultItemById(UINT f = 0) {
		auto i = GetMenuDefaultItem(handle, FALSE, f);
		if(i < 0) {
			throwLastError();
		}
		return i;
	}
	
	/// ditto
	size_t getDefaultItemByPosition(UINT f = 0) {
		auto i = GetMenuDefaultItem(handle, TRUE, f);
		if(i < 0) {
			throwLastError();
		}
		return i;
	}
	
	/// 項目数。
	size_t length() {return GetMenuItemCount(handle);}
	
	/// 項目ID。
	UINT id(size_t i) {return GetMenuItemID(handle, i);}
	
	/// メニュー情報。
	void getItemInfoById(UINT id, inout MENUITEMINFO info) {
		info.cbSize = MENUITEMINFO.sizeof;
		checkApi(GetMenuItemInfo(handle, id, FALSE, &info));
	}
	
	/// ditto
	void getItemInfoByPosition(size_t i, inout MENUITEMINFO info) {
		info.cbSize = MENUITEMINFO.sizeof;
		checkApi(GetMenuItemInfo(handle, i, TRUE, &info));
	}
	
	/// ditto
	void setItemInfoById(UINT id, inout MENUITEMINFO info) {
		info.cbSize = MENUITEMINFO.sizeof;
		checkApi(SetMenuItemInfo(handle, id, FALSE, &info));
	}
	
	/// ditto
	void setItemInfoByPosition(size_t i, inout MENUITEMINFO info) {
		info.cbSize = MENUITEMINFO.sizeof;
		checkApi(SetMenuItemInfo(handle, i, TRUE, &info));
	}
	
	/// 項目の矩形を得る。
	Rect getItemRect(size_t i) {
		Rect r;
		checkApi(GetMenuItemRect(cast(HWND) null, handle, i, r.ptr));
		return r;
	}
	
	/// 項目の状態を得る。
	UINT getStateById(UINT id) {return GetMenuState(handle, id, MF_BYCOMMAND);}
	
	/// ditto
	UINT getStateByPosition(size_t i) {return GetMenuState(handle, i, MF_BYPOSITION);}
	
	/// 文字列を得る。
	int stringById(UINT id, TCHAR[] buf) {
		return GetMenuString(handle, id, buf.ptr, buf.length, MF_BYCOMMAND);
	}
	
	/// ditto
	int stringByPosition(UINT id, TCHAR[] buf) {
		return GetMenuString(handle, id, buf.ptr, buf.length, MF_BYPOSITION);
	}
	
	/// 文字列の長さを得る。
	int stringLengthById(UINT id) {
		return GetMenuString(handle, id, null, 0, MF_BYCOMMAND);
	}
	
	/// ditto
	int stringLengthByPosition(UINT id, TCHAR[] buf) {
		return GetMenuString(handle, id, null, 0, MF_BYPOSITION);
	}
	
	/// サブメニュー。
	void duringSubMenu(size_t i, void delegate(Menu) dg) {
		auto m = GetSubMenu(handle, i);
		checkApi(m);
		scope Menu menu = new Menu(m, null);
		dg(menu);
	}
	
	/// メニュー項目の追加。
	void insertItemById(UINT id, inout MENUITEMINFO info) {
		checkApi(InsertMenuItem(handle, id, FALSE, &info));
	}
	
	/// メニュー項目の追加。
	void insertItemByPosition(size_t i, inout MENUITEMINFO info) {
		checkApi(InsertMenuItem(handle, i, TRUE, &info));
	}
	
	/// 座標から項目を得る。
	size_t itemFromPoint(Point pos) {
		return MenuItemFromPoint(cast(HWND) null, handle, *pos.ptr);
	}
	
	/// 子項目を全て破棄する。
	void removeItemsById(UINT id) {
		checkApi(RemoveMenu(handle, id, MF_BYCOMMAND));
	}
	
	/// ditto
	void removeItemsByPosition(size_t i) {
		checkApi(RemoveMenu(handle, i, MF_BYPOSITION));
	}
	
	/// ビットマップを設定する。
	void setBitmapById(UINT id, HBITMAP unchecked, HBITMAP checked) {
		checkApi(SetMenuItemBitmaps(handle, id, MF_BYCOMMAND, unchecked, checked));
	}
	
	/// ditto
	void setBitmapByPosition(size_t i, HBITMAP unchecked, HBITMAP checked) {
		checkApi(SetMenuItemBitmaps(handle, i, MF_BYPOSITION, unchecked, checked));
	}
	
	/// ポップアップメニューの表示。
	int trackPoupMenu(Point pos, UINT flag, Window w, Rect r)
	in {
		assert(w && w.isValid);
	} body {
		TPMPARAMS param;
		with(param) {
			cbSize = param.sizeof;
			rcExclude = *r.ptr;
		}
		return TrackPopupMenuEx(handle, flag, pos.x, pos.y, w.handle, &param);
	}
	
	/// 項目の追加。
	Menu add(UINT id, char[] name, uint state = MFS_ENABLED) {
		MENUITEMINFO info;
		with(info) {
			cbSize = info.sizeof;
			fMask = MIIM_TYPE | MIIM_STATE | MIIM_ID;
			fType = MFT_STRING;
			fState = state;
			wID = id;
			dwTypeData = toTString(name);
			cch = lstrlen(dwTypeData);
		}
		
		insertItemByPosition(length, info);
		return this;
	}
	
	/// ditto。
	Menu add(MENUITEMINFO info) {
		info.cbSize = info.sizeof;		
		insertItemByPosition(length, info);
		return this;
	}
	
	/// 区切りの追加。
	Menu addSeparator() {
		// メニュー情報。
		MENUITEMINFO info;
		with(info) {
			cbSize = info.sizeof;
			fMask = MIIM_TYPE;
			fType = MFT_SEPARATOR;
		}
		
		insertItemByPosition(length, info);
		return this;
	}
	
	/// サブメニューの追加。
	Menu addSubMenu(char[] name, void delegate(Menu) dg) {
		// サブメニューの作成。
		scope m = Menu.createPopup();
		dg(m);
		
		// メニュー情報。
		MENUITEMINFO info;
		with(info) {
			cbSize = info.sizeof;
			fMask = MIIM_TYPE | MIIM_SUBMENU;
			fType = MFT_STRING;
			dwTypeData = toTString(name);
			cch = lstrlen(dwTypeData);
			hSubMenu = m.detach;
		}
		
		// 失敗時にはサブメニュー破棄。
		scope(failure) DestroyMenu(info.hSubMenu);
		
		insertItemByPosition(length, info);
		return this;
	}
	
protected:
	
	/// コンストラクタ。
	this(HMENU menu, Destructor dtor) {super(menu, dtor);}
	
	/// ditto
	this(HMENU menu) {super(menu, &destroy);}
	
private:
	
	/// メニューを破棄する。
	void destroy(HMENU menu) {DestroyMenu(menu);}
}
