﻿/**
 *	ウィンドウメッセージ関連モジュール。
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module outland.dmajor.message;

import win32.windows;

import outland.dmajor.dc;
import outland.dmajor.handle;
import outland.dmajor.scrollbar;
import outland.dmajor.shape;
import outland.dmajor.window;

/// メッセージデリゲート。
alias LRESULT delegate(Window w, UINT msg, WPARAM wp, LPARAM lp) MessageDelegate;

/** メッセージハンドラを生成する。
 *
 *	Params:
 *		D = 最終的に呼び出されるハンドラデリゲートの型。
 *		T = メッセージパラメータ変換関数。static関数でなければならない。
 *		d = 最終的に呼び出されるハンドラデリゲート。
 */
MessageDelegate makeMessageDelegate(D, alias T)(D d) {
	// メッセージハンドラを保持するためのクラス。
	static class MessageHandler {
		/// ハンドラと変換関数を指定して生成する。
		this(D h) {handler_ = h;}
		
		/// メッセージを処理する。
		LRESULT handle(Window w, UINT msg, WPARAM wp, LPARAM lp) {return T(handler_, w, wp, lp);}
		
		/// ハンドラ。
		final D handler_;
	}
	
	return &(new MessageHandler(d)).handle;
}

/// ウィンドウ生成時の処理。
alias bool delegate(Window, LPCREATESTRUCT) OnCreateDelegate;

/// ditto
LRESULT translateOnCreate(OnCreateDelegate d, Window w, WPARAM wp, LPARAM lp) {
	return d(w, cast(LPCREATESTRUCT) lp) ? 0 : -1;
}

/// ditto
LRESULT translateOnNcCreate(OnCreateDelegate d, Window w, WPARAM wp, LPARAM lp) {
	return d(w, cast(LPCREATESTRUCT) lp) ? TRUE : FALSE;
}

/// ウィンドウ破棄時の処理。
alias void delegate(Window) OnDestroyDelegate;

/// ditto
LRESULT translateOnDestroy(OnDestroyDelegate d, Window w, WPARAM wp, LPARAM lp) {
	d(w);
	return 0;
}

/// ウィンドウ破棄時の処理。
LRESULT translateOnNcDestroy(OnDestroyDelegate d, Window w, WPARAM wp, LPARAM lp) {
	d(w);
	return 0;
}

/// マウス関連メッセージのハンドラ。
alias void delegate(Window, uint, Point) OnMouseDelegate;

/// マウス関連メッセージの変換。
LRESULT translateOnMouseMessage(OnMouseDelegate d, Window w, WPARAM wp, LPARAM lp) {
	d(w, wp, Point(LOWORD(lp), HIWORD(lp)));
	return 0;
}

/// マウスホイール回転時のハンドラ。
alias void delegate(Window, uint, uint, Point) OnMouseWheel;

/// マウスホイール回転メッセージの変換。
LRESULT translateOnMouseWheel(OnMouseWheel d, Window w, WPARAM wp, LPARAM lp) {
	d(w, LOWORD(wp), HIWORD(wp), Point(LOWORD(lp), HIWORD(lp)));
	return 0;
}

/// マウス関連メッセージの変換。TRUEを返すバージョン。
LRESULT translateOnMouseXMessage(OnMouseDelegate d, Window w, WPARAM wp, LPARAM lp) {
	d(w, wp, Point(LOWORD(lp), HIWORD(lp)));
	return TRUE;
}

// WM_APP=32768;
// WM_ACTIVATE=6;
// WM_ACTIVATEAPP=28;
// WM_AFXFIRST=864;
// WM_AFXLAST=895;
// WM_ASKCBFORMATNAME=780;
// WM_CANCELJOURNAL=75;
// WM_CANCELMODE=31;
// WM_CAPTURECHANGED=533;
// WM_CHANGECBCHAIN=781;
// WM_CHAR=258;
// WM_CHARTOITEM=47;
// WM_CHILDACTIVATE=34;
// WM_CLEAR=771;
// WM_CLOSE=16;

/// コマンドメッセージのハンドラ。
alias void delegate(Window, ushort, ushort, Window) OnCommandDelegate;

/// コマンドメッセージの変換。
LRESULT translateOnCommand(OnCommandDelegate d, Window w, WPARAM wp, LPARAM lp) {
	d(w, HIWORD(wp), LOWORD(wp), cast(Window) HandleDatabase.instance[cast(HWND) lp]);
	return 0;
}

// WM_COMMNOTIFY=68; // obsolete
// WM_COMPACTING=65;

/// オーナー項目比較処理。
alias void delegate(Window, UINT, LPCOMPAREITEMSTRUCT) OnCompareItemDelegate;

/// 比較メッセージの変換。
LRESULT translateOnCompareItem(OnCompareItemDelegate d, Window w, WPARAM wp, LPARAM lp) {
	d(w, wp, cast(LPCOMPAREITEMSTRUCT) lp);
	return TRUE;
}

// WM_CONTEXTMENU=123;
// WM_COPY=769;
// WM_COPYDATA=74;

/// コントロールカラーデリゲート。
alias HBRUSH delegate(Window, DeviceContext, Window) OnControlColorDelegate;

/// コントロールカラーメッセージ変換。
LRESULT translateOnControlColor(OnControlColorDelegate d, Window w, WPARAM wp, LPARAM lp) {
	HBRUSH brush = cast(HBRUSH) null;
	DeviceContext.duringObject(cast(HDC) wp, (DeviceContext dc) {
		auto ctrl = Window.fromHandle(cast(HWND) lp);
		brush = d(w, dc, ctrl);
	});
	return cast(LRESULT) brush;
}

// WM_CUT=768;
// WM_DEADCHAR=259;

/// 項目削除メッセージのハンドラ。
alias void delegate(Window, UINT, LPDELETEITEMSTRUCT) OnDeleteItemDelegate;

/// 項目削除メッセージの変換。
LRESULT translateOnDeleteItem(OnDeleteItemDelegate d, Window w, WPARAM wp, LPARAM lp) {
	d(w, wp, cast(LPDELETEITEMSTRUCT) lp);
	return TRUE;
}

// WM_DESTROYCLIPBOARD=775;
// WM_DEVICECHANGE=537;
// WM_DEVMODECHANGE=27;
// WM_DISPLAYCHANGE=126;
// WM_DRAWCLIPBOARD=776;

/// オーナードローハンドラ。
alias void delegate(Window, UINT, LPDRAWITEMSTRUCT ds) OnDrawItemDelegate;

/// オーナードロー処理。
LRESULT translateOnDrawItem(OnDrawItemDelegate d, Window w, WPARAM wp, LPARAM lp) {
	d(w, wp, cast(LPDRAWITEMSTRUCT) lp);
	return TRUE;
}

// WM_DROPFILES=563;
// WM_ENABLE=10;
// WM_ENDSESSION=22;
// WM_ENTERIDLE=289;
// WM_ENTERMENULOOP=529;
// WM_ENTERSIZEMOVE=561;
// WM_ERASEBKGND=20;
// WM_EXITMENULOOP=530;
// WM_EXITSIZEMOVE=562;
// WM_FONTCHANGE=29;
// WM_GETDLGCODE=135;
// WM_GETFONT=49;
// WM_GETHOTKEY=51;
// WM_GETICON=127;
// WM_GETMINMAXINFO=36;
// WM_GETTEXT=13;
// WM_GETTEXTLENGTH=14;
// WM_HANDHELDFIRST=856;
// WM_HANDHELDLAST=863;
// WM_HELP=83;
// WM_HOTKEY=786;

/// スクロールメッセージのハンドラ。
alias void delegate(Window, int, short, ScrollBar) OnScrollDelegate;

/// 水平スクロールメッセージの変換。
LRESULT translateOnHScroll(OnScrollDelegate d, Window w, WPARAM wp, LPARAM lp) {
	d(w, LOWORD(wp), HIWORD(lp), cast(ScrollBar) Window.fromHandle(cast(HWND) lp));
	return 0;
}

// WM_HSCROLLCLIPBOARD=782;
// WM_ICONERASEBKGND=39;
// WM_INITDIALOG=272;
// WM_INITMENU=278;
// WM_INITMENUPOPUP=279;
// WM_INPUTLANGCHANGE=81;
// WM_INPUTLANGCHANGEREQUEST=80;
// WM_KEYDOWN=256;
// WM_KEYUP=257;
// WM_KILLFOCUS=8;
// WM_MDIACTIVATE=546;
// WM_MDICASCADE=551;
// WM_MDICREATE=544;
// WM_MDIDESTROY=545;
// WM_MDIGETACTIVE=553;
// WM_MDIICONARRANGE=552;
// WM_MDIMAXIMIZE=549;
// WM_MDINEXT=548;
// WM_MDIREFRESHMENU=564;
// WM_MDIRESTORE=547;
// WM_MDISETMENU=560;
// WM_MDITILE=550;

/// サイズ計算メッセージのハンドラ。
alias void delegate(Window, UINT, LPMEASUREITEMSTRUCT) OnMeasureItemDelegate;

/// サイズ計算メッセージの変換。
LRESULT translateOnMeasureItem(OnMeasureItemDelegate d, Window w, WPARAM wp, LPARAM lp) {
	d(w, wp, cast(LPMEASUREITEMSTRUCT) lp);
	return TRUE;
}

// static if (WINVER >= 0x500) {
// 	WM_UNINITMENUPOPUP=0x0125;
// 	WM_MENURBUTTONUP=290;
// 	WM_MENUCOMMAND=0x0126;
// 	WM_MENUGETOBJECT=0x0124;
// 	WM_MENUDRAG=0x0123;
// }
// WM_MENUCHAR=288;
// WM_MENUSELECT=287;
// WM_MOVE=3;
// WM_MOVING=534;
// WM_NCACTIVATE=134;
// WM_NCCALCSIZE=131;
// WM_NCHITTEST=132;
// WM_NCLBUTTONDBLCLK=163;
// WM_NCLBUTTONDOWN=161;
// WM_NCLBUTTONUP=162;
// WM_NCMBUTTONDBLCLK=169;
// WM_NCMBUTTONDOWN=167;
// WM_NCMBUTTONUP=168;
// static if (_WIN32_WINNT >= 0x500) {
// 	WM_NCXBUTTONDOWN=171;
// 	WM_NCXBUTTONUP=172;
// 	WM_NCXBUTTONDBLCLK=173;
// 	WM_NCMOUSEHOVER=0x02A0;
// 	WM_NCMOUSELEAVE=0x02A2;
// }
// WM_NCMOUSEMOVE=160;
// WM_NCPAINT=133;
// WM_NCRBUTTONDBLCLK=166;
// WM_NCRBUTTONDOWN=164;
// WM_NCRBUTTONUP=165;
// WM_NEXTDLGCTL=40;
// WM_NEXTMENU=531;
// WM_NOTIFY=78;
// WM_NOTIFYFORMAT=85;
// WM_NULL=0;
// WM_PAINT=15;
// WM_PAINTCLIPBOARD=777;
// WM_PAINTICON=38;
// WM_PALETTECHANGED=785;
// WM_PALETTEISCHANGING=784;
// WM_PARENTNOTIFY=528;
// WM_PASTE=770;
// WM_PENWINFIRST=896;
// WM_PENWINLAST=911;
// WM_POWER=72;
// WM_POWERBROADCAST=536;
// WM_PRINT=791;
// WM_PRINTCLIENT=792;
// WM_QUERYDRAGICON=55;
// WM_QUERYENDSESSION=17;
// WM_QUERYNEWPALETTE=783;
// WM_QUERYOPEN=19;
// WM_QUEUESYNC=35;
// WM_QUIT=18;
// WM_RENDERALLFORMATS=774;
// WM_RENDERFORMAT=773;
// WM_SETCURSOR=32;
// WM_SETFOCUS=7;
// WM_SETFONT=48;
// WM_SETHOTKEY=50;
// WM_SETICON=128;
// WM_SETREDRAW=11;
// WM_SETTEXT=12;
// WM_SETTINGCHANGE=26;
// WM_SHOWWINDOW=24;

/// サイズ変更メッセージのハンドラ。
alias void delegate(Window, uint, Size) OnSizeDelegate;

/// サイズ変更メッセージの変換。
LRESULT translateOnSize(OnSizeDelegate d, Window w, WPARAM wp, LPARAM lp) {
	d(w, wp, Size(LOWORD(lp), HIWORD(lp)));
	return 0;
}

// WM_SIZECLIPBOARD=779;
// WM_SIZING=532;
// WM_SPOOLERSTATUS=42;
// WM_STYLECHANGED=125;
// WM_STYLECHANGING=124;
// WM_SYSCHAR=262;
// WM_SYSCOLORCHANGE=21;

/// システムコマンドメッセージのハンドラ。
alias void delegate(Window, UINT, LPARAM) OnSystemCommandDelegate;

/// システムコマンドメッセージの変換。
LRESULT translateOnSystemCommand(OnSystemCommandDelegate d, Window w, WPARAM wp, LPARAM lp) {
	d(w, wp, lp);
	return 0;
}

// WM_SYSDEADCHAR=263;
// WM_SYSKEYDOWN=260;
// WM_SYSKEYUP=261;
// WM_TCARD=82;
// WM_THEMECHANGED=794;
// WM_TIMECHANGE=30;

/// タイマメッセージのハンドラ。
alias void delegate(Window, uint, TIMERPROC) OnTimerDelegate;

/// タイマメッセージの変換。
LRESULT translateOnTimer(OnTimerDelegate d, Window w, WPARAM wp, LPARAM lp) {
	d(w, wp, cast(TIMERPROC) lp);
	return 0;
}

// WM_UNDO=772;
// WM_USER=1024;
// WM_USERCHANGED=84;
// WM_VKEYTOITEM=46;

/// 垂直スクロールメッセージの変換。
LRESULT translateOnVScroll(OnScrollDelegate d, Window w, WPARAM wp, LPARAM lp) {
	d(w, LOWORD(wp), HIWORD(lp), cast(ScrollBar) Window.fromHandle(cast(HWND) lp));
	return 0;
}

// WM_VSCROLLCLIPBOARD=778;
// WM_WINDOWPOSCHANGED=71;
// WM_WINDOWPOSCHANGING=70;
// WM_WININICHANGE=26;
// WM_SYNCPAINT=136;
// WM_MOUSEACTIVATE=33;
// WM_MOUSEHOVER=0x2A1;
// WM_MOUSELEAVE=0x2A3;
