﻿/**
 *	ウィンドウクラス。
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module outland.dmajor.window;

import std.string;

import win32.windows;

import outland.dmajor.application;
import outland.dmajor.dc;
import outland.dmajor.exception;
import outland.dmajor.handle;
import outland.dmajor.layout;
import outland.dmajor.menu;
import outland.dmajor.message;
import outland.dmajor.scrollbar;
import outland.dmajor.shape;
import outland.dmajor.threadlocal;
import outland.dmajor.tstring;
import outland.dmajor.trace;

/** ウィンドウクラス。
 *
 *	全てのウィンドウの基本クラスとなる。
 */
class Window : LayoutItem {
	
	/// 子ウィンドウIDの定義。
	enum : UINT {
		ID_SELF = 0,	/// 自分自身を示すためのID。
		ID_LAST = 16,	/// 派生クラスのID始点。
	}
	
	/// コマンドデリゲート。
	alias void delegate() CommandDelegate;

	/// ウィンドウを生成する。
	static Window create(LPCTSTR cls, char[] title, DWORD style, DWORD styleEx, Rect r, Window parent = null, Menu menu = null) {
		auto w = new Window();
		auto m = (menu) ? menu.handle() : null;
		auto inst = (parent) ? parent.getInstanceHandle : Application.instance.handle;
		w.createWindow(cls, title, style, styleEx, r, parent ? parent.handle : cast(HWND) null, m, inst, null);
		return w;
	}
	
	/// ditto
	static Window create(ATOM cls, char[] title, DWORD style, DWORD styleEx, Rect r, Window parent = null, Menu menu = null) {
		return create(MAKEINTRESOURCE(cls), title, style, styleEx, r, parent, menu);
	}
	
	/// 子ウィンドウを生成する。
	static Window createChild(LPCTSTR cls, char[] title, DWORD style, DWORD styleEx, Rect r, Window parent, UINT id) {
		auto w = new Window();
		auto inst = (parent) ? parent.getInstanceHandle : Application.instance.handle;
		w.createWindow(cls, title, style | WS_CHILD, styleEx, r, parent ? parent.handle : cast(HWND) null, cast(HMENU) id, inst, null);
		return w;
	}
	
	/// ditto
	static Window createChild(ATOM cls, char[] title, DWORD style, DWORD styleEx, Rect r, Window parent, UINT id) {
		return createChild(MAKEINTRESOURCE(cls), title, style, styleEx, r, parent, id);
	}
	
	/// 既存のウィンドウにオブジェクトを結びつける。
	static Window subclassWindow(HWND wnd) {
		auto w = new Window();
		w.subclass(wnd);
		return w;
	}
	
	/** ウィンドウをオブジェクトに変換する。
	 *
	 *	Params:
	 *		wnd = オブジェクトを検索・生成するウィンドウのハンドル。
	 *
	 *	Returns:
	 *		ハンドルを保持したオブジェクト。
	 *		オブジェクトが割り当てられていなかったら割り当てる。
	 *		wndがNULLの場合はnullが返る。
	 */
	static Window fromHandle(HWND wnd) {
		if(!wnd) {
			return null;
		}
		
		auto w = cast(Window) HandleDatabase.instance[wnd];
		if(!w) {
			w = subclassWindow(wnd);
		}
		return w;
	}
	
	/// ウィンドウオブジェクトを検索する。
	static Window find(HWND wnd) {return cast(Window) HandleDatabase.instance[wnd];}
	
	/// デストラクタ。
	~this() {destroy();}
	
	/// サブクラス化解除。
	void unsubclass()
	in {
		assert(defaultProcedure_ !is null);
		assert(handle_ !is null);
	} body {
		// エラー無視。
		SetWindowLongPtr(handle_, GWL_WNDPROC, cast(LONG_PTR) defaultProcedure_);
		
		// 登録解除。
		HandleDatabase.instance.unregister(handle_);
		handle_ = null;
		defaultProcedure_ = null;
	}
	
	/// ハンドル。
	HWND handle() {return handle_;}
	
	/// ウィンドウ破棄。
	void destroy() {
		if(handle_) {
			DestroyWindow(handle_);
		}
	}
	
	/// ウィンドウを閉じる。
	void close() {checkApi(CloseWindow(handle));}
	
	/// クライアント領域。
	Rect clientRect() {
		Rect rect;
		checkApi(GetClientRect(handle, rect.ptr));
		return rect;
	}
	
	/// ウィンドウ領域。
	Rect windowRect() {
		Rect rect;
		checkApi(GetWindowRect(handle, rect.ptr));
		return rect;
	}
	
	/// 通常の意味での領域。(親ウィンドウのクライアント領域内の相対座標)
	Rect rect() {
		Rect r = windowRect();
		if(HWND p = GetParent(handle)) {
			Point tl = r.topLeft;
			Point br = r.bottomRight;
			ScreenToClient(p, tl.ptr);
			ScreenToClient(p, br.ptr);
			r = Rect(tl, br);
		}
		return r;
	}
	
	/// ditto
	void rect(Rect r) {checkApi(MoveWindow(handle, r.x, r.y, r.cx, r.cy, TRUE));}
	
	/// 座標。
	Point position() {return rect.position;}
	
	/// 座標。
	void position(Point pos) {
		checkApi(SetWindowPos(handle, cast(HANDLE) null, pos.x, pos.y, 0, 0, SWP_NOSIZE | SWP_NOACTIVATE | SWP_NOZORDER));
	}
	
	/// サイズ。
	Size size() {return rect.size;}
	
	/// ditto
	void size(Size s) {
		checkApi(SetWindowPos(handle, cast(HANDLE) null, 0, 0, s.cx, s.cy, SWP_NOMOVE | SWP_NOACTIVATE | SWP_NOZORDER));
	}
	
	/// 最小サイズ。
	Size minSize() {return minSize_;}
	
	/// ditto
	void minSize(Size s) {minSize_ = s;}
	
	/// レイアウトを行う。
	void doLayout(DeferWindowPosition def, Rect rect) {
		def.move(this, rect);
	}
	
	/// 移動する。
	void move(int dx, int dy) {
		Rect r = rect;
		r.move(dx, dy);
		rect = r;
	}
	
	/// クライアント座標に変換。
	void toClient(inout Point screen) {checkApi(ScreenToClient(handle, screen.ptr));}
	
	/// スクリーン座標に変換。
	void toScreen(inout Point client) {checkApi(ClientToScreen(handle, client.ptr));}
	
	/// ウィンドウが有効かどうか返す。
	bool isValid() {return IsWindow(handle_) != FALSE;}
	
	/// ウィンドウ描画。
	void duringWindowDc(void delegate(DeviceContext) dg) {
		// デバイスコンテキスト取得。
		HDC hdc = GetWindowDC(handle);
		checkApi(hdc);
		
		scope dc = new DeviceContext(hdc, (HDC dc) {ReleaseDC(this.handle, dc);});
		
		// 描画の実行。
		dg(dc);
	}
	
	/// クライアント描画。
	void duringClientDc(void delegate(DeviceContext) dg) {
		// デバイスコンテキスト取得。
		auto hdc = GetDC(handle);
		checkApi(hdc);
		
		scope dc = new DeviceContext(hdc, (HDC dc) {ReleaseDC(this.handle, dc);});
		
		// 描画の実行。
		dg(dc);
	}
	
	/// インスタンスハンドルを得る。
	HINSTANCE getInstanceHandle() {return cast(HINSTANCE) GetWindowLongPtr(handle, GWL_HINSTANCE);}
	
	/// コントロールID。
	int controlId() {return GetDlgCtrlID(handle);}
	
	/// 子ウィンドウ。
	Window getChild(int id) {
		HWND child = GetDlgItem(handle, id);
		checkApi(child);
		return Window.fromHandle(child);
	}
	
	/// UTF8に変換されない生の文字列を得る。bufの長さ分だけ取得される。コピー先にはNULL文字も含まれる。
	size_t getText(TCHAR[] buf) {return GetWindowText(handle, buf.ptr, buf.length);}
	
	/// Win文字列に変換しないで設定する。strがNULL終端文字列でなければならない。
	void setText(LPCTSTR str) {checkApi(SetWindowText(handle, str));}
	
	/// テキスト。
	void text(char[] text) {checkApi(SetWindowText(handle, toTString(text)));}
	
	/// ditto
	char[] text() {
		size_t len = textLength();
		if(!len) {
			return null;
		}
		
		// 文字数 + null文字分のバッファ。
		auto buf = new TCHAR[len + 1];
		getText(buf);
		return fromTString(buf.ptr);
	}
	
	/// テキストの長さ。
	size_t textLength() {return GetWindowTextLength(handle);}
	
	/// スクロール情報。
	int setScrollInformation(int bar, LPSCROLLINFO si, bool redraw = true) {
		return SetScrollInfo(handle, bar, si, redraw ? TRUE : FALSE);
	}
	
	/// ditto
	void getScrollInformation(int bar, LPSCROLLINFO si) {
		checkApi(GetScrollInfo(handle, bar, si));
	}
	
	/// スクロール操作ミクスチャ。
	template ScrollMixture(int BAR) {
		/// 最小値。
		int scrollMin() {
			SCROLLINFO si;
			with(si) {
				cbSize = SCROLLINFO.sizeof;
				fMask = SIF_RANGE;
			}
			getScrollInformation(BAR, &si);
			return si.nMin;
		}
		
		/// ditto
		void scrollMin(int m) {
			SCROLLINFO si;
			with(si) {
				cbSize = SCROLLINFO.sizeof;
				fMask = SIF_RANGE;
				nMin = m;
			}
			setScrollInformation(BAR, &si);
		}
		
		/// 最大値。
		int scrollMax() {
			SCROLLINFO si;
			with(si) {
				cbSize = SCROLLINFO.sizeof;
				fMask = SIF_RANGE;
			}
			
			getScrollInformation(BAR, &si);
			return si.nMax;
		}
		
		/// ditto
		void scrollMax(int m) {
			SCROLLINFO si;
			with(si) {
				cbSize = SCROLLINFO.sizeof;
				fMask = SIF_RANGE;
				nMax = m;
			}
			
			setScrollInformation(BAR, &si);
		}
		
		/// ページ。
		size_t scrollPage() {
			SCROLLINFO si;
			with(si) {
				cbSize = SCROLLINFO.sizeof;
				fMask = SIF_PAGE;
			}
			getScrollInformation(BAR, &si);
			return si.nPage;
		}
		
		/// ditto
		void scrollPage(size_t p) {
			SCROLLINFO si;
			with(si) {
				cbSize = SCROLLINFO.sizeof;
				fMask = SIF_PAGE;
				nPage = p;
			}
			setScrollInformation(BAR, &si);
		}
		
		/// 位置。
		int scrollPosition() {
			SCROLLINFO si;
			with(si) {
				cbSize = SCROLLINFO.sizeof;
				fMask = SIF_POS;
			}
			getScrollInformation(BAR, &si);
			return si.nPos;
		}
		
		/// ditto
		void scrollPosition(int p) {
			SCROLLINFO si;
			with(si) {
				cbSize = SCROLLINFO.sizeof;
				fMask = SIF_POS;
				nPos = p;
			}
			setScrollInformation(BAR, &si);
		}
		
		/// トラッキング位置。
		int trackPosition() {
			SCROLLINFO si;
			with(si) {
				cbSize = SCROLLINFO.sizeof;
				fMask = SIF_TRACKPOS;
			}
			getScrollInformation(BAR, &si);
			return si.nTrackPos;
		}
		
		/// ditto
		void trackPosition(int tp) {
			SCROLLINFO si;
			with(si) {
				cbSize = SCROLLINFO.sizeof;
				fMask = SIF_TRACKPOS;
				nTrackPos = tp;
			}
			setScrollInformation(BAR, &si);
		}
	}
	
	/// 水平スクロールの操作。
	mixin ScrollMixture!(SB_HORZ) scrollH;
	
	/// 垂直スクロールの操作。
	mixin ScrollMixture!(SB_VERT) scrollV;
	
	/// メニュー。
	Menu menu() {
		if(auto m = GetMenu(handle)) {
			return Menu.fromHandle(m);
		}
		return null;
	}
	
	/// ditto
	void menu(Menu m) {
		if(m) {
			checkApi(SetMenu(handle, m.handle));
		} else {
			checkApi(SetMenu(handle, null));
		}
	}
	
	/// システムメニュー。
	Menu systemMenu() {
		if(auto m = GetSystemMenu(handle, FALSE)) {
			return Menu.fromHandle(m);
		}
		return null;
	}
	
	/// メニュー描画。
	void drawMenuBar() {checkApi(DrawMenuBar(handle));}
	
	/// メッセージ送信。
	LRESULT sendMessage(UINT msg, WPARAM wp = 0, LPARAM lp = 0) {return SendMessage(handle, msg, wp, lp);}
	
	/// メッセージ送信。
	void postMessage(UINT msg, WPARAM wp = 0, LPARAM lp = 0) {checkApi(SendMessage(handle, msg, wp, lp));}
	
	/// メッセージデリゲート。
	void setMessageDelegate(UINT id, MessageDelegate h) {handlers_[id] = h;}
	
	/// ditto
	MessageDelegate getMessageDelegate(UINT id) {
		auto found = id in handlers_;
		return (found) ? *found : null;
	}
	
	/// メッセージデリゲートの最適化。
	void rehashMessageDelegates() {
		handlers_.rehash;
		commands_.rehash;
	}
	
	/// ウィンドウ生成時の処理。
	void onCreate(OnCreateDelegate h) {
		setMessageDelegate(WM_CREATE, makeMessageDelegate!(OnCreateDelegate, translateOnCreate)(h));
	}
	
	/// ウィンドウ生成時の処理。
	void onNcCreate(OnCreateDelegate h) {
		setMessageDelegate(WM_NCCREATE, makeMessageDelegate!(OnCreateDelegate, translateOnNcCreate)(h));
	}
	
	/// ウィンドウ破棄時の処理。
	void onDestroy(OnDestroyDelegate h) {
		setMessageDelegate(WM_DESTROY, makeMessageDelegate!(OnDestroyDelegate, translateOnDestroy)(h));
	}
	
	/// ウィンドウ破棄時の処理。
	void onNcDestroy(OnDestroyDelegate h) {
		setMessageDelegate(WM_NCDESTROY, makeMessageDelegate!(OnDestroyDelegate, translateOnNcDestroy)(h));
	}
	
	/// マウス移動時の処理。
	void onMouseMove(OnMouseDelegate h) {
		setMessageDelegate(WM_MOUSEMOVE, makeMessageDelegate!(OnMouseDelegate, translateOnMouseMessage)(h));
	}
	
	/// マウスホイール回転時の処理。
	void onMouseWheel(OnMouseWheel h) {
		setMessageDelegate(WM_NCDESTROY, makeMessageDelegate!(OnMouseWheel, translateOnMouseWheel)(h));
	}
	
	/// マウス左ボタン押下時の処理。
	void onLButtonDown(OnMouseDelegate h) {
		setMessageDelegate(WM_LBUTTONDOWN, makeMessageDelegate!(OnMouseDelegate, translateOnMouseMessage)(h));
	}
	
	/// マウス左ボタン押上時の処理。
	void onLButtonUp(OnMouseDelegate h) {
		setMessageDelegate(WM_LBUTTONUP, makeMessageDelegate!(OnMouseDelegate, translateOnMouseMessage)(h));
	}
	
	/// マウス左ボタンダブルクリック時の処理。
	void onLButtonDoubleClick(OnMouseDelegate h) {
		setMessageDelegate(WM_LBUTTONUP, makeMessageDelegate!(OnMouseDelegate, translateOnMouseMessage)(h));
	}
	
	/// マウス中ボタン押下時の処理。
	void onMButtonDown(OnMouseDelegate h) {
		setMessageDelegate(WM_MBUTTONDOWN, makeMessageDelegate!(OnMouseDelegate, translateOnMouseMessage)(h));
	}
	
	/// マウス中ボタン押上時の処理。
	void onMButtonUp(OnMouseDelegate h) {
		setMessageDelegate(WM_MBUTTONUP, makeMessageDelegate!(OnMouseDelegate, translateOnMouseMessage)(h));
	}
	
	/// マウス中ボタンダブルクリック時の処理。
	void onMButtonDoubleClick(OnMouseDelegate h) {
		setMessageDelegate(WM_MBUTTONUP, makeMessageDelegate!(OnMouseDelegate, translateOnMouseMessage)(h));
	}
	
	/// マウス右ボタン押下時の処理。
	void onRButtonDown(OnMouseDelegate h) {
		setMessageDelegate(WM_RBUTTONDOWN, makeMessageDelegate!(OnMouseDelegate, translateOnMouseMessage)(h));
	}
	
	/// マウス右ボタン押上時の処理。
	void onRButtonUp(OnMouseDelegate h) {
		setMessageDelegate(WM_RBUTTONUP, makeMessageDelegate!(OnMouseDelegate, translateOnMouseMessage)(h));
	}
	
	/// マウス右ボタンダブルクリック時の処理。
	void onRButtonDoubleClick(OnMouseDelegate h) {
		setMessageDelegate(WM_RBUTTONUP, makeMessageDelegate!(OnMouseDelegate, translateOnMouseMessage)(h));
	}
	
	static if(_WIN32_WINNT >= 0x500) {
	
		/// マウスボタン押下時の処理。
		void onButtonDown(MouseHandler h) {
			setMessageDelegate(WM_XBUTTONDOWN, makeMessageDelegate!(OnMouseDelegate, translateOnMouseXMessage)(h));
		}
		
		/// マウスボタン押上時の処理。
		void onButtonUp(MouseHandler h) {
			setMessageDelegate(WM_XBUTTONUP, makeMessageDelegate!(OnMouseDelegate, translateOnMouseXMessage)(h));
		}
		
		/// マウスボタンダブルクリック時の処理。
		void onButtonDoubleClick(MouseHandler h) {
			setMessageDelegate(WM_XBUTTONUP, makeMessageDelegate!(OnMouseDelegate, translateOnMouseXMessage)(h));
		}
	}
	
	/// コマンドメッセージの処理。
	void onCommandMessage(OnCommandDelegate h) {
		setMessageDelegate(WM_COMMAND, makeMessageDelegate!(OnCommandDelegate, translateOnCommand)(h));
	}
	
	/// オーナー項目比較処理。
	void onCompareItem(OnCompareItemDelegate h) {
		setMessageDelegate(WM_COMPAREITEM, makeMessageDelegate!(OnCompareItemDelegate, translateOnCompareItem)(h));
	}
	
	/// コントロール種類。
	enum ControlType {
		CTL_BUTTON	= WM_CTLCOLORBTN - WM_CTLCOLORMSGBOX,		/// ボタン。
		CTL_DIALOG	= WM_CTLCOLORDLG - WM_CTLCOLORMSGBOX,		/// ダイアログ。
		CTL_EDIT	= WM_CTLCOLOREDIT - WM_CTLCOLORMSGBOX,		/// エディットボックス。
		CTL_LISTBOX	= WM_CTLCOLORLISTBOX - WM_CTLCOLORMSGBOX,	/// リストボックス。
		CTL_MSGBOX	= WM_CTLCOLORMSGBOX - WM_CTLCOLORMSGBOX,	/// メッセージボックス。
		CTL_SCROLL	= WM_CTLCOLORSCROLLBAR - WM_CTLCOLORMSGBOX,	/// スクロールバー。
		CTL_STATIC	= WM_CTLCOLORSTATIC - WM_CTLCOLORMSGBOX,	/// スタティックテキスト。
	}
	
	/// ボタン背景色の設定。
	void onControlColorButton(OnControlColorDelegate h) {
		setMessageDelegate(WM_CTLCOLORBTN, makeMessageDelegate!(OnControlColorDelegate, translateOnControlColor)(h));
	}
	
	/// ダイアログ背景色の設定。
	void onControlColorDialog(OnControlColorDelegate h) {
		setMessageDelegate(WM_CTLCOLORDLG, makeMessageDelegate!(OnControlColorDelegate, translateOnControlColor)(h));
	}
	
	/// エディットボックス背景色の設定。
	void onControlColorEdit(OnControlColorDelegate h) {
		setMessageDelegate(WM_CTLCOLOREDIT, makeMessageDelegate!(OnControlColorDelegate, translateOnControlColor)(h));
	}
	
	/// リストボックス背景色の設定。
	void onControlColorListBox(OnControlColorDelegate h) {
		setMessageDelegate(WM_CTLCOLORLISTBOX, makeMessageDelegate!(OnControlColorDelegate, translateOnControlColor)(h));
	}
	
	/// メッセージボックス背景色の設定。
	void onControlColorMessageBox(OnControlColorDelegate h) {
		setMessageDelegate(WM_CTLCOLORMSGBOX, makeMessageDelegate!(OnControlColorDelegate, translateOnControlColor)(h));
	}
	
	/// スクロールバー背景色の設定。
	void onControlColorScrollBar(OnControlColorDelegate h) {
		setMessageDelegate(WM_CTLCOLORSCROLLBAR, makeMessageDelegate!(OnControlColorDelegate, translateOnControlColor)(h));
	}
	
	/// スタティックテキスト背景色の設定。
	void onControlColorStatic(OnControlColorDelegate h) {
		setMessageDelegate(WM_CTLCOLORSTATIC, makeMessageDelegate!(OnControlColorDelegate, translateOnControlColor)(h));
	}
	
	/// オーナー項目削除処理。
	void onDeleteItem(OnDeleteItemDelegate h) {
		setMessageDelegate(WM_DELETEITEM, makeMessageDelegate!(OnDeleteItemDelegate, translateOnDeleteItem)(h));
	}
	
	/// オーナードロー処理。
	void onDrawItem(OnDrawItemDelegate h) {
		setMessageDelegate(WM_DRAWITEM, makeMessageDelegate!(OnDrawItemDelegate, translateOnDrawItem)(h));
	}
	
	/// 水平スクロール時の処理。
	void onHScroll(OnScrollDelegate h) {
		setMessageDelegate(WM_HSCROLL, makeMessageDelegate!(OnScrollDelegate, translateOnHScroll)(h));
	}

	/// オーナーサイズ処理。
	void onMeasureItem(OnMeasureItemDelegate h) {
		setMessageDelegate(WM_MEASUREITEM, makeMessageDelegate!(OnMeasureItemDelegate, translateOnMeasureItem)(h));
	}
	
	/// サイズ変更時の処理。
	void onSize(OnSizeDelegate h) {
		setMessageDelegate(WM_SIZE, makeMessageDelegate!(OnSizeDelegate, translateOnSize)(h));
	}
	
	/// システムコマンドの処理。
	void onSystemCommand(OnSystemCommandDelegate h) {
		setMessageDelegate(WM_SYSCOMMAND, makeMessageDelegate!(OnSystemCommandDelegate, translateOnSystemCommand)(h));
	}

	/// タイマー処理。
	void onTimer(OnTimerDelegate h) {
		setMessageDelegate(WM_TIMER, makeMessageDelegate!(OnTimerDelegate, translateOnTimer)(h));
	}
	
	/// 垂直スクロール時の処理。
	void onVScroll(OnScrollDelegate h) {
		setMessageDelegate(WM_VSCROLL, makeMessageDelegate!(OnScrollDelegate, translateOnVScroll)(h));
	}
	
	/// コマンドメッセージの登録。
	void onCommand(uint id, uint notify, CommandDelegate h) {commands_[CommandKey(id, notify)] = h;}
	
	/// ditto
	void onCommand(uint id, CommandDelegate h) {
		onMenu(id, h);
		onAccelerator(id, h);
	}
	
	/// メニューハンドラの登録。
	void onMenu(uint id, CommandDelegate h) {onCommand(id, 0, h);}
	
	/// アクセラレータハンドラの登録。
	void onAccelerator(uint id, CommandDelegate h) {onCommand(id, 1, h);}
	
	/// 返送コマンドメッセージの登録。
	void onReflectCommand(uint notify, CommandDelegate h) {onCommand(ID_SELF, notify, h);}
	
	/**
	 *	返送コントロールカラーメッセージの処理。
	 *
	 *	Params:
	 *		dc = 設定対象のデバイスコンテキスト。
	 *
	 *	Returns:
	 *		背景を塗りつぶすためのブラシ。nullの場合は処理しなかったことを示す。
	 */
	HBRUSH onReflectControlColor(DeviceContext dc) {return cast(HBRUSH) null;}
	
protected:
	
	/// デフォルトコンストラクタ。
	this() {
		// スレッド状態がまだなければ生成する。
		if(!threadState_.value) {
			threadState_.value = new ThreadState;
		}
	}

	/** ウィンドウ破棄後の処理。
	 *
	 *	ウィンドウハンドルは既に所有していないことに注意。デフォルトでは何もしない。
	 */
	void postFinalizeWindow() {}
	
	/** コントロールカラーメッセージの処理。
	 *
	 *	デフォルトではデフォルトプロシージャを呼び出す。
	 *
	 *	Params:
	 *		t	= ウィンドウの種類。
	 *		dc	= 設定対象のデバイスコンテキスト。
	 *		w	= 設定対象のウィンドウ。
	 *
	 *	Returns:
	 *		背景を塗りつぶすためのブラシ。
	 *		nullの場合は処理しなかったことを示す。
	 */
	HBRUSH onControlColor(ControlType t, DeviceContext dc, Window w) {
		return cast(HBRUSH) callDefaultProcedure();
	}
	
	/// コマンドメッセージのルーティング。
	LRESULT routeCommand(uint id, uint notify, Window w) {
		// 親ウィンドウにコマンドハンドラがあるか探す。
		auto h = CommandKey(id, notify) in commands_;
		if(h) {
			(*h)();
			return 0;
		}
		
		// 子ウィンドウに返送ハンドラがあるか探す。
		if(w && (h = CommandKey(ID_SELF, notify) in w.commands_) !is null) {
			(*h)();
			return 0;
		}
		
		// デフォルト処理。
		return callDefaultProcedure();
	}
	
	/// コントロールカラーのルーティング。
	LRESULT routeControlColor(uint msg, Window w, DeviceContext dc)
	in {
		assert(w !is null);
	} body {
		// 返送を試みる。返送処理が行われなければ自前で処理を行う。
		LRESULT result = cast(LRESULT) w.onReflectControlColor(dc);
		if(!result) {
			result = cast(LRESULT) onControlColor(cast(ControlType)(msg - WM_CTLCOLORMSGBOX), dc, w);
		}
		return result;
	}
	
	/// サブクラス化。
	void subclass(HWND wnd)
	in {
		assert(defaultProcedure_ is null);
		assert(handle_ is null);
		assert(IsWindow(wnd));
	} out {
		assert(IsWindow(handle_));
		assert(defaultProcedure_ !is null);
	} body {
		// ウィンドウプロシージャの置き換え。
		defaultProcedure_ = cast(WNDPROC) SetWindowLongPtr(wnd, GWL_WNDPROC, cast(LONG_PTR) &windowProcedure);
		checkApi(defaultProcedure_);
		
		// ハンドル登録失敗時にはサブクラス化を元に戻す。
		scope(failure) {
			SetWindowLongPtr(wnd, GWL_WNDPROC, cast(LONG_PTR) defaultProcedure_);
			defaultProcedure_ = null;
		}
		
		// ハンドルを登録する。
		HandleDatabase.instance.register(wnd, this);
		handle_ = wnd;
	}
	
	/// ウィンドウプロシージャ。
	LRESULT procedure(HWND wnd, UINT msg, WPARAM wp, LPARAM lp) {
		// メッセージ処理結果。
		LRESULT result = 0;
		
		// デフォルト呼び出しのために現在のメッセージを記憶。
		auto oldMessage = threadState.message;
		scope(exit) threadState.message = oldMessage;
		with(threadState.message) {
			hwnd = wnd;
			message = msg;
			wParam = wp;
			lParam = lp;
			time = Application.instance.currentMessage.time;
			pt = Application.instance.currentMessage.pt;
		}
		
		// メッセージハンドラに処理を渡す。
		if(MessageDelegate* dg = msg in handlers_) {
			result = (*dg)(this, msg, wp, lp);
		} else {
			// 生成メッセージの場合は最小サイズを設定する。
			if(msg == WM_CREATE && lp) {
				auto cs = cast(LPCREATESTRUCT) lp;
				minSize_ = Size(cs.cx, cs.cy);
			}
			
			// 対応するメッセージハンドラがない場合はメッセージIDに従って処理を行う。
			switch(msg) {
			// コマンドハンドラを検索して実行する。
			case WM_COMMAND:
				result = routeCommand(LOWORD(wp), HIWORD(wp), Window.fromHandle(cast(HWND) lp));
				break;
			// コントロールカラーの処理。
			case WM_CTLCOLORBTN:
			case WM_CTLCOLORDLG:
			case WM_CTLCOLOREDIT:
			case WM_CTLCOLORLISTBOX:
			case WM_CTLCOLORMSGBOX:
			case WM_CTLCOLORSCROLLBAR:
			case WM_CTLCOLORSTATIC:
				if(Window w = Window.fromHandle(cast(HWND) lp)) {
					scope dc = new DeviceContext(cast(HDC) wp, null);
					result = routeControlColor(msg, w, dc);
				}
				break;
			// デフォルト処理。
			default:
				result = callDefaultProcedure();
				break;
			}
		}
		
		// ウィンドウ破棄時の処理。
		if(msg == WM_NCDESTROY) {
			// メニューの登録解除。
			if(auto m = GetMenu(handle)) {
				HandleDatabase.instance.unregister(m);
			}
			if(auto m = GetSystemMenu(handle, FALSE)) {
				HandleDatabase.instance.unregister(m);
			}
			
			// サブクラス化解除。
			unsubclass();
			
			// ウィンドウ破棄時の処理呼び出し。
			postFinalizeWindow();
		}
		
		return result;
	}
	
	/// デフォルトのウィンドウプロシージャ呼び出し。
	LRESULT callDefaultProcedure() {
		with(threadState.message) {
			return CallWindowProc(defaultProcedure_, hwnd, message, wParam, lParam);
		}
	}
	
	/// ウィンドウ生成。
	void createWindow(LPCTSTR cls, char[] title, DWORD style, DWORD styleEx, Rect r, HWND parent, HMENU menu, HINSTANCE inst, void* ptr) {
		LPTSTR str = toTString(title);
		duringCreateHook({
			checkApi(CreateWindowEx(styleEx, cls, str, style, r.x, r.y, r.cx, r.cy, parent, menu, inst, ptr));
		});
	}
	
	/** ウィンドウ生成フック中に処理を行う。
	 *
	 *	Params:
	 *		fn = 実行する処理。この処理の中でフックを解除すること。
	 */
	void duringCreateHook(void delegate() fn)
	in {
		assert(threadState.initWindow is null);
		assert(!threadState.hook);
	} out {
		assert(threadState.initWindow is null);
		assert(!threadState.hook);
	} body {
		// スレッドローカル領域に記憶。
		threadState.initWindow = this;
		
		// ウィンドウフックの開始。
		threadState.hook = SetWindowsHookEx(WH_CBT, &hookProcedure, Application.instance.handle, GetCurrentThreadId());
		checkApi(threadState.hook);
		
		// 処理実行。
		fn();
	}
	
private:
	
	/// 初期化データ。
	struct ThreadState {
		/// 初期化中ウィンドウ。
		Window initWindow;
		
		/// フックハンドル。
		HHOOK hook;
		
		/// 処理中メッセージ。
		MSG message;
	}
	
	/// 初期化データ。
	static ThreadState* threadState() {return cast(ThreadState*) threadState_.value;}
	
	/// 初期化データストレージ。
	static ThreadLocal!(ThreadState) threadState_;
	
	/// 静的コンストラクタ。
	static this() {
		threadState_ = new ThreadLocal!(ThreadState)();
	}
	
	/// 生成されたウィンドウのためのウィンドウプロシージャ。
	extern(Windows) static LRESULT windowProcedure(HWND wnd, UINT msg, WPARAM wp, LPARAM lp) {
		// ウィンドウに対応するオブジェクトを検索して処理を行う。
		if(auto w = cast(Window) HandleDatabase.instance[wnd]) {
			return  w.procedure(wnd, msg, wp, lp);
		}
		
		// 対応オブジェクトなし。デフォルト処理を行う。本来はありえない。
		assert(false);
		return DefWindowProc(wnd, msg, wp, lp);
	}
	
	/// 生成フックプロシージャ。
	extern(Windows) static  LRESULT hookProcedure(int code, WPARAM wp, LPARAM lp)
	in {
		assert(threadState !is null);
		assert(threadState.initWindow !is null);
		assert(threadState.hook);
	} body {
		// 無視するコード。
		if(code < 0) {
			return CallNextHookEx(threadState.hook, code, wp, lp);
		} else if(code != HCBT_CREATEWND) {
			return 0;
		}
		
		// フック解除。
		UnhookWindowsHookEx(threadState.hook);
		threadState.hook = cast(HHOOK) null;
		auto w = threadState.initWindow;
		threadState.initWindow = null;
		
		// サブクラス化。
		try {
			w.subclass(cast(HWND) wp);
		} catch(Object o) {
			// 例外発生時は中断。
			return 1;
		}
		
		// 初期化成功。
		return 0;
	}
	
	/// 描画処理の実行。
	void duringPaint(void delegate(DeviceContext, LPPAINTSTRUCT) dg) {
		// 描画開始。
		PAINTSTRUCT ps;
		auto h = BeginPaint(handle, &ps);
		checkApi(h);
		
		// 描画終了の予約。
		scope(exit) EndPaint(handle, &ps);

		// 描画の実行。
		scope dc = new DeviceContext(h, null);
		dg(dc, &ps);
	}
	
	/// コマンド検索時のキー。
	struct CommandKey {
		
		/// 生成する。
		static CommandKey opCall(WPARAM wp) {return opCall(LOWORD(wp), HIWORD(wp));}
		
		/// 生成する。
		static CommandKey opCall(uint id, uint notify) {
			CommandKey key;
			key.id = id;
			key.notify = notify;
			return key;
		}
		
		uint id;		/// コントロールID。
		uint notify;	/// 通知コード。
	}
	
	/// ハンドル。
	HWND handle_;
	
	/// 元のウィンドウプロシージャ。
	WNDPROC defaultProcedure_;
	
	/// メッセージハンドラテーブル。
	MessageDelegate[UINT] handlers_;
	
	/// コマンドメッセージハンドラテーブル。
	CommandDelegate[CommandKey] commands_;
	
	/// グローバルなコントロールカラーハンドラ。
	OnControlColorDelegate controlColor_;
	
	/// 最小サイズ。
	Size minSize_;
}

/// 親ウィンドウ。
class ParentWindow : Window {
	
	/// デフォルトコンストラクタ。
	this() {
		onSize = &handleOnSize;
	}
	
	/// レイアウト。
	void layout(LayoutItem l) {
		layout_ = l;
	}
	
	/// ditto
	LayoutItem layout() {return layout_;}
	
	/// サイズ変更時の処理。
	protected void handleOnSize(Window w, uint f, Size s) {
		if(layout_) {
			scope def = new DeferWindowPosition;
			layout_.doLayout(def, Rect(Point(0, 0), s));
		}
	}
	
	/// レイアウト項目。
	private LayoutItem layout_;
}

/// ウィンドウクラス管理オブジェクト。
class WindowClassDatabase {
	
	/// 唯一のインスタンスを返す。
	static WindowClassDatabase instance() {return instance_;}
	
	/// ウィンドウクラスを登録する。
	synchronized ATOM register(
			DWORD st, HICON icon, HICON sicon, HCURSOR cursor, HBRUSH brush, LPCTSTR menu) {
		// クラスの設定に従った文字列を生成する。
		auto str = toTString(
			format("DMAJOR_WINDOW_CLASS_%d%d%d%d%d%d", st, icon, sicon, cursor, brush, menu));
		TCHAR[] name = str[0 .. lstrlen(str)];
		
		// 既に登録されていた場合はそのままクラス名を返す。
		if(auto found = name in atoms_) {
			return *found;
		}
		
		// ウィンドウクラス構造体の設定。
		WNDCLASSEX cls;
		with(cls) {
			cbSize = cls.sizeof;
			style = st;
			lpfnWndProc = &DefWindowProc;
			hInstance = Application.instance.handle;
			hIcon = icon;
			hCursor = cursor;
			hbrBackground = brush;
			lpszMenuName = menu;
			lpszClassName = name.ptr;
			hIconSm = sicon;
		}
				
		// 登録。
		auto atom = RegisterClassEx(&cls);
		checkApi(atom);
				
		// ハッシュ登録失敗時は登録解除。
		scope(failure) UnregisterClass(MAKEINTRESOURCE(atom), Application.instance.handle);
	
		// ハッシュに登録。
		names_[atom] = name;
		scope(failure) names_.remove(atom);
		
		atoms_[name] = atom;
		return atom;
	}
	
	/// ウィンドウクラスを登録解除する。
	synchronized void unregister(ATOM atom) {
		TCHAR[] name = names_[atom];
		names_.remove(atom);
		atoms_.remove(name);
		checkApi(UnregisterClass(MAKEINTRESOURCE(atom), Application.instance.handle));
	}
	
private:
	
	/// デフォルトコンストラクタ。
	this() {}
	
	/// 静的コンストラクタ。
	static this() {instance_ = new WindowClassDatabase;}
	
	/// ウィンドウクラスアトム。
	ATOM[TCHAR[]] atoms_;
	
	/// ウィンドウクラス名。
	TCHAR[][ATOM] names_;
	
	/// 唯一のインスタンス。
	static WindowClassDatabase instance_;
}
