﻿/**
 *	連結リストのモジュール。
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module outland.tl.list;

import std.string;

public import outland.tl.algorithm : swap;

/// 両方向連結リストのクラス。
class List(T) {
	
	/// 反復子。
	struct Iterator {
		
		/// コンストラクタ。
		static Iterator opCall(List list)
		in {
			assert(list);
		} body {
			return opCall(list.begin_);
		}
		
		/// コンストラクタ。
		static Iterator opCall(Entry* entry) {
			Iterator iter;
			iter.entry_ = entry;
			return iter;
		}
		
		/// 反復子のクローンを新たに生成する。
		Iterator clone() {return *this;}
		
		/// 次の要素を得る。
		T next() {
			Entry* next = entry_.next;
			T val = entry_.value;
			entry_.next = next;
			return val;
		}
		
		/// 取り出せる要素があるか返す。
		bool hasMore() {return entry_ !is null;}
		
	private:
		
		/// エントリ。
		Entry* entry_;
	}
	
	/// デフォルトコンストラクタ。
	this() {}
	
	/// コピーコンストラクタ。
	this(List src)
	in {
		assert(src is null);
	} body {
		this();
		foreach(e; src) pushBack(e);
	}
	
	/// デストラクタ。
	~this() {clear();}
	
	/// 全要素の破棄。
	void clear() {
		if(begin_) {
			// 末尾と開始を切断。
			begin_.prev.next = null;
			
			// 全要素を破棄。
			for(Entry* e = begin_; e;) {
				Entry* next = e.next;
				deallocate(e);
				e = next;
			}
			begin_ = null;
			length_ = 0;
		}
	}
	
	/// サイズを得る。
	size_t length() {return length_;}
	
	/// 先頭に要素を追加する。
	void pushFront(T value) {
		// 末尾に追加。
		pushBack(value);
		// 環状リストを回転させて追加された末尾を先頭にする。
		begin_ = begin_.prev;
	}
	
	/// 先頭の要素を削除する。
	void popFront() {
		if(begin_) {
			Entry* front = begin_;
			
			// 最後の要素だった場合はbeginも破棄。
			// 要素が残っている場合はポインタを付け替える。
			if(front.next == begin_) {
				begin_ = null;
			} else {
				front.next.prev = front.prev;
				front.prev.next = front.next;
				begin_ = front.next;
			}
			deallocate(front);
			--length_;
		}
	}
	
	/// 末尾に要素を追加する。
	void pushBack(T value) {
		Entry* entry = allocate();
		entry.value = value;
		if(begin_) {
			Entry* before = begin_.prev;
			begin_.prev = entry;
			before.next = entry;
			entry.next = begin_;
			entry.prev = before;
		} else {
			// 最初の要素だった場合。
			entry.next = entry;
			entry.prev = entry;
			begin_ = entry;
		}
		++length_;
	}
	
	/// 末尾の要素を削除する。
	void popBack() {
		if(begin_) {
			Entry* back = begin_.prev;
			
			// 最後の要素だった場合はbeginも破棄。
			// 要素が残っている場合はポインタを付け替える。
			if(back == begin_) {
				begin_ = null;
			} else {
				back.next.prev = back.prev;
				back.prev.next = back.next;
			}
			deallocate(back);
			--length_;
		}
	}
	
	/// 先頭の要素を得る。
	T front()
	in {
		assert(begin_ !is null);
	} body {
		return begin_.value;
	}
	
	/// 末尾の要素を得る。
	T back()
	in {
		assert(begin_ !is null && begin_.prev !is null);
	} body {
		return begin_.prev.value;
	}
	
	/// 巡回する。
	int opApply(int delegate(inout T) dg) {
		if(Entry* e = begin_) {
			for(;; e = e.next) {
				if(int result = dg(e.value)) return result;
				if(e.next is begin_) break;
			}
		}
		return 0;
	}
	
	/// 巡回する。
	int opApply(int delegate(inout size_t, inout T) dg) {
		if(Entry* e = begin_) {
			for(size_t i = 0; ; ++i, e = e.next) {
				if(int result = dg(i, e.value)) return result;
				if(e.next is begin_) break;
			}
		}
		return 0;
	}

	/// 逆順に巡回する。
	int opApplyReverse(int delegate(inout T) dg) {
		if(begin_) {
			for(Entry* e = begin_.prev; ; e = e.prev) {
				if(int result = dg(e.value)) return result;
				if(e is begin_) break;
			}
		}
		return 0;
	}

	
private:
	
	/// リストのエントリ。
	struct Entry {
		Entry* next;
		Entry* prev;
		T value;
	}
	
	/// エントリの新規作成。
	/// いずれオブジェクトプールで実装するかもしれない。
	static Entry* allocate() {return new Entry;}
	
	/// エントリの破棄。
	/// いずれオブジェクトプールで実装するかもしれない。
	static void deallocate(Entry* e) {delete e;}
	
	/// 始点。
	Entry* begin_;
	
	/// 要素数。
	size_t length_;
}

unittest {
	auto list = new List!(int);
	
	list.pushBack(1);
	list.pushBack(2);
	list.pushBack(3);
	
	assert(list.front == 1);
	list.popFront();
	assert(list.front == 2);
	list.popFront();
	assert(list.front == 3);
	list.popFront();
	assert(list.length == 0);
	
	list.pushFront(1);
	list.pushFront(2);
	list.pushFront(3);
	
	assert(list.back == 1);
	list.popBack();
	assert(list.back == 2);
	list.popBack();
	assert(list.back == 3);
	list.popBack();
	assert(list.length == 0);
	
	list.pushFront(1);	// 1
	list.pushBack(2);	// 12
	list.pushFront(3);	// 312
	list.pushBack(4);	// 3124
	list.pushFront(5);	// 53124
	list.pushBack(6);	// 531246
	
	// 531246
	assert(list.front == 5);
	assert(list.back == 6);
	assert(list.length == 6);
	
	// 53124
	list.popBack();
	assert(list.front == 5);
	assert(list.back == 4);
	assert(list.length == 5);
	
	// 3124
	list.popFront();
	assert(list.front == 3);
	assert(list.back == 4);
	assert(list.length == 4);
	
	// 312
	list.popBack();
	assert(list.front == 3);
	assert(list.back == 2);
	assert(list.length == 3);
	
	// 12
	list.popFront();
	assert(list.front == 1);
	assert(list.back == 2);
	assert(list.length == 2);
	
	// 1
	list.popBack();
	assert(list.front == 1);
	assert(list.back == 1);
	assert(list.length == 1);
	
	// 空
	list.popFront();
	assert(list.length == 0);
	
	list.pushBack(list.length);
	list.pushBack(list.length);
	list.pushBack(list.length);
	list.pushBack(list.length);
	list.pushBack(list.length);
	list.pushBack(list.length);
	foreach(i, e; list) assert(i == e);

	list.clear();
	assert(list.length == 0);
	foreach(e; list) {assert(false);}
	list.pushFront(999);
	foreach(e; list) {assert(e == 999);}
	foreach_reverse(e; list) {assert(e == 999);}
}

/// リスト操作ミクスチャ。
template ListMixture() {
	
	// ポインタ型の定義。
	alias typeof(this) Pointer;
	
	/// 検索する。見つからなかった場合はendが返る。
	Pointer find(bool delegate(Pointer) pred, Pointer end = null) {
		Pointer p = this;
		for(; p !is end && !pred(p); p = p.next) {}
		return p;
	}
	
	/// 検索する。見つからなかった場合はbeginが返る。
	Pointer findReverse(bool delegate(Pointer) pred, Pointer begin = null) {
		Pointer p = this;
		for(; p !is begin && !pred(p); p = p.before) {}
		return p;
	}
	
	/// 要素の切り離し。
	void erase() {
		if(before) {
			before.next = next;
		}
		if(next) {
			next.before = before;
		}
		
		before = null;
		next = null;
	}
	
	/// 要素の挿入。
	void insertBefore(Pointer src)
	in {
		assert(src !is null);
	} body {
		// 以前のリストからの切り離し。
		src.erase();
		
		// 繋ぐ。
		src.before = before;
		if(before !is null) {
			before.next = src;
		}
		before = src;
		src.next = this;
	}
	
	/// ditto
	void insertNext(Pointer src)
	in {
		assert(src !is null);
	} body {
		// 以前のリストからの切り離し。
		src.erase();
		
		// 繋ぐ。
		src.next = next;
		if(next !is null) {
			next.before = src;
		}
		next = src;
		src.before = this;
	}
	
	/// 逆順にする。
	void reverse(Pointer end = null) {
		// 要素が1つだけの場合は何もしない。
		if(end is this || next is end) {
			return;
		}
		
		// 最初と最後の要素を記憶する。
		auto front = this;
		auto back = this;
		
		// リンクを反転する。
		for(Pointer p = this; p !is end; back = p, p = p.before) {
			swap(p.before_, p.next_);
		}
		
		// 端を繋ぎかえる。
		if(back.before !is null) {
			back.before.before = front;
		}
		if(front.next !is null) {
			front.next.next = before;
		}
		swap(front.next_, back.before_);
	}
	
	/// 連結する。[begin, end)
	void spliceBefore(Pointer begin, Pointer end = null)
	in {
		assert(begin !is null);
	} body {
		// beginを切り離す。
		if(begin.before !is null) {
			begin.before.next = end;
		}
		
		// 終端を切り離す。
		if(end !is null) {
			end.before = begin.before;
		}
		
		// 連結する最後の要素。
		auto back = begin.back(end);
		assert(back !is null);
		
		// 連結前の前の要素を設定。
		if(before !is null) {
			before.next = begin;
		}
		begin.before = before;
		
		before = back;
		back.next = this;
	}
	
	/// 連結する。[begin, end)
	void spliceNext(Pointer begin, Pointer end = null)
	in {
		assert(begin !is null);
	} body {
		// beginを切り離す。
		if(begin.before !is null) {
			begin.before.next = end;
		}
		
		// 終端を切り離す。
		if(end !is null) {
			end.before = begin.before;
		}
		
		// 連結する最後の要素。
		auto back = begin.back(end);
		assert(back !is null);
		
		// 連結前の次の要素を設定。
		if(next !is null) {
			next.before = back;
		}
		back.next = next;
		
		// 次に繋ぐ。
		next = begin;
		begin.before = this;
	}
	
	/// 先頭を探す。
	Pointer front(Pointer begin = null) {
		auto p = this;
		for(; p.before !is begin; p = p.before) {}
		return p;
	}
	
	/// 終端を探す。
	Pointer back(Pointer end = null) {
		auto p = this;
		for(; p.next !is end; p = p.next) {}
		return p;
	}
	
	/// 巡回する。
	int opApply(int delegate(inout Pointer) dg) {
		for(Pointer p = this; p !is null; p = p.next) {
			if(int result = dg(p)) return result;
		}
		return 0;
	}
	
	/// 巡回する。
	int opApply(int delegate(inout size_t, inout Pointer) dg) {
		size_t i = 0;
		for(Pointer p = this; p !is null; p = p.next, ++i) {
			if(int result = dg(i, p)) return result;
		}
		return 0;
	}

	/// 逆順に巡回する。
	int opApplyReverse(int delegate(inout Pointer) dg) {
		for(Pointer p = this; p !is null; p = p.before) {
			if(int result = dg(p)) return result;
		}
		return 0;
	}
	
	/// 次の要素の設定。
	Pointer next() {return next_;}
	
	/// 前の要素の設定。
	Pointer before() {return before_;}
	
	/// 上に移動。
	void up(Pointer end = null) {
		auto p = next;
		if(p !is end) {
			erase();
			p.insertNext(this);
		}
	}
	
	/// 下に移動。
	void down(Pointer begin = null) {
		auto p = before;
		if(p !is begin) {
			erase();
			p.insertBefore(this);
		}
	}
	
private:

	/// 次の要素の設定。
	void next(Pointer p) {next_ = p;}
	
	/// 前の要素の設定。
	void before(Pointer p) {before_ = p;}

	/// 次の要素。
	Pointer next_;
	
	/// 前の要素。
	Pointer before_;
}

unittest {
	struct Data {
		mixin ListMixture;
		int value;
	}
	
	Data d1, d2, d3;
	
	// d2 d3 d1
	d1.insertBefore(&d2);
	d1.insertBefore(&d3);
	
	assert(d1.next is null);
	assert(d3.next is &d1);
	assert(d2.next is &d3);
	
	assert(d1.before is &d3);
	assert(d3.before is &d2);
	assert(d2.before is null);
}

unittest {
	struct Data {
		mixin ListMixture;
		int value;
	}
	
	Data d1, d2, d3;
	
	// d1 d3 d2
	d1.insertNext(&d2);
	d1.insertNext(&d3);
	
	assert(d1.next is &d3);
	assert(d3.next is &d2);
	assert(d2.next is null);
	
	assert(d1.before is null);
	assert(d3.before is &d1);
	assert(d2.before is &d3);
}

unittest {
	struct Data {
		mixin ListMixture;
		int value;
	}
	
	Data d1, d2, d3, d4;
	
	// d1 d2 d3 d4
	d1.insertNext(&d2);
	d2.insertNext(&d3);
	d3.insertNext(&d4);
	
	// d1 d2 d4
	d3.erase();

	assert(d1.next is &d2);
	assert(d2.next is &d4);
	assert(d3.next is null);
	assert(d4.next is null);
	
	assert(d1.before is null);
	assert(d2.before is &d1);
	assert(d3.before is null);
	assert(d4.before is &d2);
	
	// d1 d4
	d2.erase();

	assert(d1.next is &d4);
	assert(d2.next is null);
	assert(d3.next is null);
	assert(d4.next is null);
	
	assert(d1.before is null);
	assert(d2.before is null);
	assert(d3.before is null);
	assert(d4.before is &d1);
}

unittest {
	struct Data {
		mixin ListMixture;
		int value;
	}
	
	Data d1, d2, d3, d4;
	
	// d1 d2 d3 d4
	d1.insertNext(&d2);
	d2.insertNext(&d3);
	d3.insertNext(&d4);
	
	// d2 d3 d4
	d1.erase();

	assert(d1.next is null);
	assert(d2.next is &d3);
	assert(d3.next is &d4);
	assert(d4.next is null);
	
	assert(d1.before is null);
	assert(d2.before is null);
	assert(d3.before is &d2);
	assert(d4.before is &d3);
	
	// d2 d3
	d4.erase();

	assert(d1.next is null);
	assert(d2.next is &d3);
	assert(d3.next is null);
	assert(d4.next is null);
	
	assert(d1.before is null);
	assert(d2.before is null);
	assert(d3.before is &d2);
	assert(d4.before is null);
}

unittest {
	struct Data {
		mixin ListMixture;
		int value;
	}
	
	Data d1, d2, d3, d4;
	
	// d1 d2 d3 d4
	d1.insertNext(&d2);
	d2.insertNext(&d3);
	d3.insertNext(&d4);
	
	// d2 d1 d3 d4
	d1.reverse(&d3);
	
	assert(d2.next is &d1);
	assert(d1.next is &d3);
	assert(d3.next is &d4);
	assert(d4.next is null);
	
	assert(d2.before is null);
	assert(d1.before is &d2);
	assert(d3.before is &d1);
	assert(d4.before is &d3);
	
	// d2 d1 d4 d3
	d3.reverse();
	
	assert(d2.next is &d1);
	assert(d1.next is &d4);
	assert(d4.next is &d3);
	assert(d3.next is null);
	
	assert(d2.before is null);
	assert(d1.before is &d2);
	assert(d4.before is &d1);
	assert(d3.before is &d4);
	
	// d2 d1 d4 d3
	d4.reverse(&d4);
	
	assert(d2.next is &d1);
	assert(d1.next is &d4);
	assert(d4.next is &d3);
	assert(d3.next is null);
	
	assert(d2.before is null);
	assert(d1.before is &d2);
	assert(d4.before is &d1);
	assert(d3.before is &d4);
	
	// d3 d4 d1 d2 
	d2.reverse();
	
	assert(d3.next is &d4);
	assert(d4.next is &d1);
	assert(d1.next is &d2);
	assert(d2.next is null);
	
	assert(d3.before is null);
	assert(d4.before is &d3);
	assert(d1.before is &d4);
	assert(d2.before is &d1);
}

unittest {
	struct Data {
		mixin ListMixture;
		int value;
	}
	
	Data d1, d2, d3, d4;
	
	// d1 d2 d3 d4
	d1.insertNext(&d2);
	d2.insertNext(&d3);
	d3.insertNext(&d4);
	
	// d2 d3 d4 d1
	d1.spliceBefore(&d2);
	
	assert(d2.next is &d3);
	assert(d3.next is &d4);
	assert(d4.next is &d1);
	assert(d1.next is null);
	
	assert(d2.before is null);
	assert(d3.before is &d2);
	assert(d4.before is &d3);
	assert(d1.before is &d4);
	
	// d2 d4 d1 d3
	d3.spliceBefore(&d4);
	
	assert(d2.next is &d4);
	assert(d4.next is &d1);
	assert(d1.next is &d3);
	assert(d3.next is null);
	
	assert(d2.before is null);
	assert(d4.before is &d2);
	assert(d1.before is &d4);
	assert(d3.before is &d1);
}

unittest {
	struct Data {
		mixin ListMixture;
		int value;
	}
	
	Data d1, d2, d3, d4;
	
	// d1 d2 d3 d4
	d1.insertNext(&d2);
	d2.insertNext(&d3);
	d3.insertNext(&d4);
	
	// d4 d1 d2 d3
	d4.spliceNext(&d1, &d4);
	
	assert(d4.next is &d1);
	assert(d1.next is &d2);
	assert(d2.next is &d3);
	assert(d3.next is null);
	
	assert(d4.before is null);
	assert(d1.before is &d4);
	assert(d2.before is &d1);
	assert(d3.before is &d2);
	
	// d4 d2 d3 d1
	d4.spliceNext(&d2);
	
	assert(d4.next is &d2);
	assert(d2.next is &d3);
	assert(d3.next is &d1);
	assert(d1.next is null);
	
	assert(d4.before is null);
	assert(d2.before is &d4);
	assert(d3.before is &d2);
	assert(d1.before is &d3);
}

unittest {
	struct Data {
		mixin ListMixture;
		int value;
	}
	
	Data d1, d2, d3, d4;
	
	// d1 d2 d3 d4
	d1.insertNext(&d2);
	d2.insertNext(&d3);
	d3.insertNext(&d4);
	
	assert(d1.front is &d1);
	assert(d2.front is &d1);
	assert(d3.front is &d1);
	assert(d4.front is &d1);
	
	assert(d1.back is &d4);
	assert(d2.back is &d4);
	assert(d3.back is &d4);
	assert(d4.back is &d4);
}

unittest {
	struct Data {
		mixin ListMixture;
		int value;
	}
	
	Data d1, d2, d3, d4;
	
	// d1 d2 d3 d4
	d1.insertNext(&d2);
	d2.insertNext(&d3);
	d3.insertNext(&d4);
	
	foreach(i, e; d1) {
		e.value = i;
	}
	
	assert(d1.value == 0);
	assert(d2.value == 1);
	assert(d3.value == 2);
	assert(d4.value == 3);
	
	int[] values;
	foreach_reverse(e; d4) {
		values ~= e.value;
	}
	
	assert(values[0] == 3);
	assert(values[1] == 2);
	assert(values[2] == 1);
	assert(values[3] == 0);
}

unittest {
	struct Data {
		mixin ListMixture;
		int value;
	}
	
	Data d1, d2, d3, d4;
	
	// d1 d2 d3 d4
	d1.insertNext(&d2);
	d2.insertNext(&d3);
	d3.insertNext(&d4);
	
	foreach(i, e; d1) {
		e.value = i;
	}
	
	assert(d1.find((Data* p) {return p.value == 3;}) is &d4);
	assert(d1.find((Data* p) {return p.value == 2;}) is &d3);
	assert(d1.find((Data* p) {return p.value == 1;}) is &d2);
	assert(d1.find((Data* p) {return p.value == 0;}) is &d1);
	assert(d1.find((Data* p) {return p.value == 4;}) is null);
	
	assert(d4.findReverse((Data* p) {return p.value == 3;}) is &d4);
	assert(d4.findReverse((Data* p) {return p.value == 2;}) is &d3);
	assert(d4.findReverse((Data* p) {return p.value == 1;}) is &d2);
	assert(d4.findReverse((Data* p) {return p.value == 0;}) is &d1);
	assert(d4.findReverse((Data* p) {return p.value == 4;}) is null);
}

unittest {
	struct Data {
		mixin ListMixture;
		int value;
	}
	
	Data d1, d2, d3, d4;
	
	// d1 d2 d3 d4
	d1.insertNext(&d2);
	d2.insertNext(&d3);
	d3.insertNext(&d4);
	
	// d2 d1 d3 d4
	d2.down();
	
	assert(d2.next is &d1);
	assert(d1.next is &d3);
	assert(d3.next is &d4);
	assert(d4.next is null);
	
	assert(d2.before is null);
	assert(d1.before is &d2);
	assert(d3.before is &d1);
	assert(d4.before is &d3);
	
	// d2 d1 d3 d4
	d2.down();
	
	assert(d2.next is &d1);
	assert(d1.next is &d3);
	assert(d3.next is &d4);
	assert(d4.next is null);
	
	assert(d2.before is null);
	assert(d1.before is &d2);
	assert(d3.before is &d1);
	assert(d4.before is &d3);
	
	// d1 d2 d3 d4
	d2.up();
	
	assert(d1.next is &d2);
	assert(d2.next is &d3);
	assert(d3.next is &d4);
	assert(d4.next is null);
	
	assert(d1.before is null);
	assert(d2.before is &d1);
	assert(d3.before is &d2);
	assert(d4.before is &d3);
	
	// d1 d2 d3 d4
	d4.up();
	
	assert(d1.next is &d2);
	assert(d2.next is &d3);
	assert(d3.next is &d4);
	assert(d4.next is null);
	
	assert(d1.before is null);
	assert(d2.before is &d1);
	assert(d3.before is &d2);
	assert(d4.before is &d3);
}
