﻿/**
 *	フォント関連モジュール。
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module outland.poet.font;

import std.string;
import std.stdio;

import derelict.sdl.ttf;
import derelict.sdl.types;

import outland.poet.application;
import outland.poet.error;
import outland.poet.shape;
import outland.poet.surface;

/// フォントクラス。
class Font {
	
	/// 文字スタイルの型。
	typedef int Style;
	
	/// スタイル。
	enum : Style {
		NORMAL,
		BOLD = cast(Style) TTF_STYLE_BOLD,
		ITALIC = cast(Style) TTF_STYLE_ITALIC,
		UNDERLINE = cast(Style) TTF_STYLE_UNDERLINE,
	}
	
	/** フォントファイル名・文字サイズを指定して生成する。
	 *	Params:
	 *		fn		= フォントファイル名。
	 *		size	= サイズ(ポイント)
	 *	Throws:
	 *		SdlException	エラー発生時に投げられる。
	 */
	this(char[] fn, int size) {
		font_ = TTF_OpenFont(fn.ptr, size);
		if(!font_) throw new SdlException();
	}
	
	/// フォントの破棄。
	~this() {
		if(font_ && DerelictSDLttf.loaded) TTF_CloseFont(font_);
	}
	
	/// スタイル。
	void style(Style s) {
		TTF_SetFontStyle(font_, s);
	}
	
	/// ditto
	Style style() {return cast(Style) TTF_GetFontStyle(font_);}
	
	/** 文字列のサイズを求める。
	 *	Params:
	 *		str	= サイズを求める文字列。
	 *	Returns:
	 *		文字列のサイズ。
	 *	Throws:
	 *		SdlException	エラー発生時に投げられる。
	 */
	Size textSize(char[] str) {
		int w, h;
		if(TTF_SizeUTF8(font_, str.ptr, &w, &h) < 0) {
			throw new SdlException();
		}
		return Size(cast(Uint16) w, cast(Uint16) h);
	}
	
	/** 文字列を描画する。
	 *	Params:
	 *		text	= 描画する文字列。
	 *		color	= 描画色。
	 *	Returns:
	 *		描画結果のサーフェイス。
	 */
	Surface renderSolid(char[] text, Color color) {
		auto sur = TTF_RenderUTF8_Solid(font_, text.ptr, color);
		if(!sur) throw new SdlException();
		return new Surface(sur);
	}
	
	/** 文字列を描画する。背景も塗りつぶす。
	 *	Params:
	 *		text	= 描画する文字列。
	 *		color	= 描画色。
	 *		back	= 背景色。
	 *	Returns:
	 *		描画結果のサーフェイス。
	 */
	Surface renderShaded(char[] text, Color color, Color back) {
		auto sur = TTF_RenderUTF8_Shaded(font_, text.ptr, color, back);
		if(!sur) throw new SdlException();
		return new Surface(sur);
	}
	
	/** 文字列を描画する。アルファチャンネル付き。
	 *	Params:
	 *		text	= 描画する文字列。
	 *		color	= 描画色。
	 *	Returns:
	 *		描画結果のサーフェイス。
	 */
	Surface renderBlended(char[] text, Color color) {
		auto sur = TTF_RenderUTF8_Blended(font_, text.ptr, color);
		if(!sur) throw new SdlException();
		return new Surface(sur);
	}
	
	/// フォントの高さ。
	int height() {return TTF_FontHeight(font_);}
	
	/// アセント。
	int ascent() {return TTF_FontAscent(font_);}
	
	/// ディセント。
	int descent() {return TTF_FontDescent(font_);}
	
	/// 行間。
	int lineSkip() {return TTF_FontLineSkip(font_);}
	
private:
	
	/// フォント。
	TTF_Font* font_;
}
