﻿/**
 *	図形関連モジュール。
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module outland.poet.shape;

import std.string;

import derelict.sdl.types;
import derelict.sdl.video;

import outland.tl.algorithm;

/// 座標。
struct Point {

	/** コンストラクタ。
	 *	Params:
	 *		x	= X座標。
	 *		y	= Y座標。
	 *	Returns:
	 *		指定座標。
	 */
	static Point opCall(Sint16 x, Sint16 y) {
		Point p;
		p.x = x;
		p.y = y;
		return p;
	}
	
	/// 加算。
	void opAddAssign(Point rhs) {
		x += rhs.x;
		y += rhs.y;
	}
	
	/// ditto
	Point opAdd(Point rhs) {return Point(x + rhs.x, y + rhs.y);}
	
	/// 減算。
	void opSubAssign(Point rhs) {
		x -= rhs.x;
		y -= rhs.y;
	}
	
	/// ditto
	Point opSub(Point rhs) {return Point(x - rhs.x, y - rhs.y);}
	
	/// 乗算。
	void opMulAssign(Point rhs) {
		x *= rhs.x;
		y *= rhs.y;
	}
	
	/// ditto
	void opMulAssign(Sint16 scale) {
		x *= scale;
		y *= scale;
	}
	
	/// ditto
	Point opMul(Point rhs) {return Point(x * rhs.x, y * rhs.y);}
	
	/// ditto
	Point opMul(Sint16 scale) {return Point(x * scale, y * scale);}
	
	/// 除算。
	void opDivAssign(Point rhs) {
		x /= rhs.x;
		y /= rhs.y;
	}
	
	/// ditto
	void opDivAssign(Sint16 scale) {
		x /= scale;
		y /= scale;
	}
	
	/// ditto
	Point opDiv(Point rhs) {return Point(x / rhs.x, y / rhs.y);}
	
	/// ditto
	Point opDiv(Sint16 scale) {return Point(x / scale, y / scale);}
	
	/// 文字列に変換する。
	char[] toString() {return format("(%d,%d)", x, y);}
	
	Sint16 x;	/// X座標。
	Sint16 y;	/// Y座標。
}

/// サイズ。
struct Size {
	
	/** コンストラクタ。
	 *	Params:
	 *		w	= 幅。
	 *		h	= 高さ。
	 *	Returns:
	 *		指定サイズ。
	 */
	static Size opCall(Uint16 w, Uint16 h) {
		Size s;
		s.w = w;
		s.h = h;
		return s;
	}
	
	/// 文字列に変換する。
	char[] toString() {return format("[%d,%d]", w, h);}

	Uint16 w;	/// 幅。
	Uint16 h;	/// 高さ。
}

/// 矩形の構造体。
struct Rect {
	
	static assert(Rect.sizeof == SDL_Rect.sizeof);
	
	/** コンストラクタ。
	 *	Params:
	 *		x	= X座標。
	 *		y	= Y座標。
	 *		w	= 幅。
	 *		h	= 高さ。
	 *	Returns:
	 *		矩形。
	 */
	static Rect opCall(Sint16 x, Sint16 y, Uint16 w, Uint16 h) {
		Rect r;
		r.pos(x, y);
		r.size(w, h);
		return r;
	}
	
	/** コンストラクタ。
	 *	Params:
	 *		pos	= 座標。
	 *		s	= サイズ。
	 *	Returns:
	 *		矩形。
	 */
	static Rect opCall(Point pos, Size s) {return opCall(pos.x, pos.y, s.w, s.h);}
	
	/// ネイティブな矩形へのポインタ。
	SDL_Rect* ptr() {return &rect;}
	
	/// 位置。
	Point pos() {return Point(x, y);}
	
	/// ditto
	void pos(Point p) {pos(p.x, p.y);}
	
	/// ditto
	void pos(Sint16 x, Sint16 y) {
		rect.x = x;
		rect.y = y;
	}
	
	/// サイズ。
	Size size() {return Size(w, h);}
	
	/// ditto
	void size(Size s) {size(s.w, s.h);}
	
	/// ditto
	void size(Uint16 w, Uint16 h) {
		rect.w = w;
		rect.h = h;
	}
	
	/// X座標。
	Sint16 x() {return rect.x;}
	
	/// ditto
	void x(Sint16 x) {rect.x = x;}
	
	/// Y座標。
	Sint16 y() {return rect.y;}
	
	/// ditto
	void y(Sint16 y) {rect.y = y;}
	
	/// 幅。
	Sint16 w() {return rect.w;}
	
	/// ditto
	void w(Sint16 w) {rect.w = w;}
	
	/// 高さ。
	Sint16 h() {return rect.h;}
	
	/// ditto
	void h(Sint16 h) {rect.h = h;}
	
	/// 右端。
	Sint16 right() {return x + w;}
	
	/// 下端。
	Sint16 bottom() {return y + h;}
	
	/// and演算。
	void opAndAssign(Rect r) {
		Sint16 nx = max(x, r.x);
		Sint16 ny = max(y, r.y);
		Sint16 nw = min(right, r.right) - nx;
		Sint16 nh = min(bottom, r.bottom) - ny;
		if(nw < 0 || nh < 0) {
			*this = Rect.init;
		} else {
			*this = Rect(Point(nx, ny), Size(nw, nh));
		}
	}
	
	/// ditto
	Rect opAnd(Rect r) {
		Rect tmp = *this;
		tmp &= r;
		return tmp;
	}
	
	/// 空かどうか返す。
	bool isEmpty() {return w == 0 || h == 0;}
	
	/// 文字列に変換する。
	char[] toString() {return format("%s%s", pos(), size());}
	
	/// ネイティブな矩形。
	SDL_Rect rect;
}
