﻿/**
 *	スプライト関連モジュール。
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module outland.poet.sprite;

import outland.tl.list;

import outland.poet.shape;
import outland.poet.surface;
import outland.poet.quadtree;

/// スプライト。
interface Sprite {
	
	/// 座標。
	Point position();
	
	/// ditto
	void position(Point pos);
	
	/// サーフェイスに描画する。
	void draw(Point org, Surface suf);
	
	/// 更新領域を得る。複数の更新領域が指定される場合もある。
	void getUpdateArea(void delegate(Rect*) dg);
}

/// サーフェイススプライト。
class SurfaceSprite : Sprite {
	
	/// デフォルトコンストラクタ。
	this() {}
	
	/// サーフェイスを指定して生成する。
	this(Surface suf)
	in {
		assert(suf !is null);
	} body {
		setup(suf, Rect(Point(0, 0), suf.size));
	}
	
	/// サーフェイスを指定して生成する。
	this(Surface suf, Rect r) {setup(suf, r);}
	
	Point position() {return position_;}
	void position(Point pos) {position_ = pos;}
	
	void draw(Point org, Surface suf)
	in {
		assert(suf !is null);
	} body {
		if(surface_ !is null) {
			suf.blit(org + this.position, surface_, rect_);
		}
		updated_ = false;
	}
	
	void getUpdateArea(void delegate(Rect*) dg) {
		if(updated_) {
			dg(&Rect(position_, rect_.size));
		}
	}
	
	/// サーフェイスと領域の設定。
	void setup(Surface suf, Rect r) {
		surface_ = suf;
		rect_ = r;
		updated_ = true;
	}
	
private:
	
	/// サーフェイス。
	Surface surface_;
	
	/// 領域。
	Rect rect_;
	
	/// 位置。
	Point position_;
	
	/// 更新されたかどうか。
	bool updated_;
}

/// スプライトグループ。
class SpriteGroup : Sprite {
	
	/// グループ項目。
	class Item : Sprite {
		
		/// コンストラクタ。
		this(Sprite sp) {sprite_ = sp;}
		
		Point position() {return sprite_.position;}
		void position(Point pos) {sprite_.position = pos;}
		
		void draw(Point org, Surface suf) {sprite_.draw(org, suf);}
		
		void getUpdateArea(void delegate(Rect*) dg) {sprite_.getUpdateArea(dg);}
		
		/// スプライトを上に移動。
		void up() {list_.up();}
		
		/// スプライトを下に移動。
		void down() {list_.down(begin_);}
		
		/// スプライトを最上位に移動。
		void top() {
			auto p = list_.back;
			if(p !is this) {
				list_.erase();
				p.insertNext(this);
			}
		}
		
		/// スプライトを最下位に移動。
		void bottom() {
			auto p = list_.front(begin_);
			if(p !is this) {
				list_.erase();
				p.insertNext(this);
			}
		}
		
		/// グループからの削除。オブジェクトも破棄されるので注意。
		void remove() {
			list_.erase();
			destroyItem(this);
		}
		
	private:
		
		/// 項目の追加。
		void add(Item i) {
			list_.back.insertNext(i);
		}
		
		/// 内部のスプライト。
		Sprite sprite_;
		
		/// リスト操作の取り込み。
		mixin ListMixture list_;
	}
	
	/// デフォルトコンストラクタ。
	this() {begin_ = createItem(null);}
	
	/// デストラクタ。
	~this() {destroyItem(begin_);}
	
	Point position() {return pos_;}
	void position(Point pos) {pos_ = pos;}
	
	void draw(Point org, Surface suf) {
		org += this.position;
		foreach(item; this) {
			item.draw(org, suf);
		}
	}
	
	void getUpdateArea(void delegate(Rect*) dg) {
		Rect tmp;
		foreach(item; this) {
			item.getUpdateArea((Rect* r) {
				tmp = *r;
				tmp.pos = tmp.pos + this.position;
				dg(&tmp);
			});
		}
	}
	
	/// スプライトの追加。
	Item add(Sprite sp)
	in {
		assert(sp !is null);
	} body {
		auto item = createItem(sp);
		begin_.add(item);
		return item;
	}
	
	/// 巡回する。
	int opApply(int delegate(inout size_t, inout Item) dg) {
		if(begin_.next !is null) {
			foreach(i, item; begin_.list_.next) {
				if(auto result = dg(i, item)) return result;
			}
		}
		return 0;
	}
	
	/// ditto
	int opApply(int delegate(inout Item) dg) {
		if(begin_.next !is null) {
			foreach(item; begin_.list_.next) {
				if(auto result = dg(item)) return result;
			}
		}
		return 0;
	}
	
protected:
	
	/// 項目の生成。
	Item createItem(Sprite sp) {return new Item(sp);}
	
	/// 項目の削除。
	void destroyItem(Item item) {delete item;}
	
private:
	
	/// 開始項目。
	Item begin_;
	
	/// 位置。
	Point pos_;
}
