﻿/**
 *	実行コンテキスト関連モジュール。
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module outland.script.context;

import std.string;

import outlandkarasu.script.bytecode;
import outlandkarasu.script.value;
import outlandkarasu.tl.stack;

/// 実行コンテキストクラス。
class Context {
	
	/** プログラムを指定して生成する。
	 *	Params:
	 *		src	= プログラム。
	 */
	this(ubyte[] src) {
		stack_ = new Stack!(Value);
		callStack_ = new Stack!(uint);
		program_ = src;
	}
	
	/// まだプログラムが残っているか返す。
	bool hasMore() {return ip_ < program_.length;}
	
	/**	次のバイトコードを取り出す。
	 *	Returns:
	 *		次のバイトコード。
	 *	Throws:
	 *		ScriptException	エラー発生時に投げられる。
	 */
	ByteCode nextCode()
	in {
		assert(ip_ < program_.length);
	} out(result) {
		assert(ip_ < program_.length);
		assert(ByteCode.min <= result && result <= ByteCode.max);
	} body {
		return program[ip_++];
	}
	
	/**	次の1バイト値を取り出す。
	 *	Returns:
	 *		次の1バイト。
	 *	Throws:
	 *		ScriptException	エラー発生時に投げられる。
	 */
	ubyte nextByte()
	in {
		assert(ip_ < program_.length);
	} out(result) {
		assert(ip_ < program_.length);
	} body {
		return program[ip_++];
	}
	
	/**	次の4バイト値を取り出す。
	 *	Returns:
	 *		次の4バイト。
	 *	Throws:
	 *		ScriptException	エラー発生時に投げられる。
	 */
	int nextSint()
	in {
		assert(ip_ < program_.length);
	} out(result) {
		assert(ip_ < program_.length);
	} body {
		int result;
		memcpy(&result, program.ptr + ip_, result.sizeof);
		ip_ += result.sizeof;
		return result;
	}
	
	/// ditto
	uint nextUint()
	in {
		assert(ip_ < program_.length);
	} out(result) {
		assert(ip_ < program_.length);
	} body {
		uint result;
		memcpy(&result, program.ptr + ip_, result.sizeof);
		ip_ += result.sizeof;
		return result;
	}
	
	/**	ジャンプする。
	 *	Params:
	 *		addr	= ジャンプ先アドレス。
	 *	Throws:
	 *		ScriptException	エラー発生時に投げられる。
	 */
	void jump(uint addr)
	in {
		assert(addr < program_.length);
	} body {
		ip_ = addr;
	}
	
	/**	関数コールする。
	 *	Params:
	 *		addr	= 関数のアドレス。
	 *	Throws:
	 *		ScriptException	エラー発生時に投げられる。
	 */
	void call(uint addr)
	in {
		assert(addr < program_.length);
	} body {
		callStack_ ~= ip_;
	}
	
	/** 関数から戻る。
	 *	Throws:
	 *		ScriptException	エラー発生時に投げられる。
	 */
	void returnCall()
	in {
		assert(callStack_.length > 0);
		assert(callStack_.back < program_.length);
	} body {
		ip_ = callStack_.back;
		callStack_.popBack();
	}
	
private:
	
	/// スタック。
	Stack!(Value) stack_;
	
	/// 関数コール用のスタック。
	Stack!(uint) callStack_;
	
	/// 実行中のプログラム。
	ubyte[] program_;
	
	/// プログラムカウンタ。
	uint ip_;
}
