﻿/**
 *	メモリプールのモジュール。
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module outland.tl.pool;

/// メモリプール。
class MemoryPool(size_t LENGTH) {
	
	/// コンストラクタ。初期サイズを指定する。
	this(size_t len = 16) {reserve(len);}
	
	/// オブジェクト確保。
	void* allocate() {
		Entry* e;
		if(begin_ is null) {
			e = new Entry;
		} else {
			e = begin_;
			begin_ = begin_.next;
		}
		return e;
	}
	
	/// オブジェクト解放。
	void deallocate(void* p) {
		auto e = cast(Entry*) p;
		e.data[] = 0;
		e.next = begin_;
		begin_ = e;
	}
	
	/// 予約する。
	void reserve(size_t len) {
		for(size_t i = 0; i < len; ++i) {
			deallocate(new Entry);
		}
	}
	
private:
	
	/// エントリ。
	union Entry {
		/// データ部分。
		ubyte[LENGTH] data;
		
		/// プール時のリストのリンク。
		Entry* next;
	}
	
	/// プールされているオブジェクトの始点。
	Entry* begin_;
}

/// 構造体プール。
class StructPool(T) {
	
	/// コンストラクタ。初期サイズを指定する。
	this(size_t len = 16) {pool_ = new Pool(len);}
	
	/// デストラクタ。
	~this() {delete pool_;}
	
	/// オブジェクト確保。
	T* allocate() {
		auto p = cast(T*) pool_.allocate();
		*p = T.init;
		return p;
	}
	
	/// オブジェクト解放。
	void deallocate(T* p) {pool_.deallocate(p);}
	
	/// 予約する。
	void reserve(size_t len) {pool_.reserve(len);}
	
private:
	
	/// プールの型。
	alias MemoryPool!(T.sizeof) Pool;
	
	/// プール。
	Pool pool_;
}
