﻿/**
 *	list module.
 *
 *	Version:
 *		$Revision$
 *	Date:
 *		$Date$
 *	License:
 *		MIT/X Consortium License
 *	History:
 *		$Log$
 */

module drt.list;

public import drt.algorithm : swap;

/// A double head linkde list class
class List(T) {
	
	struct Iterator {
		
		static Iterator opCall(List list)
		in {
			assert(list);
		} body {
			return opCall(list.begin_);
		}
		
		static Iterator opCall(Entry* entry) {
			Iterator iter;
			iter.entry_ = entry;
			return iter;
		}
		
		Iterator clone() {return *this;}
	
		T next() {
			Entry* next = entry_.next;
			T val = entry_.value;
			entry_.next = next;
			return val;
		}
		
		bool hasMore() {return entry_ !is null;}
		
	private:
		
		Entry* entry_;
	}
	
	this() {}
	~this() {clear();}
	
	/// clear all elements
	void clear() {
		if(begin_) {
			begin_.prev.next = null;
			
			for(Entry* e = begin_; e;) {
				Entry* next = e.next;
				deallocate(e);
				e = next;
			}
			begin_ = null;
			length_ = 0;
		}
	}
	
	size_t length() {return length_;}
	
	bool pushFront(T value) {
		if(!pushBack(value)) {
			return false;
		}
		begin_ = begin_.prev;
	}
	
	void popFront() {
		if(begin_) {
			Entry* front = begin_;
			
			if(front.next == begin_) {
				begin_ = null;
			} else {
				front.next.prev = front.prev;
				front.prev.next = front.next;
				begin_ = front.next;
			}
			deallocate(front);
			--length_;
		}
	}
	
	bool pushBack(T value) {
		Entry* entry = allocate();
		if(entry is null) {
			return false;
		}
		
		entry.value = value;
		if(begin_) {
			Entry* before = begin_.prev;
			begin_.prev = entry;
			before.next = entry;
			entry.next = begin_;
			entry.prev = before;
		} else {
			entry.next = entry;
			entry.prev = entry;
			begin_ = entry;
		}
		++length_;
	}
	
	void popBack() {
		if(begin_) {
			Entry* back = begin_.prev;
			
			if(back == begin_) {
				begin_ = null;
			} else {
				back.next.prev = back.prev;
				back.prev.next = back.next;
			}
			deallocate(back);
			--length_;
		}
	}
	
	T front()
	in {
		assert(begin_ !is null);
	} body {
		return begin_.value;
	}
	
	T back()
	in {
		assert(begin_ !is null && begin_.prev !is null);
	} body {
		return begin_.prev.value;
	}
	
	int opApply(int delegate(inout T) dg) {
		if(Entry* e = begin_) {
			for(;; e = e.next) {
				if(int result = dg(e.value)) return result;
				if(e.next is begin_) break;
			}
		}
		return 0;
	}
	
	int opApply(int delegate(inout size_t, inout T) dg) {
		if(Entry* e = begin_) {
			for(size_t i = 0; ; ++i, e = e.next) {
				if(int result = dg(i, e.value)) return result;
				if(e.next is begin_) break;
			}
		}
		return 0;
	}

	int opApplyReverse(int delegate(inout T) dg) {
		if(begin_) {
			for(Entry* e = begin_.prev; ; e = e.prev) {
				if(int result = dg(e.value)) return result;
				if(e is begin_) break;
			}
		}
		return 0;
	}

protected:

	struct Entry {
		Entry* next;
		Entry* prev;
		T value;
	}
	
	Entry* allocate() {return new Entry;}
	
	void deallocate(Entry* e) {delete e;}
	
private:
	
	/// 始点。
	Entry* begin_;
	
	/// 要素数。
	size_t length_;
}

/// list operation mixture
template ListMixture() {
	
	alias typeof(this) Pointer;
	
	Pointer find(bool delegate(Pointer) pred) {
		Pointer p = this;
		for(; p !is null && !pred(p); p = p.next) {}
		return p;
	}
	
	void erase() {
		if(before) {
			before.next = next;
		}
		if(next) {
			next.before = before;
		}
		
		before = null;
		next = null;
	}
	
	/// src insert this list.
	void insertBefore(Pointer src)
	in {
		assert(src !is null);
	} body {
		src.erase();
		
		src.before = before;
		if(before !is null) {
			before.next = src;
		}
		before = src;
		src.next = this;
	}
	
	/// ditto
	void insertNext(Pointer src)
	in {
		assert(src !is null);
	} body {
		src.erase();
		
		src.next = next;
		if(next !is null) {
			next.before = src;
		}
		next = src;
		src.before = this;
	}
	
	void reverse(Pointer end = null) {
		if(end is this || next is end) {
			// only list
			return;
		}
		
		auto front = this;
		auto back = this;
		
		for(Pointer p = this; p !is end; back = p, p = p.before) {
			swap(p.before_, p.next_);
		}
		
		if(back.before !is null) {
			back.before.before = front;
		}
		if(front.next !is null) {
			front.next.next = before;
		}
		swap(front.next_, back.before_);
	}
	
	/// splice [begin, end)
	void spliceBefore(Pointer begin, Pointer end = null)
	in {
		assert(begin !is null);
	} body {
		if(begin.before !is null) {
			begin.before.next = end;
		}
		
		if(end !is null) {
			end.before = begin.before;
		}
		
		auto back = begin.find((Pointer p) {return p.next is end;});
		assert(back !is null);
		
		if(before !is null) {
			before.next = begin;
		}
		begin.before = before;
		
		before = back;
		back.next = this;
	}
	
	/// splice [begin, end)
	void spliceNext(Pointer begin, Pointer end = null)
	in {
		assert(begin !is null);
	} body {
		if(begin.before !is null) {
			begin.before.next = end;
		}
		
		if(end !is null) {
			end.before = begin.before;
		}
		
		auto back = begin.find((Pointer p) {return p.next is end;});
		assert(back !is null);
		
		if(next !is null) {
			next.before = back;
		}
		back.next = next;
		
		next = begin;
		begin.before = this;
	}
	
	Pointer front() {
		Pointer p = this;
		for(; p.before !is null; p = p.before) {}
		return p;
	}
	
	Pointer back() {
		Pointer p = this;
		for(; p.next !is null; p = p.next) {}
		return p;
	}
	
	int opApply(int delegate(inout Pointer) dg) {
		for(Pointer p = this; p !is null; p = p.next) {
			if(int result = dg(p)) return result;
		}
		return 0;
	}
	
	int opApply(int delegate(inout size_t, inout Pointer) dg) {
		size_t i = 0;
		for(Pointer p = this; p !is null; p = p.next, ++i) {
			if(int result = dg(i, p)) return result;
		}
		return 0;
	}

	int opApplyReverse(int delegate(inout Pointer) dg) {
		for(Pointer p = this; p !is null; p = p.before) {
			if(int result = dg(p)) return result;
		}
		return 0;
	}
	
	Pointer next() {return next_;}
	Pointer before() {return before_;}
	
	size_t length() {
		size_t n = 0;
		for(Pointer p = this; p !is null; p = p.next, ++n) {}
		return n;
	}
	
private:

	void next(Pointer p) {next_ = p;}
	void before(Pointer p) {before_ = p;}

	Pointer next_;
	Pointer before_;
}
